// =========================================
// mShelf macro 1.1
// -----------------------------------------
// Copyright Gilles Tran 2000
// -----------------------------------------
// Feel free to use and modify
// -----------------------------------------
// This macro creates a book shelf where the books lean towards each other : \\\\
// The shelf extends on the x axis, the backs of the books point towards the -z axis
// The shelf is divided in "sets" ; each set leans towards the left or the right, alternatively
// The "reverse" set (///) is actually an inverted normal set (\\\) translated of the set length
// 
//  \\\\\\///////\\\\\\\///////
//  Set 1  Set 2  Set 3  Set 4
//
// xShelf : length of the shelf (the true length will be slightly shorter)
// yShelf : height of the shelf
// zShelf : depth of the shelf
// nBook : number of books
// nSet   : number of succeding book "sets" (each set leans to the opposite of the previous one)
// typeShelf : 0 = each set is made of identical books, 1 = each book is different
// slackShelf : controls the "slackness" of the books (0 = all books are strickly vertical)
// xBook, sdxBook : average and pseudo standard deviation of book thickness
// sdyBook : pseudo standard deviation of book height
// sdzBook : pseudo standard deviation of book depth
// 
// -----------------------------------------
// Notes
// -----------------------------------------
// Randomness : if you don't declare a seed before launching the macro, it will default to rd=seed(0)
// To obtain different shelves, put #declare rd=seed(0) before the macro (see example)
// -----------------------------------------
// The shelf is contained in the box :
// box{<0,0,-(zBook+sdzBook*0.6)>,<xShelf,(yBook+sdyBook*0.5)*cos(atan2(xBook+sdxBook*0.5,yBook+sdyBook*0.5))+(xBook+sdxBook*0.5)*sin(atan2(xBook+sdxBook*0.5,yBook+sdyBook*0.5)),0>}
// -----------------------------------------
// The formulas are geometrically right (I hope), but this macro doesn't take
// into account the weight of the books, so it can behave strangely sometimes
// -----------------------------------------
// Declare the font path and names before using this file.
// For example :
// #declare fontpath="\windows\fonts\\"
// #declare fontname_1="arial.ttf"
// #declare fontname_2="times.ttf"
// ----------------------------------------- 
#include "colors.inc"

#declare titleDir=0;
#declare titleVariety=24;
#declare rd=seed(100);                                            
#macro mShelf(xShelf,yShelf,zShelf,nSet,typeShelf,slackShelf,xBook,sdxBook,sdyBook,sdzBook)
  #ifndef (rd) #declare rd=seed(0); #end // a seed must be declared
  #local i=0;         
  #local ixSetTrue=0; // current true set length
  #local xShelfTrue=0; // current trut shelf length
  #if (rand(rd)<0.5)
    #local sideLean=0; // lean to the left at the beginning
  #else 
    #local sideLean=1; // lean to the right at the beginning
  #end
  union{
  // shelf loop
  #while (i<nSet) // i = set number
    // book set loop : all book lean to the left but the first
    #if (typeShelf=0) // if the type of shelf is set to "series" : all books are equal for this set
      // -------------------------------------------
      // Creation of book (series)
      // -------------------------------------------
      #local xB=xBook+sdxBook*(rand(rd)-0.5);
      #if (xB<=0) #local xB=xBook; #end
      #local yB=yShelf-sdyBook*rand(rd);
      #local zB=zShelf-sdzBook*rand(rd);
      #local Book=array[titleVariety]
      #local f=0;
      #while (f<titleVariety)   
        #local Book[f]=mBook(xB,yB,zB,titleDir)
        #local f=f+1;
      #end
    #end
    union{ // set loop
      #local j=0; // j = book number in the set
      #local ixSet_tmp=0; 
      #local xShelfTrue=xShelfTrue+ixSetTrue; // true current length of the shelf
      #if (i=nSet-1)
        #local ixSet=xShelf-xShelfTrue; // last set is adjusted to the self length
      #else                             
        #local ixSet=(xShelf/nSet)*(1.5-rand(rd));  // set length is randomized
      #end
      #while (ixSet_tmp<ixSet) // progression along the set length
      //#warning concat(str(i,0,0)," ",str(j,0,0)," ",str(ixSet_tmp,0,3)," ",str(ixSet,0,3),"\n")
        #local lMax=rand(rd)*slackShelf;
        #if (typeShelf != 0) // all books are different
        // -------------------------------------------
        // Creation of book (full random)
        // -------------------------------------------
          #local xB=xBook+sdxBook*(rand(rd)-0.5);
          #if (xB<=0) #local xB=xBook; #end
          #local yB=yShelf-sdyBook*rand(rd);
          #local zB=zShelf-sdzBook*rand(rd);
          #local Book=array[titleVariety]
          #local f=0;
          #while (f<titleVariety)
            #local Book[f]=mBook(xB,yB,zB,titleDir)
            #local f=f+1;
          #end
        #end                                                     
        // -------------------------------------------
        // Calculation of book placing parameters posB and aB
        // -------------------------------------------
        #if (j=0) // the first book of a set is always straight
          #local posB=0;
          #local aB=0;
        #else     // following books                   
          #local lB=cos(radians(aB_old))+lMax*rand(rd); // distance from the previous book
          #local posB=posB_old+xB_old*lB;
        // -------------------------------------------
        // Calculation of book angle aB
        // The solution was supposed to be straightforward but
        // it took me 2 days to figure it out !
        // -------------------------------------------
          #if (lB>1/cos(radians(aB_old))) // present book leans on the previous one
            #local dA=posB-posB_old-xB_old/cos(radians(aB_old));
            #local dB=yB;
            #local dR=yB_old+xB_old*tan(radians(aB_old));
            #local dQ=sin(radians(aB_old))*dR;                            
            #if (dA*dA*(dQ*dQ-dR*dR)+dB*dB*dR*dR>=0)
              #local dX=(-(dQ*dA)+sqrt(dA*dA*(dQ*dQ-dR*dR)+dB*dB*dR*dR))/dR;
              #if (dX<dR) // present book leans on the previous book's cover
                #local aB=90-degrees(acos(((dX*dQ/dR)+dA)/dB));
              #else       // present book leans on the previous book's top corner
                #local aB=90-degrees(atan2(dR*cos(radians(aB_old)),(dQ+dA)));
              #end
            #else
              #local aB=0; // strange case, happens sometimes
            #end
          #else // present book does not lean on the previous one but is held by its bottom corner  
            #if (slackShelf=0)   // no slack, all books are vertical
              #local aB=0;
            #else
              #local aB=degrees(atan2(lB-cos(radians(aB_old)),sin(radians(aB_old))));
            #end
          #end
          // -------------------------------------------
          // End of calculation of book angle aB
          // -------------------------------------------
        #end
        #if (posB+xB*cos(radians(aB))<=ixSet) // there's still room on the set
          // -------------------------------------------
          // This is where the books are placed
          // -------------------------------------------   
          object{Book[rand(rd)*titleVariety]
            #if (sideLean=1) // invert on x axis ; necessary to invert the box textures if any
              scale <-1,1,1>
              translate x*xB
            #end
            rotate z*aB     // box is rotated to the left (aB positive)
            translate x*posB // box is translated to the right on the x axis
            translate -z*zB  // box is aligned on z=0 (far wall of the shelf)
          } 
          sphere{0,0.01 pigment{Black} translate <-100+xB,-100,-100>}
          #local j=j+1;
          #local xB_old=xB;
          #local yB_old=yB;
          #local zB_old=zB;
          #local aB_old=aB;
          #local posB_old=posB;
          #local ixSetTrue=posB+xB*cos(radians(aB)); // current "true" length of the set
        #end
        #local ixSet_tmp=posB+xB*cos(radians(aB));
      #end
      //  box{0,<3,4,2> pigment{Red} translate x*ixSetTrue} // This is a bookend !
      #if (sideLean=0) // lean on the left
        translate x*xShelfTrue
        #local sideLean=1;
      #else           // lean on the right, by inverting the object
        scale <-1,1,1>
        translate x*(xShelfTrue+ixSetTrue)
        #local sideLean=0;
      #end
      //  #warning concat("end ",str(ixSetTrue,0,3),"\n")
    }                         
    #local i=i+1;
    #end
  }
#end
// -----------------------------------------
// End of mShelf macro
// =========================================

/*
#macro mBook(xB,yB,zB)
        box{0,<xB,yB,zB> pigment{rgb <rand(rd),rand(rd),rand(rd)>}}
#end
*/

// =========================================
// Initialises the title list
// -----------------------------------------

#declare nTitles=0;
#fopen fTitles "titles.txt" read // list of book titles
#while (defined(fTitles)) // read once to get the number of titles
        #read(fTitles,dummy)
        #declare nTitles=nTitles+1;
#end
#fclose fTitles

#declare bookTitles=array[nTitles]   // List of titles
#declare bookTitlesLength=array[nTitles] // List of title length
#declare i=0;
#fopen fTitles "titles.txt" read // second to put the titles and their length into an array
#while (defined(fTitles))
        #read(fTitles,dummy)
        #declare bookTitles[i]=dummy 
        #declare bookTitlesLength[i]=strlen(dummy);
        #declare i=i+1;
#end
#fclose fTitles

// =========================================
// Initialises the font list
// -----------------------------------------

#declare nFonts=0;
#fopen fFonts "fonts.txt" read
#while (defined(fFonts))
        #read(fFonts,dummy)
        #declare nFonts=nFonts+1;
#end
#fclose fFonts
#declare bookFonts=array[nFonts]
#declare i=0;
#fopen fFonts "fonts.txt" read
#while (defined(fFonts))
        #read(fFonts,dummy)
        #declare bookFonts[i]=dummy
        #declare i=i+1;
#end
#fclose fFonts

// =========================================
// mBook macro 1.2
// -----------------------------------------
#macro mBook(xBook,yBook,zBook,titleDir)
#ifndef (rd) #declare rd=seed(0); #end // a seed must be declared
#local colBook=array[6]{Red,Red,Red,Red,Red,Red}
#local eCover=xBook/20; // thickness of the cover
union{    
  #local flatratio=0.3+(0.3*rand(rd));
//----------
// textures
//----------
  #local colBook[0]=rgb <1+0.3*rand(rd),0.3+0.3*rand(rd),0.2*rand(rd)>*0.2;                        // dark red #1
  #local colBook[1]=rgb <0.2+0.5*rand(rd),0.8+0.3*rand(rd),0.4*rand(rd)>*0.2;                      // dark green
  #local colBook[2]=rgb <1.3+0.2*rand(rd),1+0.2*rand(rd),0.2+0.5*rand(rd)>*0.5;                    // light brown
  #local colBook[3]=rgb<1+rand(rd)*0.2,1+rand(rd)*0.2,rand(rd)*0.2>*0.8;                           // gold color
  #local colPage=rgb<0.9+0.1*rand(rd),0.8+0.1*rand(rd),0.8>;                                       // pages are yellowish
  #local finCover=finish{metallic brilliance 3 ambient 0.01 diffuse 0.4 specular 0.3 roughness 0.1}
  #local finPage=finish{ambient 0.1 diffuse 0.5 specular 0.1 roughness 0.1}
  #local colDom=colBook[max(0,int(rand(rd)*2-0.001))];  
  #local colTrim_1=colBook[max(0,int(rand(rd)*3-0.001))];
  #local colTrim_2=colBook[3];                     
  #local pigBinder=mpatBinder(colDom,colTrim_1,colTrim_2)
  #local txtCover=texture{pigment{color colDom} finish{finCover}}
  #local txtBinding=texture{ pigment{pigBinder} finish{finCover} scale <xBook,yBook,yBook>}
  #local txtTitle=texture{ pigment{ color colBook[3]} finish{finCover}}
  #local txtPage=texture{
    pigment{
      gradient x
      turbulence 0.1
      color_map{
        [0 color colPage]
        [1 color colPage*0.7]
      }
    }
    finish{finPage}
    scale xBook*rand(rd)*0.06
    translate x*xBook
  }
  // -------------------------------------------
  // Book parameters
  // -------------------------------------------
  #local rCover=eCover*0.5;            // radius of cover thickness
  #local rBinding=xBook*0.5;           // radius of binding
  #local zBinding=rBinding*flatratio;  // thickness of binding
  #local zCover=zBook-rCover-zBinding;
  #local yCover=yBook-eCover;
  #local zPage=zCover-4*eCover;
  #local yPage=yBook-4*eCover;
  #local xPage=xBook-2*eCover;
  // -------------------------------------------
  // front cover
  // -------------------------------------------
  union{
    box{0,<eCover,yCover,zCover>}          
    union{
      cylinder{0,z*zCover,rCover translate <0,yCover,0>}
      cylinder{0,z*zCover,rCover translate <0,0,0>} 
      translate x*rCover
    }                                     
    translate <rBinding-eCover,rCover,zBinding>
  }
  // -------------------------------------------
  // back cover
  // -------------------------------------------
  union{
    box{0,<eCover,yCover,zCover>}          
    union{
      cylinder{0,z*zCover,rCover translate <0,yCover,0>}
      cylinder{0,z*zCover,rCover translate <0,0,0>}
      translate x*rCover
    }                                     
    translate <rBinding-eCover,rCover,zBinding>
    scale <-1,1,1>
  }          
  // -------------------------------------------
  // binding
  // -------------------------------------------
  union{
    // -------------------------------------------
    // Book title on the binding
    // This part needs to be rewritten for use with the text functions in Megapov
    // -------------------------------------------   
    #local iBook=int(rand(rd)*nTitles);
    #local rtext=bookTitles[iBook]
    #local nLetters=bookTitlesLength[iBook]/1.8;
    #local titleBook=
      text{ttf bookFonts[rand(rd)*nFonts] rtext 1,0
      translate <0,-0.3,0> //-nletters/2
      scale <0.4,xBook*0.5,rCover>
    }
      #switch(titleDir)
        #case(0)     // title upright if book on back
          object{titleBook rotate z*-90 translate <0,yBook-3,zBinding-rBinding*flatratio> texture{txtTitle}}
        #break
        #case(1)     // title upright if book on face
          object{titleBook rotate z*90 translate <0,2,zBinding-rBinding*flatratio> texture{txtTitle}}
        #break
        #else       // title vertical on spine
          object{titleBook rotate z*-90 translate <0,yBook/2,zBinding-rBinding*flatratio> texture{txtTitle}}
        #break
      #end
       difference{               
         union{
           cylinder{0,y*yCover,rBinding}
           torus{rBinding-rCover,rCover sturm}
           torus{rBinding-rCover,rCover  sturm translate y*yCover}
         }
         cylinder{-rCover*2*y,y*(yCover+rCover)*1.1,rBinding-eCover}
         plane{z,0 inverse}                
         scale <1,1,flatratio>
         translate <0,rCover,zBinding>
         texture{txtBinding translate -x*xBook*0.5}
      } 
    }
    // -------------------------------------------
    // pages        
    // -------------------------------------------
    union{
      difference{
        box{0,<xPage,yPage,zPage>}
        cylinder{-y*0.1,y*yPage*1.1,xPage*0.5 scale <1,1,flatratio*rand(rd)> translate <xPage*0.5,0,zPage>}
      }
      difference{
        cylinder{0,y*yPage,xPage*0.5}
        cylinder{-y*0.1,y*yPage*1.1,xPage*0.5 scale <1,1,0.5> translate -z*0.5*xPage}
        scale <1,1,flatratio> translate x*xPage*0.5
      }                                                            
      translate <-xPage*0.5,2*eCover,zBinding>
      texture{txtPage}
    } 
    translate x*xBook*0.5
    texture{txtCover}
}
#end  

// -----------------------------------------
// end of mBook macro
// =========================================

// =========================================
// mRandomText macro 1.0
// -----------------------------------------
// Copyright Gilles Tran 1999
// -----------------------------------------
// This macro creates a random text of nLetters letters
// -----------------------------------------
// local the font path and names before using this file.
// For example :
// #local fontpath="\windows\fonts\\"
// #local fontname_1="arial.ttf"
// #local fontname_2="times.ttf"
// -----------------------------------------
#macro mRandomText(nLetters)
  #local rtext=""
  #if (rand(rd)<0.5)
    #local fontname=concat(fontpath,fontname_1)
  #else
    #local fontname=concat(fontpath,fontname_2)
  #end                
//  #warning fontname
  #local i=0;
  #if (rand(rd)<0.5)
    #local asciistart=65; // majuscule
  #else
    #local asciistart=97; // minuscule
  #end
  #while (i<nLetters)
    #local rtext=concat(rtext,chr(asciistart+int(27*rand(rd))))
    #local i=i+1;                  
  #end
  text{ttf fontname rtext 1,0}
#end
// -----------------------------------------
// end of mRandomText macro
// =========================================
// =========================================
// mpatBinder macro 1.0
// -----------------------------------------
// This macro defines several patterns for an old book binder
// it returns the pattern typePattern
// colDom is the dominant color
// colTrim_1 is a secondary color
// colTrim_2 is a third color, usually gold             
// 
// the macro returns a pigment color starting on <0,0,0> going to <1,1,1>
#macro mpatBinder(colDom,colTrim_1,colTrim_2)
#local yvar=0.125; //rand(rd)*0.25; // this variable is used to "slide" the pattern vertically so that the stripes change place
pigment{
//  #switch (typePattern)
/*
//  #case (0) // dom/gold/black/gold/trim/gold/dom
  gradient y
  #declare p1=pigment{
    gradient y
    color_map{
      [0 color colDom*0.8]
      [0.1 color colDom]
      [0.5+yvar color colDom]
      [0.5+yvar color colTrim_2]
      [0.52+yvar color colTrim_2]
      [0.52+yvar color colDom*0.01]
      [0.6+yvar color colDom*0.01]
      [0.6+yvar color colTrim_2]
      [0.62+yvar color colTrim_2]
      [0.62+yvar color colTrim_1]
      [0.75+yvar color colTrim_1]
      [0.75+yvar color colTrim_2]
      [0.77+yvar color colTrim_2]
      [0.77+yvar color colDom]
      [1 color colDom*0.8]
    }
  }
  #declare p2=pigment{
    spherical
    translate y*0.5
    color_map{
      [0 colDom*0.01]
      [0.7+yvar colDom*0.01]
      [0.7+yvar colTrim_2]
      [1 colTrim_2]
    }
    scale 0.08
    translate x*0.5 
    translate y*(0.52+yvar)
  }                          
  pigment_map{
    [0 p1]
    [0.52+yvar p1]
    [0.52+yvar p2]
    [0.6+yvar p2]
    [0.6+yvar p1]
    [1 p1]
  }
//  #break
*/
/*
//  #case (2) // dom/gold/dom/gold/dom/gold/black/gold/dom/gold/dom
  gradient y
  #declare p1=pigment{
    gradient y
    color_map{
      [0 color colDom*0.8]
      [0.05+yvar color colDom]
      [0.05+yvar color colTrim_2]
      [0.06+yvar color colTrim_2]
      [0.06+yvar color colDom]
      [0.22+yvar color colDom]
      [0.22+yvar color colTrim_2]
      [0.23+yvar color colTrim_2]
      [0.23+yvar color colDom]
      [0.50+yvar color colDom]
      [0.50+yvar color colTrim_2]
      [0.51+yvar color colTrim_2]
      [0.51+yvar color colDom*0.01]
      [0.72+yvar color colDom*0.01]
      [0.72+yvar color colTrim_2]
      [0.73+yvar color colTrim_2]
      [0.73+yvar color colDom]
      [0.84+yvar color colDom]
      [0.84+yvar color colTrim_2]
      [0.85+yvar color colTrim_2]
      [0.85+yvar color colDom]
      [1 color colDom*0.8]
    }
  }
  #declare p2=pigment{
    spherical
    translate x
    warp{repeat x*2}
    color_map{
      [0 colDom*0.01]
      [0.7+yvar colDom*0.01]
      [0.7+yvar colTrim_2]
      [1 colTrim_2]
    }
    scale 0.5
    scale 0.21
    translate x*0.5 
    translate y*(0.51+yvar+0.21/2)
  }                          
  pigment_map{
    [0 p1]
    [0.51+yvar p1]
    [0.51+yvar p2]
    [0.72+yvar p2]
    [0.72+yvar p1]
    [1 p1]
  }
//  #break
*/
/*
//  #case (3) // dom/gold/ dom /gold /dom  rims
  gradient y
  #declare p1=pigment{
    gradient y
    scale 0.1+yvar
    color_map{
      [0 colDom*0.7]
      [1 colDom]
    }
  }
  #declare p2=pigment{colTrim_2}
  #declare p3=pigment{
    gradient y
    triangle_wave
    scale 0.5
    color_map{
      [0 colDom]
      [1 colDom]
    }
  }
  pigment_map{
    [0.00 p1]
    [yvar p1]
    [yvar p2]
    [0.02+yvar p2]
    [0.02+yvar p3]
    [1-yvar-0.02 p3]
    [1-yvar-0.02 p2]
    [1-yvar p2]
    [1-yvar p1 scale -1 translate y]
    [1.00 p1 scale -1 translate y]
  }
//  #break        
*/

//  #case (5) // black/gold/speckled dom/gold/black
  gradient y
  #declare p1=pigment{colDom}
  #declare p2=pigment{colTrim_2}
  #declare p3=pigment{
    bozo
    translate yvar*10
    color_map{
      [0 colDom]
      [0.7 colDom]
      [0.7 colDom*0.3]
      [1 colDom*0.3]
    }
    scale rand(rd)*0.05
  }
  pigment_map{
    [0.00 p1]
    [yvar/2 p1]
    [yvar/2 p2]
    [0.02+yvar/2 p2]
    [0.02+yvar/2 p3]
    [1-yvar-0.08 p3]
    [1-yvar-0.08 p2]
    [1-yvar-0.06 p2]
    [1-yvar-0.06 p1]
    [1-yvar-0.04 p1]
    [1-yvar-0.03 p2]
    [1-yvar p2]
    [1-yvar p1 scale -1 translate y]
    [1.00 p1 scale -1 translate y]
  }
//  #break

/*
//  #case (6) // black/gold/agate dom/gold/black
  gradient y
  #declare p0=pigment{colDom*0.001}
  #declare p1=pigment{colDom}
  #declare p2=pigment{colTrim_2}
  #declare p3=pigment{
    agate
    translate yvar*10
    color_map{
      [0 colTrim_1]
      [0.7 colTrim_1]
      [0.7 colTrim_1*0.3]
      [1 colTrim_1*0.3]
    }
    scale rand(rd)*0.1
  }
  pigment_map{
    [0.00 p3]
    [yvar p3]
    [yvar p2]
    [0.02+yvar p2]
    [0.02+yvar p1]
    [1-yvar-0.08 p1]
    [1-yvar-0.08 p2]
    [1-yvar-0.06 p2]
    [1-yvar-0.06 p0]
    [1-yvar-0.02 p0]
    [1-yvar-0.02 p2]
    [1-yvar p2]
    [1-yvar p3 scale -1 translate y]
    [1.00 p3 scale -1 translate y]
  }
//  #break
*/
/*
// #case (7) 
  gradient y
  #declare p0=pigment{colDom*0.001}
  #declare p1=pigment{colDom}
  #declare p2=pigment{colTrim_2}
  #declare p3=pigment{
    crackle
    translate yvar*10
    color_map{
      [0 colTrim_2]
      [0.025 colTrim_2]
      [0.025 colDom*0.3]
      [1 colDom*0.3]
    }
    scale <0.1,0.03,0.1>*4
  }
  pigment_map{
    [0.00 p0]
    [yvar*0.1 p0]
    [yvar*0.1 p2]
    [yvar*0.1+0.01 p2]
    [yvar*0.1+0.01 p3]
    [1-yvar-0.08 p3]
    [1-yvar-0.08 p2]
    [1-yvar-0.06 p2]
    [1-yvar-0.06 p0]
    [1-yvar-0.02 p0]
    [1-yvar-0.02 p2]
    [1-yvar p2]
    [1-yvar p0 scale -1 translate y]
    [1.00 p0 scale -1 translate y]
  }
//  #break
*/
/*
//  #case (8) 
  gradient y   
  turbulence 0.01
  scale <yvar,1,1>
  #declare p0=pigment{
    gradient x 
    turbulence 1                     
    triangle_wave
    color_map{
      [0 colDom*1.5]
      [1 colDom*0.3*1.5]
    }
  }
  #declare p1=pigment{
    gradient x 
    turbulence 1                     
    triangle_wave
    color_map{
      [0 colDom*0.5]
      [1 colDom*0.3*0.5]
    }
  }
  #declare p2=pigment{
    gradient x                      
    turbulence 1
    triangle_wave
    color_map{
      [0 colDom]
      [1 colDom*0.3]
    }
  }
  pigment_map{
    [0.00 p0]
    [0.02 p1]
    [0.5 p2]
    [1.00-yvar-0.1 p2]
    [1.00-yvar-0.1 p0]
    [1.00-yvar-0.01 p0]
    [1.00-yvar p1]
    [0.99 p1]
    [1.00 p0]
  }
//  #break
*/
/*
//  #case (9) 
  gradient y   
  turbulence 0.01
  #declare p0=pigment{
    gradient x 
    turbulence 1                     
    triangle_wave
    color_map{
      [0 colDom*1.5]
      [1 colDom*0.3*1.5]
    }
  }
  #declare p1=pigment{
    gradient x 
    turbulence 1                     
    scale <yvar,1,1>
    triangle_wave
    color_map{
      [0 colDom*0.5]
      [1 colDom*0.3*0.5]
    }
  }
  #declare p2=pigment{
    gradient x                      
    turbulence 1
    scale <yvar,1,1>
    triangle_wave
    color_map{
      [0 colDom]
      [1 colDom*0.3]
    }
  }
  #declare p3=pigment{
    spherical
    color_map{
      [0 colDom*0.3]
      [0.6 colDom*0.3]
      [0.6 colTrim_2]
      [1 colTrim_2]
    }        
    translate y*0.5
    scale 0.3*<1,0.3,1>
    translate y*(0.02+yvar*0.5)
    translate x*0.5
  }
  pigment_map{
    [0.00 p0]
    [0.02 p1]
    [0.02+yvar*0.5 p1]
    [0.02+yvar*0.5 p3]
    [0.22+yvar*0.5 p3]
    [0.22+yvar*0.5 p2]
    [0.99 p1]
    [1.00 p0]
  }
//  #break
*/
/*
//  #else
  gradient y   
  turbulence 0.01
  #declare p0=pigment{
    agate
    translate yvar*10
    color_map{
      [0 colDom*1.5]
      [1 colDom*0.3*1.5]
    }
  }
  #declare p1=pigment{
    agate
    triangle_wave
    translate yvar*10
    color_map{
      [0 colDom*0.5]
      [1 colDom*0.3*0.5]
    }
  }
  #declare p2=pigment{
    agate
    translate yvar*10
    triangle_wave
    color_map{
      [0 colDom]
      [1 colDom*0.3]
    }
  }
  #declare p3=pigment{
    gradient x
    turbulence 0.3
    triangle_wave
    scale <yvar,1,1>
    color_map{
      [0 colDom*0.3]
      [0.6 colDom*0.3]
      [0.6 colTrim_2]
      [1 colTrim_2]
    }        
  }
  pigment_map{
    [0.00 p0]
    [0.02 p1]
    [0.02+yvar*0.2 p1]
    [0.02+yvar*0.2 p3]
    [0.22+yvar*0.2 p3]
    [0.22+yvar*0.2 p2]
    [0.99 p1]
    [1.00 p0]
  }
//  #end
*/
}              
#end
// -----------------------------------------
// end of mpatBinder macro
// =========================================

#declare shelf_z_dist=14;
#declare shelf_y_dist=44;
#declare shelf_count=11;
#declare horiz_shelf=1;
#while( horiz_shelf <= 3 )
  #declare current_shelf=1;
  #while( current_shelf <= shelf_count )
  //        mShelf(xShelf,yShelf,zShelf,nSet,typeShelf,slackShelf,xBook,sdxBook,sdyBook,sdzBook)
    object{ mShelf(42,    12,    9,     1,   0.        0.5,       1.25, 1,      0,      0) scale<1,1,-1> rotate<90,0,-90> translate< 84,  10-(shelf_y_dist*(horiz_shelf-1)),06.5+(shelf_z_dist*(current_shelf-1))>}
    object{ mShelf(42,    12,    9,     1,   0.        0.5,       1.25, 1,      0,      0) scale<1,1,-1> rotate<90,0,+90> translate< 36,-120+(shelf_y_dist*(horiz_shelf-1)),06.5+(shelf_z_dist*(current_shelf-1))>}
    object{ mShelf(42,    12,    9,     1,   0.        0.5,       1.25, 1,      0,      0) scale<1,1,-1> rotate<90,0,-90> translate< 34,  10-(shelf_y_dist*(horiz_shelf-1)),06.5+(shelf_z_dist*(current_shelf-1))>}
    object{ mShelf(42,    12,    9,     1,   0.        0.5,       1.25, 1,      0,      0) scale<1,1,-1> rotate<90,0,+90> translate<-16,-120+(shelf_y_dist*(horiz_shelf-1)),06.5+(shelf_z_dist*(current_shelf-1))>}
    object{ mShelf(42,    12,    9,     1,   0.        0.5,       1.25, 1,      0,      0) scale<1,1,-1> rotate<90,0,-90> translate<-14,  10-(shelf_y_dist*(horiz_shelf-1)),06.5+(shelf_z_dist*(current_shelf-1))>}
    #declare current_shelf=current_shelf+1;
  #end
  #declare horiz_shelf=horiz_shelf+1;
#end
