/*  
A 3-d implementation of John Conway's 'Game of Life'
  
Saves last generation in file named by the variable tempCellFileName
For example:
        #declare tempCellFileName="lifecells.tmp";
The first three numbers in tempCellFileName are the
cell array dimensions. Then come the 'x layers' of cells.
Each x layer is a 'yz' array of cell values.

*/ 

// some simple but interesting initially 'live' cell arrangements
#declare OOOArray=array[3][3]{
        { 0,  0, -1},
        { 0,  0,  0},
        { 0,  0,  1}
} 
#declare liveGlider5766=array[10][3]{
        { 0,  0,  0},
        { 0,  0,  1},
        { 0,  0, -1},
        { 0, -1,  1},
        { 0, -2,  0},
        { 1,  0,  0},
        { 1,  0,  1},
        { 1,  0, -1},
        { 1, -1,  1},
        { 1, -2,  0}
} 
#declare liveGlider4555=array[10][3]{
        { 1,  0, -1},
        { 1, -1,  0},
        { 1, -1,  1},
        { 1,  0,  2},
        { 0,  0, -1},
        { 0, -1,  0},
        { 0, -1,  1},
        { 0,  0,  2},
        {-1,  0,  0},
        {-1,  0,  1}
}

#declare rules3555=array[4]{3,5,5,5} 
#declare rules5766=array[4]{5,7,6,6}
#declare rules4555=array[4]{4,5,5,5}
#declare rules3423=array[4]{3,4,2,3} 

// these are just some default values 
#declare tempCellFileName="lifecells.tmp";
#declare xSize=9;
#declare ySize=9;
#declare zSize=9;
#declare livePercent=10;
#declare initialGenerationFile="glider5766.lif";
#declare liveArray=liveGlider5766;
#declare centerTheArray=1;
#declare diffScale=1;
#declare cubeTexture=texture{
        pigment{rgb<1,0,0>}
}
#declare gbNewTexture=texture{
        pigment{rgb<0,1,0>}
}                    
#declare gbOldTexture=texture{
        pigment{rgb<1,0,0>}
}                  
#declare gbDiedTexture=texture{
        pigment{rgbt<1,1,1,.8>}
}


#declare RS1=seed(12);
#macro RND(N)
        int(N*rand(RS1))
#end 

#declare numLiveCells=0;

#macro initCellArrays(xs,ys,zs)
        #declare xsz=xs;
        #declare ysz=ys;
        #declare zsz=zs;
        #declare cells1Array=array[xsz][ysz][zsz] 
        #declare cells2Array=array[xsz][ysz][zsz]
        
        fillCellArrayWithZeroes(cells1Array)
        fillCellArrayWithZeroes(cells2Array) 
#end

#macro fillCellArrayRandomly(cArray,pct)
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        
        
        #local i=0;          
        #while (i<asz)
        
                
                #local j=0;
                #while(j<bsz)  
                        #local k=0;
                        #while (k<csz)         
                                #if(rand(RS1)<pct/100)
                                        #declare cArray[i][j][k]=1;
                                        
                                #else                            
                                        #declare cArray[i][j][k]=0;            
                                #end
                                #local k=k+1;    
                                
                        #end
                        #local j=j+1;
                #end
                #local i=i+1;
        #end
        
#end 
#macro fillCellArrayWithZeroes(cArray)
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        
        #local i=0;
        #while (i<asz)
                        
                #local j=0;
                #while(j<bsz)  
                        #local k=0;
                        #while (k<csz)
                                #declare cArray[i][j][k]=0;            
                                
                                #local k=k+1;                                
                        #end
                        #local j=j+1;
                #end
                #local i=i+1;
        #end
        
#end 

#macro liveCellsCount(cArray)
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        #local liveCount=0;
        #local i=0;
        #while (i<asz)
                        
                #local j=0;
                #while(j<bsz)  
                        #local k=0;
                        #while (k<csz)
                                #if (cArray[i][j][k]>0)
                                    #local liveCount=liveCount+1;
                                #end
                                
                                #local k=k+1;                                
                        #end
                        #local j=j+1;
                #end
                #local i=i+1;
        #end
        liveCount
#end                         
#macro translateCells(cArray)
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        #local tempArray=array[asz][bsz][csz]
        
        
        #local liveCount=0;
        #local i=0;
        #while (i<asz)
                        
                #local j=0;
                #while(j<bsz)  
                        #local k=0;
                        #while (k<csz)
                     
                                
                                #local k=k+1;                                
                        #end
                        #local j=j+1;
                #end
                #local i=i+1;
        #end
        
#end                         
#macro initLiveCellsByArray(cArray,liveArray)
        #local numLive=dimension_size(liveArray,1);
        
        #local i=0;
        #while(i<numLive)
            #declare cArray[liveArray[i][0]][liveArray[i][1]][liveArray[i][2]]=1; 
            
            #local i=i+1;
        #end                      
#end        

#macro readCells(cArray,filename)
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        
        
        #fopen inFileHandle filename read
        #read(inFileHandle,fasz)
        #read(inFileHandle,fbsz)
        #read(inFileHandle,fcsz)
        #if(fasz!=asz | fbsz!=bsz | fcsz!=csz)
            
            #debug "Array sizes in the file don't match the scene file array sizes\n" 
            #debug concat ("Reset to ",str(fasz,0,0)," ",str(fbsz,0,0)," ",str(fcsz,0,0),"\n")
            initCellArrays(fasz,fbsz,fcsz)
            #local asz=fasz;
            #local bsz=fbsz;
            #local csz=fcsz;
        #end
        
        #local i=0;
        #while (i<asz)
        
                
                #local j=0;
                #while(j<bsz)  
                        #local k=0;
                        #while (k<csz)         
                                
                                #read(inFileHandle,inCell)
                                #declare cArray[i][j][k]=inCell;
                                
                                #local k=k+1;    
                                
                        #end 
                        
                        #local j=j+1;                    
                        
                #end
                #local i=i+1;
        #end
        
        #fclose  inFileHandle
#end 

#macro writeCells(cArray, filename)                      
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        
        #fopen outFileHandle filename write 
        #write(outFileHandle, asz) 
        #write(outFileHandle,",")
        #write(outFileHandle, bsz)
        #write(outFileHandle,",")
        #write(outFileHandle, csz)
        #write(outFileHandle,",")
        #write(outFileHandle,"\n")
        #local i=0;
        #while (i<asz)
        
                
                #local j=0;
                #while(j<bsz)  
                        #local k=0;
                        #while (k<csz)         
                                
                                #write(outFileHandle,cArray[i][j][k])
                                #write(outFileHandle,",")
                                #local k=k+1;    
                                
                        #end 
                        #write(outFileHandle,"\n")
                        #local j=j+1;                    
                        
                #end  
                #write(outFileHandle,"\n")
                #local i=i+1;
        #end
        
        #fclose  outFileHandle
#end 



#macro countLiveNeighbors(cArray,i,j,k,wrap)          
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        
        #declare nbrCnt=0; 
        
        #local icnt=0;
        #local ii=i-1;
        #if(ii<0) 
                #if(wrap)
                #local ii=ii+asz;
                #else
                #local ii=0;
                #local icnt=1;
                #end
        #end          
            
        #while (icnt<3 )
                
                #local jcnt=0;
                #local jj=j-1;
                #if(jj<0)
                        #if(wrap)
                        #local jj=jj+bsz;
                        #else
                        #local jj=0;
                        #local jcnt=1;
                        #end
                #end          
                    
                #while (jcnt<3 )
                              
                        #local kcnt=0;      
                        #local kk=k-1;
                        #if(kk<0)     
                                #if(wrap)
                                #local kk=kk+csz;
                                #else
                                #local kk=0;
                                #local kcnt=1;
                                #end
                        #end          
                            
                        #while (kcnt<3 )
                                
                                #if(ii!=i | jj!=j | kk!=k)
                                        #if(cArray[ii][jj][kk] > 0)
                                                #declare nbrCnt=nbrCnt+1;
                                        #end
                                #end      
                                
                                #local kk=kk+1;
                                #if(kk>csz-1)
                                        #if(wrap)
                                        #local kk=kk-csz;
                                        #else
                                        #local kcnt=2;
                                        #end
                                #end           
                                #local kcnt=kcnt+1; 
                        #end
                        
                        
                        #local jj=jj+1;
                        #if(jj>bsz-1) 
                                #if(wrap)
                                  #local jj=jj-bsz;
                                #else
                                  #local jcnt=2;
                                #end
                        #end           
                        #local jcnt=jcnt+1; 
                #end
                        
                
                #local ii=ii+1;
                #if(ii>asz-1)
                        #if(wrap)
                        #local ii=ii-asz;
                        #else
                        #local icnt=2;
                        #end
                #end           
                #local icnt=icnt+1; 
        #end
                                                         
        nbrCnt                                                         
#end 

#macro nextGenCells(oldCArr,newCArr,rulesArray,wrap)

                #local asz=dimension_size(oldCArr,1);
                #local bsz=dimension_size(oldCArr,2);
                #local csz=dimension_size(oldCArr,3);
                
                #local i=0;
                #while (i<asz)
                
                        
                        #local j=0;
                        #while(j<bsz)  
                                #local k=0;
                                #while (k<csz)         
                                        #local numAlive=countLiveNeighbors(oldCArr,i,j,k,wrap); 
                                        
                                        #if(oldCArr[i][j][k] >0)
                                                #if(numAlive>=rulesArray[0] & numAlive<=rulesArray[1])
                                                     #declare newCArr[i][j][k]=2;
                                                     
                                                #else
                                                     #declare newCArr[i][j][k]=-1;   
                                                #end
                                        
                                        #else
                                                #if(numAlive<rulesArray[2]|numAlive>rulesArray[3])
                                                     #declare newCArr[i][j][k]=0;
                                                #else
                                                     #declare newCArr[i][j][k]=1;         
                                                     
                                                     
                                                #end
                                                                                
                                        #end                                        
                                        #local k=k+1;
                                #end
                                #local j=j+1;
                        #end
                        #local i=i+1;
                #end        

        
#end

#declare cubeCells=0;
#declare growingBlobs=1;

#declare noWrap=0;
#declare torusWrap=1; 

#declare fromRandom=0;
#declare fromFile=1;
#declare fromLiveArray=2;

#macro placeCells(cArray,cType,wrap)
        #local asz=dimension_size(cArray,1);
        #local bsz=dimension_size(cArray,2);
        #local csz=dimension_size(cArray,3);
        
          #if(cType=0)
            union{      
          #end
          #if(cType=1)
            blob{   threshold .4      
          #end                        
             
                #local i=0;
                #while (i<asz)
                
                        
                        #local j=0;
                        #while(j<bsz)  
                                #local k=0;
                                #while (k<csz)         
                                        
                                        #local cellValue=cArray[i][j][k]; 
                                        #local absCellValue=abs(cellValue);
                                        #if(absCellValue>0)
                                        
                                                #if(cType=cubeCells) 
                                                  //#if(cellValue>0)
                                                  box{
                                                        <i,j,k>,<i+1,j+1,k+1> 
                                                        
                                                        //texture{cubeTexture} 
                                                        #if(cellValue=2) 
                                                                  texture{gbOldTexture} 
                                                        #end    
                                                        #if(cellValue=1)
                                                                  texture{gbNewTexture} 
                                                        #end
                                                        #if(cellValue=-1)
                                                                  texture{gbDiedTexture} 
                                                        #end
                                                  }               
                                                  //#end
                                                
                                                #end
                                                #if(cType=growingBlobs) 
                                                          
                                                          #declare rr=1+diffScale*(absCellValue-1);
                                                          
                                                          sphere{<i+1/2,j+1/2,k+1/2>,rr,1
                                                                #if(cellValue=2) 
                                                                  texture{gbOldTexture} 
                                                                #end
                                                                #if(cellValue=1)
                                                                  texture{gbNewTexture} 
                                                                #end
                                                                #if(cellValue=-1)
                                                                  texture{gbDiedTexture} 
                                                                #end
                                                          }  
                                                          
                                                          #if(wrap)
                                                          
                                                         
                                                          #end
                                                        
                                                #end
                                        #end
                                        #local k=k+1;
                                #end
                                #local j=j+1;
                        #end
                        #local i=i+1;
                #end        
           }
        
        
#end


#macro translateLiveArray(liveArray,movex,movey,movez)
        #local numLiveCells=dimension_size(liveArray,1);
        #local i=0;
        #while (i<numLiveCells)
                   
                #declare liveArray[i][0]=liveArray[i][0]+movex;
                #declare liveArray[i][1]=liveArray[i][1]+movey;
                #declare liveArray[i][2]=liveArray[i][2]+movez;
                
                #local i=i+1;
        #end

#end 
#macro centerLiveArray(liveArray)
        #local numLiveCells=dimension_size(liveArray,1);
        #local i=0;
        #local tx=0;#local ty=0;#local tz=0;
        #while (i<numLiveCells)
                   
                #local tx=liveArray[i][0]+tx;
                #local ty=liveArray[i][1]+ty;
                #local tz=liveArray[i][2]+tz;
                
                #local i=i+1;
        #end                                
        #local avgx=int(tx/numLiveCells);
        #local avgy=int(ty/numLiveCells);
        #local avgz=int(tz/numLiveCells);
        
        #local mmx=int((xSize+1)/2)-1;
        #local mmy=int((ySize+1)/2)-1;                                              
        #local mmz=int((zSize+1)/2)-1;                                      
        
        translateLiveArray(liveArray,avgx+mmx,avgy+mmy,avgz+mmz)

#end 


#declare genNum=clock;  


#macro generateSceneCells(xS,yS,zS,rules,gen1From,cellType,wrap)
// xS = 'x' direction width of cell array (integer)

// yS = 'y' direction width of cell array (integer)

// zS = 'z' direction width of cell array (integer)

// rules = an array[4] of integers specifying the Life rules
// Using RulesArray=array[4]{r1,r2,r3,r4} 
// makes the rule: 
//        if a cell is already 'live' and it has from 
//        r1 through r2 (inclusive) live neighbors, then 
//                it remains alive in the next generation.
//                That is, it will survive. 
//        Otherwise it becomes 'dead'.
//        On the other hand. if the cell is now dead, then 
//        if it has from r3 through r4 (inclusive) live neighbors then 
//                it will be alive in the next generation.

// gen1From tells where the first generation comes from. 
//      fromLiveArray takes an array of cell indexes and each
//              cell in the array will be 'alive' in the first generation.
//              For example, using the array
//                      #declare liveArray= array[2][3]={{0,0,1},{0,0,0}}
//              would make two cells alive: the cell at 0,0,1 and the cell at 0,0,0
//              The second index of the array must be '3' since a cell is specified
//              by three indices
//   
//      fromFile gets the first generation from a file name by 'initialGenerationFile'
//              The file contains the cell Values for all of the cells in the Life game
//              cell value > 0 is alive. cell value <=0 is dead.
//              
//      fromRandom generates the first generation randomly with livePercent of the cells
//              alive.

// cellType is either 'cubeCells' (plain cubes occupy live cells) or 'growingBlobs' (where
//      live cells which were 'just born' (weren't alive in the last generation) are green.
//      Alive cells which survived from the last generation are larger and red.
//      Cells which were alive in the last generation but are now dead are small and white)

// wrap tells whether to wrap the cell array or not.
//      noWrap means that a cell on the edge of the cell array has no neighboring cells
//              outside the array.
//      torusWrap 'wraps' the array around from top to bottom and left to right. so a cell
//              at the edge has neighboring cells on the other side of the array.



        initCellArrays(xS,yS,zS) 
        #if(genNum=0) // First time through, we need to initialize the cells
                #switch(gen1From)
                        #case (fromRandom)
                        // We can initialize the Cells randomly:                     
                                fillCellArrayRandomly(cells2Array,livePercent)    
                        #break
                        #case (fromFile)
                        // Or read a file:
                                readCells(cells2Array, initialGenerationFile) 
                        #break
                        #case (fromLiveArray)
                        // Or read an array of 'live cell' indexes:
                                #if(centerTheArray)
                                        centerLiveArray(liveArray)
                                #end
                                initLiveCellsByArray(cells2Array,liveArray)
                        #break
                #end
        #else   // Second time through, and thereafter, get last generation from the file 
                // tempCellFileName and compute the next generation
                
                readCells(cells1Array, tempCellFileName ) 
                
                nextGenCells(cells1Array,cells2Array,rules,wrap)
                
        #end        
        
        // put the live cells into the scene:
        #if(liveCellsCount(cells2Array)>0)
                object{
                        placeCells(cells2Array,cellType,wrap)
                        texture{            // default texture
                                pigment{rgbt<1,1,1,1>}
                        }
                        translate <-xS/2,-yS/2,-zS/2> 
                        
                } 
        #else
                #warning "No live cells left!\n"
                box{<-xS/2,-yS/2,-zS/2>,<xS/2,yS/2,zS/2> pigment{rgbt<1,1,1,1>}}        
        #end  
        // lastly, write current cell generation to tempCellFileName for the next generation/frame
        writeCells(cells2Array,tempCellFileName) 


#end                                  

