// Balls
// Copyright (C) 2003
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// ---------------------------------------------------------------------------
// Persistence of Vision Ray Tracer Scene Description File
// File: balls.pov
// Vers: 3.5
// Desc: Balls on a shelf
// Date: 10/2003
// Auth: Collaborative work
// ---------------------------------------------------------------------------
#version 3.5;

// ========================================
//  Notes
// ========================================
//
// First pass :  povray +Iballs.pov +w400 +h400 +fn 
// Second pass:  povray +Iballs.pov +w800 +h800 +fn +a0.1
//

// ========================================
//  Switches
// ========================================
#declare radiosity_stage = 2;  // 0 - No radiosity
                               // 1 - Dummy render - save radiosity data
                               //     Disable transparencies, reflections, complex isosurfaces, etc.
                               // 2 - Final render - load radiosity data
#declare use_balls     = true;
#declare IW_Quality    = 1; //radiosity_stage-1;    // ISO-Wood quality
#declare IW_ISOmaxgrad = 2;                    // ISO-Wood max gradient


// ========================================
//  Settings
// ========================================
global_settings {
  assumed_gamma 1.0
  max_trace_level 10
  #if (radiosity_stage > 0) 
    radiosity {
      #if (radiosity_stage = 1) 
      // Save radiosity data
      pretrace_start 0.08
      pretrace_end   0.01
      count 250
      nearest_count 20
      error_bound 0.5
      recursion_limit 1
      low_error_factor .9
      gray_threshold 0
      minimum_reuse 0.015
      brightness 1.0
      adc_bailout 0.01/2
      save_file "balls.rad"
      normal on
    #else
      // Load radiosity data
      pretrace_start 1 pretrace_end 1 
      load_file "balls.rad" 
      error_bound 0.5
      recursion_limit 1
      always_sample off
      normal on
    #end
    }
  #end
} 

// ========================================
//  Camera
// ========================================
#declare EyePos  = <0.0, 6, -34>;
#declare EyeLook = <0.0, 6,  0.0>;
#declare EyeAngle = 20;
camera {
  location  EyePos
  direction 1.5*z
  right     x*image_width/image_height
  look_at   EyeLook
  angle     EyeAngle
}

// ========================================
//  Sky
// ========================================
sky_sphere {
  pigment {       
    gradient y
    color_map {
      [0 rgb 1]
      [1 rgb<55,105,201>/255]
    }
  }
}

// ========================================
//  Light
// ========================================
light_source{
  2000*y
  color rgb <253,249,237>/255*.65
  fade_power 2 fade_distance 2000
  area_light 1000*x 1000*z  10,10 jitter adaptive 1 circular orient
  rotate -x*65
  rotate y*20
}


// ========================================
//  Default settings
// ========================================
#default {
  texture {
      finish {
        ambient 0.00
        diffuse 0.85
      }
  }
}



// ========================================
//  Shelf
// ========================================
#local plank_thickness = .4;
#local box_y = 2.5;
#local box_x = 2.5;
#local box_z = 2.5;
#local box_nx = 4;
#local box_ny = 4;
#declare GammaFact=1;
#include "isowood3.inc"
#include "iw_textures.inc"
#declare CM_IW = CM_IW_03;

// Rear plank
object {
  IW_Plank_RoundW  (
    17,
    (box_nx*box_x+(box_nx+1)*plank_thickness), plank_thickness, box_ny*box_y+(box_ny+1)*plank_thickness
    2, 0, 0.05, 0.0035,
    pigment { P_IW_10 color_map {CM_IW} scale 5} , Fin_W_02 , No_Normal, 2
  )
  rotate x*-90
  translate -x*(box_nx*box_x+(box_nx+1)*plank_thickness)/2
  translate z*(box_z+plank_thickness)
}
// Black panel behind (because of rounded plank, you can see through around upper corners
box {<-1,-1,0>,<1,1,1> scale 20 translate z*(box_z+plank_thickness)}

// Top plank
object {
  IW_Plank_RoundW  (
    17,
    (box_nx*box_x+(box_nx+1)*plank_thickness), plank_thickness, box_z
    2, 0, 0.05, 0.0035,
    pigment { P_IW_10 color_map {CM_IW} scale 5 rotate y*90 translate 5} , Fin_W_02 , No_Normal, 2
  )
  translate -x*(box_nx*box_x+(box_nx+1)*plank_thickness)/2  
  translate y*(box_ny*box_y+box_ny*plank_thickness)
}

// Bottom plank
object {
  IW_Plank_RoundW  (
    15,
    (box_nx*box_x+(box_nx+1)*plank_thickness), plank_thickness, box_z
    2, 0, 0.05, 0.0035,
    pigment { P_IW_10 color_map {CM_IW} scale 5 rotate y*90 translate -3} , Fin_W_02 , No_Normal, 2
  )
  translate -x*(box_nx*box_x+(box_nx+1)*plank_thickness)/2  
}

// Left plank
object {
  IW_Plank_RoundW  (
    15,
    plank_thickness, box_ny*box_y+(box_ny-1)*plank_thickness, box_z
    2, 0, 0.05, 0.0035,
    pigment { P_IW_10 color_map {CM_IW} scale 5 rotate y*90 rotate z*90 translate -3} , Fin_W_02 , No_Normal, 2
  )
  translate -x*(box_nx*box_x+(box_nx+1)*plank_thickness)/2
  translate  y*plank_thickness
}

// Right plank
object {
  IW_Plank_RoundW  (
    15,
    plank_thickness, box_ny*box_y+(box_ny-1)*plank_thickness, box_z
    2, 0, 0.05, 0.0035,
    pigment { P_IW_10 color_map {CM_IW} scale 5 rotate y*90 rotate z*90 translate -5} , Fin_W_02 , No_Normal, 2
  )
  translate x*(box_nx*box_x+(box_nx-1)*plank_thickness)/2
  translate y*plank_thickness
}

// Horizontal interior planks
#local i=1;
#while (i<box_ny)
  object {
    IW_Plank_RoundW  (
      17+i,
      (box_nx*box_x+(box_nx-1)*plank_thickness), plank_thickness, box_z-0.05
      2, 0, 0.05, 0.0035,
      pigment { P_IW_10 color_map {CM_IW} scale 5 rotate y*90 translate 5+i} , Fin_W_02 , No_Normal, 2
    )
    translate z*0.05
    translate -x*(box_nx*box_x+(box_nx-1)*plank_thickness)/2
    translate y*i*(box_y+plank_thickness)
  }
  #local i=i+1;
#end

// Vertical interior planks
#local st = seed(1);
#local j=0;
#while (j<box_ny)
  #local i=1;
  #while (i<box_nx)
    object {
      IW_Plank_Round  (
	    i*j,
	    plank_thickness, box_y, box_z-0.05
	    2, 0, 0.05, 0.0035
	    pigment { P_IW_10 color_map {CM_IW} scale 5 rotate y*90 rotate z*90 translate -5-rand(st)*3} , Fin_W_02 , No_Normal
      )
      // Small random rotation
      #if (rand(st) < .5)
        translate -(box_z-0.05)/2
        rotate y*(2*rand(st)-1)*1
        translate (box_z-0.05)/2
      #end
      #if (rand(st) < .5)
        rotate z*(2*rand(st)-1)*1	
      #end
      
      translate z*0.05
      translate y*plank_thickness
      translate -x*(box_nx*box_x+(box_nx+1)*plank_thickness)/2
      translate  x*i*(box_y+plank_thickness)
      translate y*j*(box_y+plank_thickness)
    }
    #local i=i+1;
  #end
  #local j=j+1;
#end


// ========================================
//  Balls
// ========================================
// Macro to place balls
#macro ball_at (_x,_y)
  translate x*((box_y+plank_thickness)/2+(_x-box_nx/2)*(box_x+plank_thickness))
  translate y*((box_y+plank_thickness)/2+_y*(box_x+plank_thickness))
  translate z*box_z/2
#end

// From top left corner to bottom right corner
#if (use_balls)
  #include "ball-0x3.inc"
  #include "ball-1x3.inc"
  #include "ball-2x3.inc"
  #include "ball-3x3.inc"
  #include "ball-0x2.inc"
  #include "ball-1x2.inc"
  #include "ball-2x2.inc"
//  #include "ball-3x2.inc"
//  #include "ball-0x1.inc"
//  #include "ball-1x1.inc"
//  #include "ball-2x1.inc"
//  #include "ball-3x1.inc"
//  #include "ball-0x0.inc"
//  #include "ball-1x0.inc"
//  #include "ball-2x0.inc"
//  #include "ball-3x0.inc"
#end
