// Persistence of Vision Ray Tracer Scene Description File
// File: voidsphinx.inc
// Vers: 3.1
// Desc: include file to standardize light source, background color, etc.
// Date: 13 Oct '00
// Last: 04 Dec '00
// Auth: Dawn McKnight
// Note: Units are arbitrarily defined as centimeters.  

/* 

This is a POV-Ray include file, which requires the POV-Ray renderer to interpret
and render.  POV-Ray can be downloaded free, for most OSes, from the official
web page at <http://www.povray.org>.

This include file was written under official release 3.1g for MacIntosh.  I have run
it under the windows version of POV, but make no guarantee that it will work as I 
intend it to under any version but the one I used to write it.

This is the include file which defines the Void Sphinx fighter.  It was written
over a period of aproximately two and a half months, by Dawn McKnight, with all
coding done by hand.  Josh English provided very helpful hints on how to do the
decals.  I highly recomend Chris Colefax's galaxy include file, for creating the 
background, but I have not included it here, because it's not a standard file 
distributed with POV.

Glenn McCarter (gmccarter@hotmail.com) and N.B. (beliaev@utu.fi) wrote the "Haze Grey"
texture, which I stole from their "First Strike at Pearl" project.

You may redistribute this code, provided you credit me as the original author.  You 
may modify this code to your little heart's content, again, provided you credit me
as the original author.

The Void Sphinx, as modeled, is a really massive thing.  It's ten meters long, and
about that wide, though only about six meters tall.  It includes the ability to mark
the fighter with your unit designation, the designation for your fighter, and two 
large pieces of artwork, which by default I've made a squadron logo and an individual
piece of 'nose' art.

If you don't set the variables for these, they will default to those for VF-9, fighter
106, with the Fighting Aardvark logo and the 'Dixie' hero patch.  It's good to be
the designer. P)

The program expects a square (same pixel size high as wide) image for each of the two 
large graphics.  The squadron logo is expected to be an 8-bit palleted gif, with color
0 used for transparency.  The hero art is expected to be a .png, with alpha channel 
transparency.  If you want to change this, look down the page for the SquadPatch and
HeroPatch declarations, I'll include some notes there on how to change it.

These are the variables to set in your scene file, before including voidsphinx.inc:

#declare SquadDesig = "VF-9" 	// this, clearly, is your squadron designation.
#declare BirdDesig = "106"		// designation of the number of your fighter
#declare HeroPatch = "dixie.png"	// your personal art for the port side.
#declare SquadPatch = "faardvark.gif"	// your squadron logo for the starboard side

Change the text within the quotes to change what shows up on your fighter.  If you
don't declare them, they default as shown.

Once you've got those, you'll need to put this in:

#include "voidsphinx.inc"

and then call the fighter with:

object { VoidSphinx }

The fighter lies mostly along the X axis, extending roughly +/- 500 units, with
the nose pointing into negative X land.

Some lines may need to be adjusted for length, though I've tried to keep a standard
width.  If you get odd errors at compile time, look at that.

*/


#ifndef (SquadDesig) #declare SquadDesig = "VF-9" #end
#ifndef (BirdDesig) #declare BirdDesig = "106" #end
#ifndef (HeroPatch) #declare HeroPatch = "dixie.png" #end
#ifndef (SquadPatch) #declare SquadPatch = "faardvark.gif" #end


#include "colors.inc" 
#include "metals.inc"
#include "glass.inc"

#declare Port = 1;
#declare Starboard = 2;
#declare Upper = 1;
#declare Lower = 2;

//#declare SphinxSkin =  texture { T_Chrome_4A } // used in early development, retained for
//                                               // historical reasons

#declare P_Stealth1 = pigment {color Gray15}
#declare SphinxSkin = texture { pigment { P_Stealth1 } finish { F_MetalA  } }
#declare HazeGrey = // This is the BB_Tex from "First Strike at Pearl" renamed.
texture {
	pigment { color rgb <121, 140, 150>/255 }
	finish {
		ambient <.5,.77,1>*.1
		diffuse .6
		phong .5
		phong_size 10
		} // close finish
	normal { bumps .07 scale 7 }
	} // close texture 



// _______________ This section defines the RCS thrusters for the pylons ___________________

#local CutCorner = difference {
	
	box {
		<-1, 0, 0>
		<2, -5, -5>
		} // close box
	
	cylinder {
		<-1.1, 0, 0>,
		<2.1, 0, 0>,
		4
		} // close cylinder
	
	
	} // close difference CutCorner
	

#declare BasicPlate = difference { 	// this difference creates the basic shape of the
									// thruster box, with rounded corners.

	box {
		<0, -25, -9>,
		<1, 25, 9>
		} // close box 
	
	object {
		CutCorner
		translate <0, -21.1, -5.1>
		} // close CutCorner
		
	object {
		CutCorner
		rotate -90*x
		translate <0, -21.1, 5.1>
		} // close CutCorner
		
	object {
		CutCorner
		rotate 90*x
		translate <0, 21.1, -5.1>
		} // close CutCorner
		
	object {
		CutCorner
		rotate 180*x
		translate <0, 21.1, 5.1>
		} // close CutCorner
		
	} // close difference
	
#declare AdvPlate = difference { // cutting the center out...

	 object { BasicPlate }
	
	 object {
		BasicPlate
		scale <2, 0.85, 0.8>
		translate <-0.5, 0, 0>
		} // close object 
		
	pigment { color Yellow }
	
	} // close difference AdvPlate

#local HoriBar = difference { // horizontal bars for the grill
	
	box {
		<0, -.5, -8>,
		<1, .5, 8>
		} // close box
		
	box {
		<-.1, -1.6, -7.4>,
		<.5, 1.6, 7.4>
		} // close box
	
	} // close difference HoriBar
	
#local VertiBar = difference { //vertical bars for the grill
	
	box {
		<0, -24, -.5>,
		<1, 24, .5>
		} // close box
		
	box {
		<-.1, -23, -1.6>,
		<.5, 23, 1.6>
		} // close box

	} // close difference VertiBar

#declare Grill = union { // the grill!  Designed to keep idiots from sticking their heads
							// inside the thruster.
	
	object { HoriBar }
	object { HoriBar translate <0, 7.5, 0> }
	object { HoriBar translate <0, 15, 0> }
	object { HoriBar translate <0, -7.5, 0> }
	object { HoriBar translate <0, -15, 0> }
	object { VertiBar translate <0, 0, 4> }
	object { VertiBar translate <0, 0, -4> }
	pigment { color White }
	} // close union Grill	


#declare RCSLit =  // this is the light in the thruster.
	prism { // when the thruster is 'lit', you see this box.
		conic_sweep
		linear_spline
		0, // y=0, where the 'point' will occur
		1, // y=1, where the bounds I set will occur
		5,
		<25, 9>, <-25, 9>, <-25, -9>, <25, -9>, <25, 9>
		pigment { color Clear }		
		finish { ambient 0.8 diffuse 0.6 }
		rotate -90*z
		translate <2, 0, 0>
		interior {
			media {
	 			emission rgb 2
	 			density {
					gradient x
					color_map {
						[0.0 color White filter 1]
						[1.0 color Blue filter 1]
						} // close color_map
					} // close density
				} // close media
			} // close interior
		hollow

		} // close prism


#macro RCSBox ( Burn ) // this is what gets called to display the thruster and (possibly) the flame
	 union {
		object { AdvPlate }
		object { Grill } 
		#if (Burn)
			RCSLit 
			object {
				BasicPlate
				scale <15, 1, 1>
				pigment { Clear }
				interior {	
					media { 
						scattering { 1, rgbf 0.1 } // rgbf set low so as to get the 'glow' effect.
						} // close media
					} // close interior
				translate -5.99*x
				hollow
				} // close object BasicPlate
			#end // if statement
		} // close union RCSBox
#end // macro RCSBox

#local RCCBasic = intersection { // this thruster is for the forward edge of the wing.
									// it's designed to match the curve of the leading edge.
	
	cylinder { // same size (radius) as the one which forms the leading edge.
		<0, 26, 0>,
		<0, -26, 0>,
		25
		open
		} // close cylinder

	object { BasicPlate scale 10*x translate 17*x } // cuts the hole for the thruster to fire 
	} // close intersection RCCFace


#local RCCAdv = difference {
	object { RCCBasic }
	object { RCCBasic scale 1.1 translate -4*x }
	object { 
		BasicPlate
		scale <10, 0.85, 0.8>
		translate <18, 0, 0>
		} // close object BasicPlate

	translate -25*x
	pigment { Yellow }
	} // close difference RCCAdv

#declare RCCFace = union { // the whole thruster face plate
	object { RCCAdv }
	object { Grill translate -1.2*x}
	rotate 180*y
	} // close union RCCFace

#macro RCCBox ( Burn ) // called to display thruster and (possibly) flame
	union {
		object { RCCFace }
		#if (Burn)
		 	RCSLit 	
			object {
				BasicPlate
				scale <15, 1, 1>
				pigment { Clear }
				interior {	
					media { 
						scattering { 1, rgbf 0.1 } // rgbf set low so as to get the 'glow' effect.
						} // close media
					} // close interior
				translate -5.99*x
				hollow
				} // close object BasicPlate
	
			#end // if statement
		} // close union
	#end // macro RCCBox
	
// ____________________________________ End RCS thrusters for Pylons ___________________

// ____________ This section defines the bussard collectors for the nacelles ___________
// Note that there is a duplicate definition of 'basic nacelle' here.

#local Basic_Nacelle = difference {	// this is needed here, because I have to shape the
									// collectorto it.
	intersection {
		merge {
			cylinder {	// top half of nacelle
				<500, 0, 0>,
				<-500, 0, 0>,
				100
				} // close cylinder
			
			box {	// bottom half of nacelle
				<500, -150, -100>,
				<-500, 0, 100>
				}
				
			} // close merge 
		
		box {	// cut in the bottom half of the nacelle
			<-501, -125, -125>,
			<501, 500, 500>
			rotate -45*x
			}

		} // close intersection

	box {	// the idea here is to cut off the tip of the nacelle so I can stick in bussard collector
		<-450, -400, -400>,
		<-610, 400, 400>
		rotate 15*z
		}	// close box
	
	}	// close difference


#local Bus1 = intersection { // this forms the basic shape of half the collector

prism { 
	quadratic_spline
	linear_sweep
	-1, // height 1
	1, // height 2.. this is just for visualization purposes, I'll extend it later.
	6, // number of control points
	<200, 100>, <400, -25>, <185, -135>, <0, -25>, <200, 100>, <400, -25>
	} // close prism
	
	box { // trims the prism, giving me just the part of the curve I need.
		<214, -2, 120>,
		<-10, 2, -25>
		}
		
	} // close intersection Bus1
	

#local Bus2 = merge { // take two prisms, flip one over, weld 'em together.
	object { Bus1 }
	object { Bus1 rotate 180*x translate -49.99*z }
	} // close merge Bus2

#local BusCap = intersection {	// creates the rounded sides for the collector.  They
								// have the same slope as the prism.
	lathe {
		quadratic_spline
		4, 
		<-110, 185>, <0, 0>, <125, 200>, <0, 400>
		sturm
		} // close lathe 
	
	box { // again, trims to give me just the part I want to work with
		<-130, -10, -130>,
		<130, 214, 0>
		} // close box
	
	} // close intersection BusCap

#local BussCut = merge {	// creates the basic shape of the front of the collector
	object { Bus2 scale 25.02*y rotate 90*x rotate 90*z translate 25*x}
	object { BusCap translate -25*z }
	object { BusCap rotate 180*y translate 25*z }
	} // close merge BussCut



#local BusShape = merge { 	// extends the basic shape backwards, so that it has some
							// room to play with, in extending it into the nacelle.
	object { BussCut }
	box { <-125, 213.9, -25.001>, <125, 314, 25.001> }
	cylinder { <0, 213.9, -25>, <0, 314, -25>, 125 }
	cylinder { <0, 213.9, 25>, <0, 314, 25>, 125 }
	translate -150*y
	} // close union BusShape



#local ClippedBus0 = difference { // flips the collector into the right plane for the 
									// rest of the nacelle, and clips off the end to fit
									// the diagonally cut end of the nacelle.
	object {
		BusShape rotate -90*z 
		scale <0.75, 0.65, 0.8>
		rotate 90*x
 		} // close object BusShape

object { Basic_Nacelle translate <545, 25, 0> }

	} // close difference ClippedBus0
	
#local ClippedBus1 = difference { // creates a hollow glass exterior to the collector
	object { ClippedBus0 }
	object { ClippedBus0 scale 0.97 }	
	texture { T_Glass3 }
	} // close difference ClippedBus1


#declare BussardCollector = union {  // and, finally, the whole schebang.
	object { ClippedBus1 }
	object { 
		ClippedBus1 
		scale 0.971
		texture { T_Glass3 }
		interior {
			I_Glass
			media {
		 		emission rgb 1
		 		density {
	 				cylindrical
	 				rotate 90*x
					scale <300, 125, 125>
					translate 150*x
					color_map {
						[0.0 color OrangeRed filter 0.75]
						[0.3 color Red filter 0.75]
						[0.7 color Firebrick filter 0.75]
						[1.0 color Scarlet filter 0.75]
						} // close color_map
					} // close density
				} // close media
	  		} // close interior
		hollow
		} // close object clippedbus1
	}// close union BussardCollector
	
// __________________ End of Bussard Collector Section _________________________

// __________ This section defines the navigational running lights _____________

#macro NavLight ( LightColor ) // the navigation light.  Accepts a parameter that tells
								// it what color to be.
union {

	light_source { <0, 2.5, 0>, LightColor } // what sheds light.  Well, duh.

	difference { // this difference gives us a fixture, instead of a sphere
		merge { // merges the sphere with the torus
			sphere { // the basic shape of the light fixture.  In RL, the glass is colored, but that
					 // didn't work well here, so the light is colored, instead.
				0,
				5
				} // close sphere
			
			torus { // puts a 'lip' on the fixture for aesthetic purposes.
				5,
				1
				} // close torus
			} // close merge
		
		box { // this is the box that cuts the ringed sphere in half.
			<-9, -9, -9>,
			<9, 0, 9>
			} // close box
		
		texture { T_Glass3 } // I tried various glass finishes.  I liked this one best.
		interior { I_Glass }
		} // close difference 

	difference {	// creates the 'glow' by use of scattering media
		sphere { // to hold the media
			0,
			10
			} // close sphere 
		
		box { // cuts it in half, so we don't have any unpredictable results from scattering media
			<-11, -11, -11>,
			<11, 0, 11>
			} // close box
			
		pigment { Clear }
		hollow
		interior {	
			media { 
				scattering { 1, rgbf 0.1 } // rgbf set low so as to get the 'glow' effect.
				} // close media
			} // close interior
		
		} // close difference
	} // close union

#end // macro NavLight

// ______________________ End of Navigation definition _______________________

// ______________ This section defines the Star Fleet Shield for the decals __

#declare BaseSFShield = // the familiar delta shield
	prism {
 		linear_sweep // or conic_sweep for tapering to a point
  		cubic_spline // linear_spline | quadratic_spline | cubic_spline
  		-0.5,        // height 1
   		0.5,        // height 2
  		10,          // number of points
		<-4, 4>, // control point 1
		<5, 0>, // this is the point that has to match
		<-0.6, -3>,
		<-8, -4.9>,
		<-9.8, -4.8>,
		<-5, -1>,
		<-11, 5>,
		<-4, 4>,
		<5, 0> // matched
		<-0.6, -3> // control point 2
 		sturm
   		scale 0.9*z
		} // close prism

#local SFBox1 = difference { // and now, the red stripe...

	box {
		<-4.99, -0.5, -4.4>,
		<-64, 0.5, 4.4>
		} // close box
		
	object { BaseSFShield scale 2*y } // makes the end that fits against the shield
										// have the same curve
	
	box { <-4, -1, -5>, <-8.5, 1, -3.5> } // these both cut off excess 'bits'.
	box { <-4, -1, 5>, <-8, 1, 3.5> }
	
	translate -2*x
	} // close difference 

#declare SFStripe = union {	
	object { BaseSFShield pigment { Yellow } }

	difference {
		object { SFBox1 }
		object { SFBox1 scale <0.9, 1.1, 0.9> translate -2*x } // hollow the stripe
		box {<-68, -1, -5>, <-60, 1, 5> } // cut the end off
		pigment { Red }
		} // close difference
	
	} // close union SFStripe

// _____________ End of SFStripe / SFShield __________________________

// _____________ This section creates the nacelles. __________________
// note that there is a duplicate definition of 'basic nacelle' in the section for
// the bussard collector.


#declare Basic_Nacelle = difference {	
	intersection {
		merge {
			cylinder {	// top half of nacelle
				<500, 0, 0>,
				<-500, 0, 0>,
				100
				} // close cylinder
			
			box {	// bottom half of nacelle
				<500, -150, -100>,
				<-500, 0, 100>
				}
				
			} // close merge 
		
		box {	// cut in the bottom half of the nacelle
			<-501, -125, -125>,
			<501, 500, 500>
			rotate -45*x
			}

		} // close intersection

	box {	// the idea here is to cut off the tip of the nacelle so I can stick in bussard collector
		<-450, -400, -400>,
		<-610, 400, 400>
		rotate 15*z
		}	// close box
	
	}	// close difference
		
#macro Comp_Nacelle (Side, Tier) // fancy macro which uses the placement of the nacelle to determine
									// where the decals are supposed to go.

difference {
	object { Basic_Nacelle }
	
	difference {	// cutting box 1 -- creates hole for forward end of glow tube
		box {
			<-470, 0, -110>,
			<0, 110, 110>
			} // close box
			
		box {
			<0, -10, -120>,
			<-100, 120, 120>
			rotate 15*z
			translate <-405, 0, 0>
			} // close box
		
		box {
			<0, -10, -120>,
			<100, 120, 120>
			rotate -15*z
			translate <-25, 0, 0>
			} // close box
	
		} // close difference for cutting box 1
		
	difference { // cutting box 2 -- hole for aft end of glow tube
		box { 
			<50, 0, -110>,
			<455, 110, 100>
			} // close box
			
			box {
			<0, -10, -120>,
			<-100, 120, 120>
			rotate 15*z
			translate <75, 0, 0>
			} // close box
	
		box {
			<0, -10, -120>,
			<100, 120, 120>
			rotate -15*z
			translate <430, 0, 0>
			} // close box

		
		} // close cutting box 2

	object { // hole for RCS thruster on negative Z side.
		BasicPlate
		scale <10, 1, 1>
		rotate 90*x
		rotate -90*y
		translate <25, -50, -101>
		} // close BasicPlate
		
	object { // hole for RCS thruster on positive Z side
		BasicPlate 
		scale <10, 1, 1>
		rotate 90*x
		rotate 90*y
		translate <25, -50, 101>
		} // close BasicPlate
		
	object { // hole for RCS thruster on aft end
		BasicPlate
		scale <10, 1, 1>
		rotate 90*x
		rotate 180*y
		translate <501, -50, 0>
		} // close basic plate

#switch (Side) // this actually places the decals.
	#case (Starboard)
		#switch (Tier)
			#case (Upper)
				object {
					SFStripe
					scale <5, 1, 5>
					rotate -90*x
					rotate 180*z
					translate <-360, -40, -100.4>
					pigment { Yellow }
					}// close object SFStripe
	
				text {
					ttf
					"MicrogB.ttf"
					concat ( SquadDesig, " / ", BirdDesig )
					1
					0
					scale <30, 30, 1>
					translate <-285, -50, -100.9>
					texture { HazeGrey }
					} // close text for bird designation
				#break // end of Starboard, Upper
			#case (Lower)
				object {
					SFStripe
					scale <5, 1, 5>
					rotate 90*x
					rotate 180*z
					translate <-360, -40, 100.4>
					pigment { Yellow }
					}// close object SFStripe
	
				text {
					ttf
					"MicrogB.ttf"
					concat ( SquadDesig, " / ", BirdDesig )
					1
					0
					scale <30, 30, 1>
					rotate 180*y
					translate <-75, -50, 100.9>
					texture { HazeGrey }
					} // close text for bird designation
			#break // end of Starboard, Lower
		#end // switch (tier) for Starboard case
	#break // end case (Starboard)
	
	#case (Port)
		#switch (Tier)
			#case (Lower)
				object {
					SFStripe
					scale <5, 1, 5>
					rotate -90*x
					rotate 180*z
					translate <-360, -40, -100.4>
					pigment { Yellow }
					}// close object SFStripe
	
				text {
					ttf
					"MicrogB.ttf"
					concat ( SquadDesig, " / ", BirdDesig )
					1
					0
					scale <30, 30, 1>
					translate <-285, -50, -100.9>
					texture { HazeGrey }
					} // close text for bird designation
				#break // end of Port, Lower
			#case (Upper)
				object {
					SFStripe
					scale <5, 1, 5>
					rotate 90*x
					rotate 180*z
					translate <-360, -40, 100.4>
					pigment { Yellow }
					}// close object SFStripe
	
				text {
					ttf
					"MicrogB.ttf"
					concat ( SquadDesig, " / ", BirdDesig )
					1
					0
					scale <30, 30, 1>
					rotate 180*y
					translate <-75, -50, 100.9>
					texture { HazeGrey }
					} // close text for bird designation
						#break // end of Port, Upper
		#end // switch (tier) for Port case
	#break // end of case (Port)
#end // switch (Side)

	} // close difference Comp_nacelle
#end //macro Comp_Nacelle
	
#declare GlowTube = 
	cylinder { // this is the blue, glowing part on the nacelle.
		<-432, 0, 0>,
		<457, 0, 0>,
		90
		pigment {
			radial
			rotate -90*z
			frequency 26
			color_map {
				[0.0 color Blue]
				[0.5 color CadetBlue]
				[1.0 color Blue]
				} // close color_map
			scallop_wave
			} // close pigment
		finish { ambient 0.8 diffuse 0.6 }
		} // close cylinder 
	
#macro Nacelle ( Side, Tier )
	 merge{

		object { BussardCollector translate <-540, -27, 0>}
	
		light_source {
			<0, 0, 0>
			color Blue
			fade_distance 1000
			fade_power 2
			shadowless
			looks_like { GlowTube }
			} // close light_source 
	
		object {
			Comp_Nacelle ( Side, Tier )
			texture { SphinxSkin }
			} // close object 
		
		object {
			RCSBox (1)
			rotate 90*x
			rotate -90*y
			translate <25, -50, -100.1>
			} // close RCSBox
	
		object {
			RCSBox (1)
			rotate 90*x
			rotate 90*y
			translate <25, -50, 100.1>
			} // close RCSBox
		
		object {
			RCSBox (1)
			rotate 90*x
			rotate 180*y
			translate <500, -50, 0>
			} // close RCSBox

#switch  (Side)
	#case (Port)
		object { 
			NavLight ( Red )
			translate <25, 100, 0>
			}
	#break
		#case (Starboard)
			object {
				NavLight ( Green )
				translate <25, 100, 0>
				}
		#break
	#else
		object {
			NavLight ( Yellow )
			translate <25, 100, 0>
			}
		#end // switch side


	translate <225, 0, 0>

	} // close merge Nacelle
	
		#end // macro Nacelle
		
// ________________ End of Nacelle Definition __________________________

//_________________This section defines the torpedo and its tube ________
#declare Torp_Body = merge {		// this is both cylinders, the 'body' of the torpedo

	merge {		//this is the primary cylinder, which is actually a slightly tapered cone
	
		intersection {
			sphere {	// This is going to form the front end of a cylinder.
				<10, 0, 0>,	// centered 10 cm right, so the end cap will be at -15
				25			// same diameter as the total length of the torpedo
				pigment { color Black }
				} // close sphere

	
				cylinder {	// This won't actually show up, if I do this right.
					<-14, 0, 0>,	// Just inside the left sphere wall
					<-16, 0, 0>,	// just outside the left sphere wall
					5
					}	// close cylinder
				}	// close intersection

	

		cone { 
			<-15, 0, 0>,	// the same point as our recently composed cap. 8)
			5,
			<15, 0, 0>,		// leaving 10 cm for the end pipe.
			4
			pigment { color Black }
			}	// close cone
	
		
		} // close merge for cylinder 1.
	
	cylinder {
		<15, 0, 0>,	// congruent with the end of cylinder 1.
		<25, 0, 0>,	// right end of image
		3
		open
		texture { T_Chrome_1C }
		}	// close cylinder 
	
}	// close merge.  Body and 'tail pipe'.

// okay, so the outter shell of the warhead looks about right.  Now I'm gonna try to get the actual
// payload to look more or less right.


#declare Payload = union {
sphere {
	<0, 0, 0>,	// Simple enough... it has to go into the hole.
	4
	pigment {
		bozo
		turbulence .5
		frequency 3
		scale 2
		color_map {
			[0.00 color Red filter 1.0]
			[0.33 color Yellow filter 1.0]
			[0.66 color Blue filter 1.0]
			[1.00 color White filter 1.0]
			} // close color_map
		} // close pigment
		finish { 
			    reflection 0.1
    specular 0.8
    roughness 0.003
    phong 1
    phong_size 400
			ambient .8 
			diffuse .6 }
	} // close sphere

	light_source {
		<0, 0, 0>	// concentric with warhead
		color White
		fade_distance 1000
		fade_power 2
		shadowless
		}	// close light_source
 } // close union payload

// Okay, now we come to the hardest part... the warhead.  It's going to be a 10 cm sphere, cut by
// two quarter-cylinders, and hollowed in the center.  Then I'll drop a red sphere in the center,
// and make that 'glow'.  

#declare Warhead = merge {	// the total warhead, for which we've moved the Payload dec up.

		difference {	// 'quartered' holes in warhead

		sphere {
			<-20, 0, 0>,	// centered so that the right end slightly overlaps the end cap.  
			5				// 5cm radius gives 10 cm diameter
			texture { T_Chrome_3A }
			}	// close sphere
	
		difference {	// this is the cutting surface to open up the sphere
			cylinder {
				<-20, -7, 0>,	// slightly below the bottom of the r. 5cm sphere
				<-20, 7, 0>,	// slightly above the top of the r. 5cm sphere
				4.9				// we'll see how this looks
				texture { T_Chrome_3A }	// given a texture / color because it will effect the inside surfaces.
				} // close cylinder
	
			box {	// to cut the cylinder to the proper quarter-cylinder shape.
				<-20, -8, -5>,	// We're getting rid of the right half of the cylinder, so we put one corner
							// at the center of the cylinder's base.
				<-16, 8, 5>
				}	// Close box.  Box doesn't actually appear, needs no texture.
		
			box {
				<-19, -8, -.75>,	// trying something radical here...
				<-25, 8, .75>
				} // close box
		
			} // close difference for cutting object 1
		
	sphere {	// hollowing out the inside of the first warhead sphere
		<-20, 0, 0>,	// concentric with outer housing
		4				// 4cm radius gives 8cm diameter
		pigment { color Red }	// not really necisary, since I'm going to turn around and stick
								// something in there, but it'll be a nice intermediate step
		}	// close sphere

	} // close difference for holes
	
object { Payload translate -20*x }
	
} // close merge

	

#declare TriC_Torp = merge {	// and now, the whole jobbie...
	object { Torp_Body }
	object { Warhead }
	} // close merge
	

#declare SqCirc = lathe {
	5,
	<5.5, 0>, <5.5, 2>, <7.5, 2>, <7.5, 0>, <5.5, 0>
	rotate 90*x
	rotate -90*y
	} // close lathe

// and now, the entire enchelada!
	
#declare TorpTube = merge {

	object {
		SqCirc
		pigment { color Red }
		translate <-28, 0, 0>
		}
	
	 cylinder {
		<-28.3, 0, 0>,
		<25, 0, 0>,
		6.85
		open
		texture { T_Chrome_1C }
		}
		
	} // close merge TorpTube 
	

#declare TorpInTube = union {	
	object { TorpTube }
	object { TriC_Torp }
	} // close union
	
// ___________________ End of Torpedo Definition ___________________

//___________________This section defines the impulse engines ______

	
#local ImpVert = box { // If you look at the models of the various Enterprises, you'll see
						// black lines across the impulse exhaust.  This is a try at that.
	<0, -47, -0.25>
	<0.1, 47, 0.25>
	pigment { color Black }
	} // close box 

#local GlowBox = object { // provides (Surprise, surprise) the glow of the exhaust.
	BasicPlate
	scale <1, 2, 2.666>
	scale .9
	translate <1, 0, 0>
	pigment { color Red }
	finish { ambient 0.9 diffuse 0.7 }
	} // close GlowBox


#declare ImpEng = union {

	object {
		AdvPlate
		scale <1, 2, 2.666>
		} // close plate
	
	object { ImpVert translate 0.1*x }	
		box { <0.1, 46, 1>, <0.2, 41, -1> pigment { color Black } }
		cylinder { <0.1, 41, 0>, <0.2, 41, 0>, 1 pigment { color Black } }
		box { <0.1, -46, 1>, <0.2, -41, -1> pigment { color Black } }
		cylinder { <0.1, -41, 0>, <0.2, -41, 0>, 1 pigment { color Black } }

	object { ImpVert translate <0.1, 0, -5> }
		box { <0.1, 46, -4>, <0.2, 36, -6> pigment { color Black } }
		cylinder { <0.1, 36, -5>, <0.2, 36, -5>, 1 pigment { color Black } }
		box { <0.1, -46, -4>, <0.2, -36, -6> pigment { color Black } }
		cylinder { <0.1, -36, -5>, <0.2, -36, -5>, 1 pigment { color Black } }

	object { ImpVert translate <0.1, 0, -10> }
		box { <0.1, 46, -9>, <0.2, 26, -11> pigment { color Black } }
		cylinder { <0.1, 26, -10>, <0.2, 26, -10>, 1 pigment { color Black } }
		box { <0.1, -46, -9>, <0.2, -26, -11> pigment { color Black } }
		cylinder { <0.1, -26, -10>, <0.2, -26, -10>, 1 pigment { color Black } }

	object { ImpVert translate <0.1, 0, -15> }
		box { <0.1, 46, -14>, <0.2, 16, -16> pigment { color Black } }
		cylinder { <0.1, 16, -15>, <0.2, 16, -15>, 1 pigment { color Black } }
		box { <0.1, -46, -14>, <0.2, -16, -16> pigment { color Black } }
		cylinder { <0.1, -16, -15>, <0.2, -16, -15>, 1 pigment { color Black } }

object { ImpVert translate <0.1, 0, 5> }
		box { <0.1, 46, 4>, <0.2, 36, 6> pigment { color Black } }
		cylinder { <0.1, 36, 5>, <0.2, 36, 5>, 1 pigment { color Black } }
		box { <0.1, -46, 4>, <0.2, -36, 6> pigment { color Black } }
		cylinder { <0.1, -36, 5>, <0.2, -36, 5>, 1 pigment { color Black } }

	object { ImpVert translate <0.1, 0, 10> }
		box { <0.1, 46, 9>, <0.2, 26, 11> pigment { color Black } }
		cylinder { <0.1, 26, 10>, <0.2, 26, 10>, 1 pigment { color Black } }
		box { <0.1, -46, 9>, <0.2, -26, 11> pigment { color Black } }
		cylinder { <0.1, -26, 10>, <0.2, -26, 10>, 1 pigment { color Black } }

	object { ImpVert translate <0.1, 0, 15> }
		box { <0.1, 46, 14>, <0.2, 16, 16> pigment { color Black } }
		cylinder { <0.1, 16, 15>, <0.2, 16, 15>, 1 pigment { color Black } }
		box { <0.1, -46, 14>, <0.2, -16, 16> pigment { color Black } }
		cylinder { <0.1, -16, 15>, <0.2, -16, 15>, 1 pigment { color Black } }

	light_source {
		<1, 0, 0>
		color Red
		area_light <0, 100, 0>, <0, 0, 30>, 4, 3
		looks_like { GlowBox }
		fade_distance 1000
   fade_power 2
		} // close light_source

	
	} // close union ImpEng
	
//________________________ End of Impulse Engine Definition __________________________

//_________________ This defines the pylons, and joins the pylon to the nacelle _______


#local BasePylon = 
	merge {	// adding cylinders to front edge to make a 'wing' effect.

		box {	// this forms the basic form of the pylon.
			<-100, -350, -25>,	// 2 meters wide, 7 meters long, half a meter thick
			<100, 470, 25>
			}	// close box
	
		cylinder { // forward edge
			<-100, -350, 0>,
			<-100, 470, 0>,
			25
			}	// close cylinder
		
		} // close merge BasePylon
			
#local ClippedPylon = difference { // the clipped pylon -- clipped so odd bits don't hang out.
	object { BasePylon }

	box { // chops the top off the box, so it doesn't overrun the nacelle's boundries.
		<-250, 200, -30>,
		<150, 400, 30>
		rotate 45*z
		translate <77, 200, 0>
		} // close box
	
	box { // now the bottom
		<-150, -200, -30>,
		<250, -400, 30>
		rotate 45*z
		translate <-172, -105, 0>
		} // close box
	rotate -45*z
	} // close difference for clipped pylon	

#declare MountBox = difference {  // mounting box... adds eye candy at the nacelle joint.
									// I knew what I was doing when I wrote this code, but
									// can't decypher it now.  Render without the difference,
									// if you really want to know specificly what each piece 
									// does on its own.

	box { // joins the pylon to the nacelle.
		<-150, 150, -10>,
		<450, 250, 10>
		} // close box
	
	cylinder {	
		<-36, 165, -60>,
		<-36, 165, 60>,
		20
		} // close cylinder
		
	box { // likewise.
		<-36, 185, -30>,
		<-155, 140, 30>
		} // close box
		
	difference { 

		box {
			<-85, 250, -14>,
			<-155, 140, 14>
			} // close box
		
		cylinder {
			<-85, 250, -15>
			<-85, 250, 15>,
			65
			} // close cylinder

	} // close difference 
	
	cylinder {	
		<347, 143, -50>,
		<347, 143, 50>,
		40
		} // close cylinder
	

	box {
		<250, 170, -20>,
		<451, 143, 20>
		} // close box 

		
	difference { 
	
		box {
			<347, 250, -14>,
			<455, 140, 14>
			} // close box
		
		cylinder {
			<343, 270, -15>
			<343, 270, 15>,
			87
			scale 1.5*x
			translate -170*x
			} // close cylinder



	} // close difference 

} // close difference for mount box
	

#local PylonWTorp = union { // need to stick the torpedo launchers in here...
	
	difference { // cut holes in the pylon for the torp launcher.
		
		object { ClippedPylon } // The angled and chopped pylon.
		

		cylinder { // torp launcher 1
			<-98, 101, 0>,
			<15, 101, 0>,
			7.5
			} // close cylinder
		
		cylinder { // torp launcher 2
			<-116, 83, 0>,
			<15, 83, 0>,
			7.5
			} // close cylinder

		cylinder { // torp launcher 3
			<-134, 65, 0>,
			<15, 65, 0>,
			7.5
			} // close cylinder

		cylinder { // torp launcher 4
			<-152, 47, 0>,
			<15, 47, 0>,
			7.5
			} // close cylinder


		
		object { // this cuts a spot for the RCS thruster on the front of the pylon
			BasicPlate
			scale <10, 1, 1> // I want the "slot" for the thruster to be 10 cm-ish
			rotate -45*z	// to match the angle of the surface 
			translate <-35, 143, 0>	// so that we don't just create a slot in the middle of the wing
			} // close BasePlate


		object { // slot for RCS thruster aft.
			BasicPlate
			scale <10, 1, 1>
			rotate 135*z
			translate <287, 145, 0>
			} // close aft BasePlate

		object { // slot for Impulse Engine 
			BasicPlate
			scale <10, 2, 2.666>
			rotate 135*z
			translate <236, 85, 0>
			} // close aft ImpEng 1
			
	box { // on 25 Oct '00, I decided to substantially shorten the length of the pylon.		
			<-350, -10, -40>,
			<350, -300, 40>
			} // close box 
		
		} // close difference
		
	object { // launcher 1
		TorpInTube
		translate <-37, 101, 0>
		} // close TorpInTube

	object { // launcher 2
		TorpInTube
		translate <-56, 83, 0>
		} // close TorpInTube

	object { // launcher 3
		TorpInTube
		translate <-74, 65, 0>
		} // close TorpInTube

	object { // launcher 4
		TorpInTube
		translate <-92, 47, 0>
		} // close TorpInTube
		
// The RCS thrusters are all set to be 'on', for visual purposes.  This wouldn't ever really
// occur.  If you'd like to render your figher without them lit, change the (1)'s in the 
// code below to (0)'s.

	object {
		RCCBox (1)
		rotate -45*z
		translate <-34.4, 142.4, 0>
		} // close RCSBox fore
	
	
	object {
		RCSBox (1)
		rotate 135*z	// turn it around to face 'out', and align to surface
		translate <286.5, 145.5, 0>
		} // close RCSBox aft	
	

	object {
		ImpEng
		rotate 135*z
		translate <226.5, 85.5, 0>
		}	
		
	} // close union PylonWTorp


#declare Pylon = merge {
	object { PylonWTorp } 
	object { MountBox translate <30, 35, 0> 	} 
	texture { SphinxSkin }
	} // close merge

#macro Wing ( Side, Tier )
  merge {	// total wing
	object { Pylon }
	object { 
		Nacelle ( Side, Tier )
		translate <0, 400, 0>
		} // close nacelle
	translate 10*y // change made on 3 Nov '00, after I extended the nacelle 10 cm to give clear
					// firing path for the torpedos.
	} // close merge for total wing	
#end // macro wing

//_____________________ End of Pylon definition _________________________________

// ____________________ This section defines the canard _________________________
// This was actually the first part of the fighter I modeled, though originally, it
// was substantially longer.  It was also my first successful POV object.


#declare BaseCan = merge {	// basic wing
	
	prism {
		linear_spline	//I'm constructing this from straight lines
		linear_sweep	// size should be the same at top as at bottom.
		1, 25, 7
		<-300, 200>, <300, 200>, <300, 100>, <100, 0>, <-100, 0>, <-300, 100>, <-300, 200>

		} // close prism
		
	// Okay, so the basic wing shape works.  Now I'm going to put spheres and cylinders along all the 
	// edges, so that it looks like a wing, rather than just a slab o' metal.
	
	sphere	{ <-300, 12.5, 200>, 12.5 }
	sphere	{ <300, 12.5, 200>, 12.5 }
	sphere	{ <300, 12.5, 100>, 12.5 }
	sphere	{ <100, 12.5, 0>, 12.5 }
	sphere	{ <-100, 12.5, 0>, 12.5 }
	sphere	{ <-300, 12.5, 100>, 12.5 }
	cylinder	{ <-300, 12.5, 200>, <300, 12.5, 200>, 12.5 }
	cylinder	{ <300, 12.5, 200>, <300, 12.5, 100>, 12.5 }
	cylinder	{ <300, 12.5, 100>, <100, 12.5, 0>, 12.5 }
	cylinder	{ <100, 12.5, 0>, <-100, 12.5, 0>, 12.5 }
	cylinder	{ <-100, 12.5, 0>, <-300, 12.5, 100>, 12.5 }
	cylinder	{ <-300, 12.5, 100>, <-300, 12.5, 200>, 12.5 }
	
	texture { SphinxSkin }
	
	} // close merge

#declare RCS_Cyl = union {	// forms the RCS 'Cans' on either end of the canard.

	merge {

		cylinder {
			<-70, 0, 0>,
			<70, 0, 0>,
			20
			open
			texture { SphinxSkin }
			} // close cylinder
	
		torus	{	// I have to say that the torus would be a lot more useful if I could just create it in place.
			17.5,	// major radius is 17.5, giving a diameter of 35, which is 5 less than 40
			2.5	// minor radius is 2.5, gives a 5 cm ring.  35 + 5 = 40.
			rotate 90*z	// put it parallel to the y plane
			translate <-70, 0, 0>	// coincident to the front of the tube
			pigment { color Yellow }
			}	// close torus foreward
	
		torus	{	
			17.5,	// major radius is 17.5, giving a diameter of 30, which is ten less than 40
			2.5	// minor radius is 2.5, gives a 5 cm ring.  35 + 5 = 40.
			rotate 90*z	// put it parallel to the y plane
			translate <70, 0, 0>	// coincident to the front of the tube
			pigment { color Yellow }
			}	// close torus aft
		
		} // close merge
	
	disc {
		<0, 0, 0>,
		x,
		19.999 // slightly smaller, 'cause I don't want it causing a problem on the exterior 
		texture { SphinxSkin }
		} // close disc
	
	} // close union RCS_Cyl

#declare RCSDisc = // once again, our friend the thruster flame.
	disc {
		<0, 0, 0>,
		y,
		19.5
		pigment {
			cylindrical
			scale 20
			color_map {
				[0.0 color Blue filter 1]
				[1.0 color White filter 1]
				} // close color_map
			} // close pigment
		rotate 90*z
		}// close disc


#macro RCSCan ( Fore, Aft )
	union {
		object { RCS_Cyl } // placed here so that it always appears
		#if (Fore)	
			union { // combines the elements of the lighted thruster
				light_source { <67.5, 0, 0> White }
				object { RCSDisc translate 68*x }
				cone {
					<68, 0, 0>,
					17.5,
					<88, 0, 0>,
					0
					pigment { Clear }
					hollow
					interior {	
						media { 
							scattering { 1, rgbf 0.1 } // rgbf set low so as to get the 'glow' effect.
							} // close media
						} // close interior
					}// close cone
				} // close union 
			#end // if fore

		#if (Aft)
			union {
				light_source { <-67.5, 0, 0> White }
				object { RCSDisc translate -68*x }
				cone {
					<-68, 0, 0>,
					17.5,
					<-80, 0, 0>,
					0
					pigment { Clear }
					hollow
					interior {	
						media { 
							scattering { 1, rgbf 0.1 } // rgbf set low so as to get the 'glow' effect.
							} // close media
						} // close interior
					}// close cone

				} // close union 
			#end //if aft
		}// close union
	#end // macro RCSCan
			


#macro Canard (SF, SA, PF, PA) // four thrusters -- Port and Starboard, Fore and Aft.
	 union {
		object { RCSCan (SF, SA) translate <150, 12.5, 320> }
		object { RCSCan (PF, PA) translate <150, 12.5, -320> }
		object	{ BaseCan rotate 90*y }
		} // close union
	#end // Canard
	
//______________________________ End of Canard definition ________________

//__________________ This section defines the navigational deflector ______

	
#declare DefDis1 = disc { // this is the forward, blue/white disc.
	<0, 0, 0>,
	<1, 0, 0>
	30
	scale 2*z
	rotate 90*z
	texture { // t1
		pigment {
			radial
			color_map {
				[0.0 color Clear]
				[1.0 color Blue]

				} // close color_map
			triangle_wave
			frequency 20

			} // close pigment
			finish { ambient 0.8 diffuse 0.6 }
		} // close t1

	texture { //t2
		pigment {
			cylindrical
			color_map {
				[0.0 color Clear]
				[1.0 color Blue]

			} // close color_map
			scale <30, 1, 60>
			frequency 2

			}// close pigment
 	finish { ambient 0.8 diffuse 0.6 }
		} // close t2
rotate -90*z	

	} // close disc

#local DefDis2 = disc { // the inner, all-white one.
			<0.1, 0, 0>
			x
			30
			scale <1, 1, 2>
			pigment { color White }
			finish { ambient 0.8 diffuse 0.6 }
			} // close disc DefDis2


#declare Deflector = union {

	light_source {
	<0, 0, 0>
	color White
		looks_like { DefDis2 }
	}

light_source{
	<0, 0, 0>
	color Blue
	looks_like { DefDis1 }
	
// As I look at this, three months after starting, I think I'd do it differently if I did it
// today... probably union a lightsource behind two filtering elements.  But I'm not doing it
// today, so I'll leave this as-is.

	} // close light_source

} // close union Deflector

//___________________________________End of Deflector definition _________________

//________________ This section defines the cockpit area _________________________
// Alright, that's somehting of a misnomer.  What it actually defines is the canopy,
// and a hole, where the cockpit will someday be, when / if I get around to building
// it.  Probably not for this revision of the fighter.

// note that there's a duplicate of the definition of the cutting block and big block
// here, from the fusilage section.

#local cutting_block =  difference {
		
		
		box {
			<150, -214, -10>,
			<-550, 214, 400>
			pigment { Green transmit .75 }
			}
			
	
		prism {
			linear_sweep
			quadratic_spline 
			-215,
			215,
			11,
			<0, -30> // control point -- will not appear
			<-500, -110>
			<-250, 125>
			<150, 250>
			<250, 251>
			<250, 80>
			<250, -200>
			<-100, -200>
			<-300, -200>
			<-500,  -100>
			<-500, -110>
			//<-610, 120> // control point
			} // close prism
			
			rotate -90*x

		} // close difference	

#local BigBlock = 
	union {
	
		object {
			cutting_block
			
			} // close cutting_block
	
		
		
		object {
			cutting_block
			rotate 45*x
			translate <0, 10, 10>
			} // close cutting_block
		
		object {
			cutting_block
			rotate -45*x
			translate <0, 10, -10>
			} // close cutting_block		
		
		
	} // close difference
	
#declare Canopy = difference { // one last difference to hollow out the inside of the structure...

	union { // now I add glass to the cockpit...
		difference { // now to cut chunks out of it, so that the first shape forms the braces.
	
			difference { // the second pass at the cockpit.  This one is conformal.
				box {
					<-72, 50, -200>
					<-350, 150, 200>
					} // close box
				
				object { BigBlock translate -50*y scale .925 }
			
				} // close difference
		
			difference {
				box {
					<-77, 55, -201>
					<-340, 150, 201>
					}
			
				object { cutting_block translate -50*y scale .9 }
			
			} // close difference	
	
			difference { 
				box {
					<-72, 50, -200>
					<-350, 150, 200>
					} // close box
				
				object { BigBlock translate -50*y scale .925 }
			
			scale .89
			translate 20*y
			translate -12*x
	
			} // close difference

		texture { T_Silver_1A }

		} // close difference

	
		difference { 
			box {
				<-72, 50, -200>
				<-350, 150, 200>
				} // close box
			
			object { BigBlock translate -50*y scale .924 }
	
			texture { T_Glass3 }
			interior { I_Glass }
				
			} // close difference

	} // close union

	difference { 
			box {
				<-72, 50, -200>
				<-350, 150, 200>
				} // close box
			
			object { BigBlock translate -50*y scale .9 }
			
			texture { T_Glass3 }
			interior { I_Glass }
			
			} // close difference

	difference { // this is essentially the same as above, but I need to move it back and 
				 // down, to cut out the sides and bottom to create a true canopy.
			box {
				<-72, 50, -200>
				<-350, 150, 200>
				} // close box
			
			object { BigBlock translate -50*y scale .9 }
			
			texture { T_Glass3 }
			interior { I_Glass }
			translate <10, -10, 0>
			
			} // close difference

	scale 1.4*x
	translate 32*x
	
} // close difference for final cockpit object

#declare CockCutter = // cuts a space in the fusilage for the cockpit
difference { // making sure there's no join seam in the back.
	merge {
		intersection { // creating the slab for under the canopy
			difference { 
				box {
					<-72, 50, -200>
					<-350, 150, 200>
					} // close box
				
				object { BigBlock translate -50*y scale .9 }
				
				} // close difference

			box {
				<-72, 50, -200>
				<-350, 51, 200>
				}
			
			translate <95, -50, 0>
			scale <1.35, 100, 1>		
			rotate 180*x
			
			} // close intersection

		intersection { // creating the slab for aft of the canopy
			difference { 
				box {
					<-72, 50, -200>
					<-350, 150, 200>
					} // close box
				
				object { BigBlock translate -50*y scale .9 }
				
			} // close difference

			box {
				<-72, 50, -200>
				<-73, 150, 200>
				}
			
			translate <73, -50, 0>
			scale <10, 1, 1>		
			} // close intersection
		} // close merge
	
	box {
		<5, 151, -200>,
		<50, -151, 200>
		}
	
	} // close difference CockCutter
	
// ____________________________ End of Cockpit definition _______________________

// ___________________ This section defines the Impulse Crystal _________________
// This is essentially one big greeblie (A greeblie, if you don't know, is just a
// detail that's been added on to look neat, not to have any function).  It keeps
// the fusilage from being a flat, featureless box.

#include "stones.inc"

#local ImpCyl = difference { // d3
	difference { // d2
		difference { // d1
			cylinder { // c1
				<0, 0, 0>
				<0, 10, 0>
				100
				texture { T_Silver_1A }
				}
	
			cylinder { // c2
				<0, 8, 0>
				<0, 11, 0>
				95
				texture { SphinxSkin }
				} // close cyl
		
			} // close difference d1 to cut into the c1 cylinder.
		
		sphere {
			<0, 10, 0>
			10
			scale <9, 1, 9>
			texture { 
				pigment {
					radial
					frequency 64
					color_map {
						[0.0 color White]
						[1.0 color Black]
						} // close color_map
					triangle_wave
					} // pigment close
				}// texture  attribute
			} // close sphere
		
		} // close difference d2

	sphere {
		<0, -200, 0>
		210
		pigment { color Blue }
		} // close sphere

	} // close difference d3
	
#local ImpCry = intersection {
	cylinder { // c1
		<0, 0, 0>
		<0, 10, 0>
		100
		}
		
	sphere {
		<0, -200.1, 0>
		210
		pigment {
			agate
			scale 200
			turbulence 1
			lambda 2
			frequency 2
			color_map {
				[0.0 color CornflowerBlue]
				[0.5 color White]
				[1.0 color Blue]
				} // close color_map
			} // close pigment
		finish { ambient 0.8 diffuse 0.6 }
		} // close sphere

	} // close intersection 

#local AdvImpCyl = union {
	
	object { ImpCyl }

	light_source {
		<0, 0, 0>
		color DarkSlateBlue
		looks_like { ImpCry }
		} // close light_source

	} // close union AdvImpCyl
	
#local BDeck = difference { 
	
	prism { 
		linear_sweep
		linear_spline
		0,
		8,
		7,
		<0, 101>, <150, 120>, <170, 25>, <170, -25>, <150, -120>, <0, -101>, <0, 101>
		
		} // close prism 
		
	cylinder { // c1
		<0, -1, 0>
		<0, 10, 0>
		100
		}
	
	} // close difference
	
#local ADeck = difference {
	object { BDeck texture { T_Silver_1A }}
	object { 
		BDeck
		scale .9
		translate 3*y
		texture { SphinxSkin }
		} // close cutting deck
	} // close difference
	
#local GreebCol = union {

	box {
		<-5, 9, -12.5>,
		<5, 3, 12.5>
		}
		
	box {
		<-5, 9, 17.5>,
		<5, 3, 42.5>
		}
	
	box { 
		<-5, 9, -17.5>,
		<5, 3, -42.5>
		}
	

	texture { T_Copper_1C }
	
} // close union GreebCol

#local GreebStickA = merge {
	
	object { 
		GreebCol
		translate 45*z
		} // close GreebCol
		
	cylinder {
		<0, 3, -15>,
		<0, 3, 85>,
		3
		} // close cylinder
		
	texture { T_Copper_1C }

	} // close GreebStick
	
#local GreebGroupA = difference {

union {

object { 
	GreebStickA
		rotate -12*y
	translate <143, 0, 15>
	} // close GreebCol

object { 
	GreebStickA
		rotate -12*y
	translate <128, 0, 10>
	} // close GreebCol
	
object { 
	GreebStickA
		rotate -12*y
	translate <113, 0, 5>
	} // close GreebCol

} // close union 

box {
	<100, -10, -50>,
	<170, 40, 0>
	} // close box

} // close difference

#local GreebStickB = merge {
	
	object { 
		GreebCol
		translate -45*z
		} // close GreebCol
		
	cylinder {
		<0, 3, 15>,
		<0, 3, -85>,
		3
		} // close cylinder
		
	texture { T_Copper_1C }

	} // close GreebStick


#local GreebGroupB = difference {

union {

object { 
	GreebStickB
		rotate 12*y
	translate <143, 0, -15>
	} // close GreebCol

object { 
	GreebStickB
		rotate 12*y
	translate <128, 0, -10>
	} // close GreebCol
	
object { 
	GreebStickB
		rotate 12*y
	translate <113, 0, -5>
	} // close GreebCol

} // close union 

box {
	<100, -10, 50>,
	<170, 40, 0>
	} // close box

} // close difference

#local Greeblies = union {

object { GreebGroupA }
object { GreebGroupB }

cylinder {
	<99.9, 3, 0>,
	<159, 3, 0>
	3
	texture { T_Copper_1C }
	}


} // close union Greeblies	
	
#declare ImpDeck = union {

object { AdvImpCyl }
object { ADeck }
object { Greeblies }

} // close union ImpDeck

// _____________________________ End of Impulse Crystal definition ________________

// __________________________ This section defines the phaser cannons _____________
// Well, I forgot to stick 'em in the prototype, so I'm goofing around now
// with phaser cannons for the VoidSphinx.  I'm kind of sort of generally
// basing them on the gattling cannon for the F16 Fighting Falcon.
// they extend along the X axis, roughly +/- 15, with an interior diameter of 15, and an
// exterior diameter of 18.


#declare PhaserCannon = difference {
	
	lathe {
		cubic_spline
		6 // number of points
		<7.5, 0> // control point, not visible
		<7.5, 50> // match point
		<8.5, 48>
		<7.5, 0>
		<7.5, 50> // match point
		<8.5, 48> // control point
		sturm
		} // close lathe
	
	cylinder {
		<0, 53, 0>
		<0, 51, 0>
		18
		}
	
	cylinder {
		<0, 49, 0>
		<0, 47, 0>
		18
		}
	
	texture { SphinxSkin }
	rotate 90*z
	translate 25*x	
	} // close difference
	
// ______________________ End of Phaser Cannon definition __________________________

//_______________________ This section defines the fusilage ________________________
// note that there is a duplicate instance of the cutting block up in the section for
// the cockpit.


// Okay, the idea here is that the long axis of the fusilage is along the X axis, just as in the
// other parts of this project.  Nose is -x, whole thing is centered around the origin.
// I figure the cockpit is actually the smallest space requirement... a meter and a half cube,
// at least as far as pilot space goes.  So I figure four meters wide, three deep, and ten long. 
// there are a couple of different ideas I have on how to form it, so away we go...

// If you're using an image that isn't the expected format defined at the top, this is 
// the section where you change that.  Look down a few lines.

#local Empty = pigment { color Clear }


#local DangerPort =
	pigment {
		image_map {
			png 
			"danger.png"
			interpolate 2
			once
			} // close image_map
	scale 15
		translate <-135, 25, 0>
} // close Danger

#local DangerSTBD =
	pigment {
		image_map {
			png 
			"danger.png"
			interpolate 2
			once
			} // close image_map
	scale 15
	rotate 180*y
		translate <-120, 25, 0>
	} // close pigment DangerPort

#local Danger_1 = 
 	pigment { 
		radial
		rotate -90*z
		pigment_map {
			[ 0.5 DangerPort ]
			[ 0.5 DangerSTBD ]
			} 
		}

#local Danger = 
	pigment {
		cylindrical
		rotate 90*z
		pigment_map {
			[0.41 Danger_1 ]
			[0.41 Empty ] 
			} 
		}


                                
#local HeroArt = 
	pigment { 
		image_map { 	
			png HeroPatch // if you're using something besides a png, change the type here.
			//gif HeroPatch // uncomment this line for a gif, comment out above.
  			map_type 0 
			once 
			} // image_map
			scale 250
			translate <-60, -120, 0>
			}

#local SquadArt = 
	pigment { 
		image_map { 
			gif SquadPatch // If you're using something besides a gif, change the type here.
			//png SquadPatch // for a png, uncomment here, comment out above.
			filter 0, 1.0
			map_type 0 
			once 
			} // image_map
		scale 250
		translate <-140, -120, 0>
        }
                                                              
#local NoseArt1 = 
 	pigment { 
		radial
		rotate -90*z
		pigment_map {
			[ 0.5 HeroArt ]
			[ 0.5 SquadArt rotate 180*y ]
			} 
		}

#local NoseArt2 = 
	pigment {
		cylindrical
		rotate 90*z
		pigment_map {
			[0.41 NoseArt1 ]
			[0.41 Empty ] 
			} 
		}
                                
			
#local bfb = // basic fighter box
	merge {	
		box {
			<-510, -150, -150>,
			<500, 150, 150>
			} // close box		

		difference {
			cylinder {
				<-510, 0, 0>
				<500, 0, 0>
				210
				} // close cylinder
	
			box {
				<-501, -210, -149>,
				<501, 210, 149>
				} // close box
		
			} // close difference
		} // close merge	

#local cutting_block =  difference {
		
		
		box {
			<150, -214, -10>,
			<-550, 214, 400>
			}
			
	
		prism {
			linear_sweep
			quadratic_spline 
			-215,
			215,
			11,
			<0, -30> // control point -- will not appear
			<-500, -110>
			<-250, 125>
			<150, 250>
			<250, 251>
			<250, 80>
			<250, -200>
			<-100, -200>
			<-300, -200>
			<-500,  -100>
			<-500, -110>
			} // close prism
			
			rotate -90*x

		} // close difference	

#local BigBlock = 
	union {
	
		object {
			cutting_block
			
			} // close cutting_block
	
		
		
		object {
			cutting_block
			rotate 45*x
			translate <0, 10, 10>
			} // close cutting_block
		
		object {
			cutting_block
			rotate -45*x
			translate <0, 10, -10>
			} // close cutting_block		
		
		
	} // close difference
	


#local afb = // advanced fighter body
	difference {
	
		object { bfb }
		object { BigBlock translate <0, -50, 0> }
		object {
			BigBlock
			rotate 180*x
			translate <0, -60, 0>
			scale <1, .5, 1>
		} // close big block
		object { CockCutter translate <-70, 50.5, 0> }

		texture { SphinxSkin }
		texture { NoseArt2 }
			texture { Danger }
			//texture { CanopyCont }
	} // close difference afb


#local FSB1	= difference { // cut a hole in the fusilage for the cockpit.
	object { afb }//scale 1.5*x translate 200*x}
		
	box {
		<-115, 50, -300>
		<-500, 200, 300>
		texture { SphinxSkin }
		} // close box
		scale <1.5, 1, 1>
		translate <100, 0, 0>
			
		} // close difference d1
	
#local FSB2 = difference { //d2 when I scaled it to get the nose to look right, I end up with a tube that's
						// way too long.  So I'm cutting it back at the far end.	
				
	object { FSB1 }
				
	box { 
		<1000, -350, -350>
		<700, 350, 350>
		} // this is the box that actually chops the aft end.
			
cylinder { // port phaser cannon
	<-416, 0, -75>,
	<-350, 0, -75>,
	7.5
	pigment { Red }
	}

cylinder { // Starboard phaser cannon
	<-416, 0, 75>,
	<-350, 0, 75>,
	7.5
	pigment { Red }
	}

		} //close difference FSB2 for cutting cockpit and aft end space

#declare RECap = 	difference { // creates the rounded end cap
	intersection {
		object { bfb }
		sphere {
			<-500, 0, 0>,
			300
			}
	
		} // close intersection

	box {
		<-300, -301, -301>,
		<-550, 301, 301> 
		} // close box
			
	translate 1000*x
	texture { SphinxSkin }
	
	} // close difference RECap

#declare FSB3 = 	difference { // d3, cutting the space for the deflector
		
		object { FSB2 }	
		cylinder { // cutting a hole for the navigational deflector
			<-500, -89, 0>,
			<-85, -89, 0>,
			30
			scale <1, 1, 2> 
			texture { SphinxSkin }

			} // close cylinder
	
		texture { SphinxSkin }
	} // close difference d3 fusilage



#macro Fusilage ( CSF, CSA, CPF, CPA ) // thruster locations on the canard

 union {

	object { FSB3 }
			
object { 
		Canard (CSF, CSA, CPF, CPA)
		translate <-515, -60, 0>
	 	} // close cannard

	object { RECap }

object {
	PhaserCannon
	translate <-368, 0, -74.9>
}	

object {
	PhaserCannon
	translate <-368, 0, 74.9>
}	


	object {
		Deflector 
		translate <-99, -89, 0>
		}		
	
	object { 
		ImpDeck
		translate <375, 150, 0>
		}
	
	object { Canopy }	

	} // close union fusilage.
#end // macro Fusilage

// __________________________ End of Fusilage Definition ____________________________

// __________________________ This section puts everything together _________________

#declare VoidSphinx = union {

// If you don't want the RCS thrusters lit, change the 1's to 0's below. also look at the pylon 
// and nacelle sections.

object { Fusilage ( 1, 1, 1, 1 ) }

object {
	Wing (Starboard, Upper )
	rotate 77.5*x
	translate <525, 123, 150>
	}

object {
	Wing ( Starboard, Lower )
	rotate 102.5*x
	translate <525, -123, 150>
	}
	 
object {
	Wing ( Port, Upper )
	rotate -77.5*x
	translate <525, 123, -150>
	}
	
object {
	Wing (Port, Lower )
	rotate -102.5*x
	translate <525, -123, -150>
	}
 
 } // close union VoidSphinx
