// Persistence of Vision Ray Tracer Scene Description File
// File: lightning.inc
// Vers: 3.1g
// Desc: Lightning Include File
// Date: Aug/2000
// Auth: Tom Melly, http://www.tomandlu.co.uk, tom@tomandlu.co.uk
//
// ----------------------------------------

/* Instructions for Use

1 Place the file "lightning.inc" in an appropriate directory
  (either in a directory included in your library path or the
  directory of the file you wish to use the lightning in)  

2 Call the include file with the line "#include "lightning.inc"

3 Set or change any parameters you wish to alter from the defaults (see below for list of parameters and
  their meaning)
  
4 Call the macro, Make_Bolt, with the following syntax "Make_Bolt()"

5 This will create a lightning bolt object called "Lightning_Bolt.
  To use, add an object wrapper around it -  "object{Lightning_Bolt}"
  
  See the example file, lightning_demo.pov, for further examples.
  
  NOTE ON PARAMETERS
  
  All parameters have a default value.
  
  If you wish to generate several bolts with similiar features, it is only necessery to re-define those
  parameters you wish to change (such as LSeed). Any parameters you have set for previous bolts will be
  carried over to the new bolts.
  
  If you wish to reset all parameters to the original defaults, you can call the macro "LReset", which will
  undefine all parameters, forcing "Make_Bolt" to reset them to the defaults.

*/

#macro Make_Bolt()

  #ifndef(doMerge)      #declare doMerge =      false;  #end // [true/false]  creates bolt as csg merge, otherwise union (merge required for smooth media interior)
  #ifndef(doLights)     #declare doLights =     false;  #end // [true/false]  makes bolt illuminate
  #ifndef(lightColour)  #declare lightColour =  <1,1,1>;#end // [color]       if doLights, this is colour of light
  #ifndef(LArraySize)   #declare LArraySize =   1000;   #end // [1 - n]       increase this if array bounds reached
  #ifndef(LSeed)        #declare LSeed =        6767;   #end // [n]           seed for random numbers used in bolt
  #ifndef(LWidth)       #declare LWidth =       0.05;   #end // [>0]          width of bolt threads
  #ifndef(LHeight)      #declare LHeight =      10;     #end // [>0]          height of bolt
  #ifndef(LSpread)      #declare LSpread =      0.2;    #end // [0 - n}       spread of bolt (i.e. does bolt go straight down or wander)
  #ifndef(LSpike)       #declare LSpike =       0.5;    #end // [0 - n]       spikiness of bolt
  #ifndef(LBranch)      #declare LBranch =      0.02;   #end // [0 - n]       low for no branches, high for many branches
  #ifndef(LCuttoff)     #declare LCutoff =      0.0;    #end // [0 - LHeight] point at which branching can start
  
  #declare posArray = array[LArraySize]
  #declare levelArray = array[LArraySize]
  #declare checkCount = -1;
  #declare highCount = 0;
  #declare arrayCount = 0;
  #declare posArray[arrayCount] = <0,0,0>;
  #declare levelArray[arrayCount] = 1;
  #declare lightCount = 0;
  #declare LRand = seed(LSeed); 

  #declare Lightning_Bolt =

  #if (doMerge) merge{ #else union{ #end
    #if (doLights)
      light_source{0, lightColour fade_distance LHeight/5 fade_power 1}
    #end
    #while (checkCount < highCount)
      #declare checkCount = checkCount + 1;  
      #declare segPosA = posArray[arrayCount];
      #declare currentLevel = levelArray[arrayCount];
      #declare arrayCount = arrayCount + 1;
      #declare xPlus = (rand(LRand)-0.5)*LSpread;
      #declare zPlus = (rand(LRand)-0.5)*LSpread;
      #while(segPosA.y < LHeight)
        #declare xShift = segPosA.x + xPlus + (rand(LRand)-0.5)*LSpike;
        #declare yShift = segPosA.y + (rand(LRand)/5);
        #declare zShift = segPosA.z + zPlus + (rand(LRand)-0.5)*LSpike;
        #declare atBottom = false;
        #if (yShift > LHeight)
          #declare yShift = LHeight;
          #declare atBottom = true;
        #end
        #declare segPosB = <xShift, yShift, zShift>;
        cylinder{segPosA, segPosB, LWidth}
        sphere{segPosB, LWidth + 0.0001}
        #if (atBottom & doLights)
          light_source{segPosB, lightColour fade_distance LHeight/5 fade_power 1}
        #end
        #if ( (rand(LRand) < (LBranch/(pow(currentLevel,2))) ) & (segPosA.y > LCutoff) )
          #declare highCount = highCount + 1;
          #declare posArray[highCount] = segPosA;
          #declare levelArray[highCount] = currentLevel + 1;
          #if (doLights)
            light_source{segPosA, lightColour fade_distance LHeight/5 fade_power 1}
          #end
        #end
        #declare segPosA = segPosB;
      #end
    #end
    rotate x*180
    no_shadow
  }
#end

#macro LReset()
  #undef doMerge
  #undef doLights
  #undef lightColour
  #undef LArraySize
  #undef LSeed
  #undef LWidth
  #undef LHeight
  #undef LSpread
  #undef LSpike
  #undef LBranch
  #undef LCutoff
#end