// Persistence of Vision Ray Tracer Include File 
// Part of the IRTC GIP object archives 
// Version: 1.0

// macro: IR_RoadBridge
//   (RBBridgeLength, RBRailingLength, RBWidth, SidewalkLevel, WaterLevel)
// Object Description: A stone bridge for leading a road over
//                     a 4m wide stream. 
// Author: Ib Rasmussen
// Email: ir@ibras.dk
// Web page: http://www.ibras.dk/
// Date (yyyy-mm-dd): 2000.01.25

// Origin: Center of bridge. 
// Orientation: Road along x-axis, stream along z-axis.
// Dimensions: 1 unit = 1 cm
// Size: x-axis: -RBBridgeLength/2 to RBBridgeLength/2
//       y-axis: 366
//       z-axis: -RBWidth/2 to RBWidth/2

// Includes required: colors.inc, stones1.inc, infr011n.ttf 
// Example: object { IR_RoadBridge(1600, 640, 1000, 140, -50) }
// Variables: 

// Comments: The length of the bridge side and the bride railing
//           are controlled individually by . 
//           A road and sidewalk surface has to be added by the
//           calling scene file, as does a water surface.
//           The bridge is designed for a difference between 
//           SidewalkLevel and WaterLevel about 200-250 cm.
//           The bridge side is designed to disappear into the 
//           banks at each side of the stream or canal.    
//           The top of the wall is translated to SidewalkLevel.
//           WaterLevel is used to place a band of algae just above
//           the water line.
//           
//           REMEMBER to declare the variable FontPath to point at the directory
//           containing the TrueType font INFR011N.TTF before including this file.
//           Example: #declare FontPath="c:\\winnt\\fonts\\"

// Scaling ///////////////////////////////////////////////////// 
// 1 cm = 1 baseUnit 
#ifndef(baseUnit) #local baseUnit = 1; #end 
/////// DO NOT CHANGE ANYTHING BELOW ///////////////////////////// 
// Metric 
#ifndef (m) /***meter****/ #local m  = 100 *baseUnit; #end 
#ifndef(cm) /*centimeter*/ #local cm = 1.0 *baseUnit; #end 
#ifndef(mm) /*millimeter*/ #local mm = 0.1 *baseUnit; #end 
// English 
#ifndef(in) /*inch*/ #local in = 2.54 *baseUnit; #end 
#ifndef(ft) /*foot*/ #local ft = 12*in*baseUnit; #end 
#ifndef(yd) /*yard*/ #local yd = 3*ft *baseUnit; #end 
////////////////////////////////////////////////////////////////// 

global_settings { assumed_gamma 1.0 } 

/*********************** Control Variables ***************************/

#ifndef (ShowDetail)
   #local ShowDetail=1;
#end   

// Local

/************************* Declarations ******************************/
 
/*-------------------------- Textures -------------------------------*/

#declare T_RBRailing=texture {
#if(ShowDetail = 1)
   T_Stone10      
#else   
   pigment { Gray30 quick_color Red }
#end   
}

#declare T_RBBridge=texture {
#if(ShowDetail = 1)
   T_Stone10      
#else   
   pigment { Gray30 quick_color Yellow }
#end   
}

#declare T_RBBridgeMortar=texture {
#if(ShowDetail=1)
   T_Stone20
#else
   pigment { Gray30 quick_color Magenta }
#end   
}

#declare T_RBInside=texture {                   
   pigment { rgb <0.7, 0.7, 0.6> quick_color Blue }
}               

#declare T_RBAlgae=
texture {   
   pigment { 
      gradient y
      color_map{
         [0.00, 0.090  color rgbf <0.5, 0.7, 0.5, 0.8>
	               color rgbf <1.0, 1.0, 1.0, 1.0> ]
         [0.08, 1.001  color rgbf <1, 1, 1, 1>
	               color rgbf <1, 1, 1, 1> ]
               }//end map               
      quick_color  Aquamarine  
           }//end pigmnt
   scale <1, 450, 1>
   translate <0, -450, 0>
        } // end texture                        

/*-------------------------- Objects --------------------------------*/

/* ---------------- Railing -------------------*/


#declare RBRailingTop=union{
   difference {
      box { <0, 83, 0>, <48, 99, -3> }
      cylinder { <-1, 0, 0>, <50, 0, 0>, 7 scale <1, 1, 3/7> translate <0, 84, -3> }
   }
   cylinder { <0, 95, -3>, <48, 95, -3>, 4 }
   cylinder { <0, 82.5, 0>, <48, 82.5, 0>, 1.5 }
   texture { T_RBRailing }
}   
      
#declare RBRailingMidSemiCirc=intersection{
   cylinder { <0, 0, -11>, <0, 0, 11>, 16 }
   box { <0, -16, -12>, <16, 16, 12> }
}
   
#declare RBRailingMidQuaterTorus=intersection {
   torus { 13, 2 rotate <90, 0, 0> }
   box { <0, -16, -3>, <16, 0, 0> }
}   
                             
#declare RBRailingMidDecoPart=union {
   object { RBRailingMidQuaterTorus translate <0, 38, -11> }
   cylinder { <13, 38, -11>, <13, 64, -11>, 2 }
   object { RBRailingMidQuaterTorus scale <1, -1, 1> translate <0, 64, -11> }
}   
 
#declare RBRailingMidDeco=union{
   object { RBRailingMidDecoPart }
   object { RBRailingMidDecoPart scale <-1, 1, 1> translate <48, 0, 0> }
   torus { 9, 2 rotate <90, 0, 0> translate <24, 51, -11> }
}   
                                  
#declare RBRailingMidCutout=union {
   cylinder { <0, 38, -20>, <0, 38, 20>, 10 }
   box { <-10, 38, -20>, <10, 64, 20> }
   cylinder { <0, 64, -20>, <0, 64, 20>, 10 }
}                                       

#declare RBRailingMid=union {
   difference {
      union {
         box { <0, 0, -10>, <48, 100, 10> }
         object { RBRailingMidSemiCirc translate <0, 38, 0> }
         object { RBRailingMidSemiCirc translate <0, 64, 0> }
         object { RBRailingMidSemiCirc scale <-1, 1, 1> translate <48, 38, 0> }
         object { RBRailingMidSemiCirc scale <-1, 1, 1> translate <48, 64, 0> }
         box { <0, 38, -11>, <16, 64, 11> }
         box { <32, 38, -11>, <48, 64, 11> }
         cylinder { <24, 51, -11>, <24, 51, 11>, 12 }
      }   
      object { RBRailingMidCutout translate < 0, 0, 0> }
      object { RBRailingMidCutout translate <48, 0, 0> }    
      sphere { <24, 51, -11>, 6 }  
      sphere { <24, 51,  11>, 6 }
   }
   object { RBRailingMidDeco }
   object { RBRailingMidDeco scale <1, 1, -1> }
      
   texture { T_RBRailing }
}      
   
#declare RBRailingBottom=union{
   box { <0,  0, -9>, <48,  9, 0> }
   box { <0,  9, -7>, <48, 11, 0> }
   box { <0, 11, -4>, <48, 17, 0> }
   cylinder { <0, 0, 0>, <48, 0, 0>, 6 scale <1, 1, 2/6> translate <0, 11, -4> }
   difference {
      box { <0, 17, -3>, <48, 21, 0> }
      cylinder { <-1, 21, -3>, <50, 21, -3>, 2 }
   }  
   box { <0, 21, -1>, <48, 22, 0> }
  
   texture { T_RBRailing }
}      
     
#declare RBRailingSection=union {          
   object { RBRailingTop translate <0, 0, -10> }
   object { RBRailingTop scale <1, 1, -1> translate <0, 0, 10> }
   object { RBRailingMid }              
   object { RBRailingBottom translate <0, 0, -10> }
   object { RBRailingBottom scale <1, 1, -1> translate <0, 0, 10> }
   
}


#macro RBRailingEnd(RBRailingEndLen)
union{
   difference {
      box { <0, 0, -20>, <RBRailingEndLen, 100, 20> }
      box { <RBRailingEndLen+1, 31, -11>, <RBRailingEndLen-2, 74, 11> }
      box { <RBRailingEndLen+1, 33,  -9>, <RBRailingEndLen-4, 72,  9> }
      box { <-1, 31, -11>, <2, 74, 11> }
      box { <-1, 33,  -9>, <4, 72,  9> }
      box { < 9, 31, -21>, <RBRailingEndLen-9,  74, -18> }
      box { <11, 33, -21>, <RBRailingEndLen-11, 72, -16> }
      box { < 9, 31,  21>, <RBRailingEndLen-9,  74,  18> }
      box { <11, 33,  21>, <RBRailingEndLen-11, 72,  16> }
   }
   difference {   
      union{
         object { RBRailingTop scale <60, 1, 1> translate <-30, 0, 0> }
         object { RBRailingBottom scale <60, 1, 1> translate <-30, 0, 0> }
      }   
      plane { x, 0 rotate <0, 45, 0> translate <-20, 0, 0> }
      plane {-x, 0 rotate <0, -45, 0> translate <20, 0, 0> }
      rotate <0, 90, 0>
   }   
   difference {   
      union{
         object { RBRailingTop scale <60, 1, 1> translate <-30, 0, 0> }
         object { RBRailingBottom scale <60, 1, 1> translate <-30, 0, 0> }
      }   
      plane { x, 0 rotate <0, 45, 0> translate <-20, 0, 0> }
      plane {-x, 0 rotate <0, -45, 0> translate <20, 0, 0> }
      rotate <0, -90, 0> translate <RBRailingEndLen, 0, 0> 
   }
   difference {
      union{   
         object { RBRailingTop scale <RBRailingEndLen/48+20, 1, 1> translate <-10, 0, 0> }
         object { RBRailingBottom scale <RBRailingEndLen/48+20, 1, 1> translate <-10, 0, 0> }
      }
      plane { x, 0 rotate <0, 45, 0> }
      plane {-x, 0 rotate <0, -45, 0> translate <RBRailingEndLen, 0, 0> }
      translate <0, 0, -20>
   }
   difference {
      union{   
         object { RBRailingTop scale <RBRailingEndLen/48+20, 1, -1> translate <-10, 0, 0> }
         object { RBRailingBottom scale <RBRailingEndLen/48+20, 1, -1> translate <-10, 0, 0> }
      }
      plane { x, 0 rotate <0, -45, 0> }
      plane {-x, 0 rotate <0, 45, 0> translate <RBRailingEndLen, 0, 0> }
      translate <0, 0, 20>
   }
   texture { T_RBRailing }
}
#end

#macro RBRailing(RBRailingLength)      
#declare RBRailSecLen = RBRailingLength - 120;
#declare RBRailSecLen = int(RBRailSecLen/48) * 48;
#local RBRailingEndLen = (RBRailingLength - RBRailSecLen) / 2 - 9;  
union{                  
  object { RBRailingEnd(RBRailingEndLen) scale <-1, 1, 1> translate <-RBRailSecLen/2, 0, 0> }
  #local xpos = -RBRailSecLen/2;
  #while (xpos < RBRailSecLen/2)
     object { RBRailingSection translate <xpos, 0, 0> }
     #local xpos = xpos + 48;
  #end   
  object { RBRailingEnd(RBRailingEndLen) translate <xpos, 0, 0> }
}
#end
                     
/* ---------------- Bridge -------------------*/

#declare BridgeBase=difference {
   box { <0, -20, -5>, <400, 300, 0> }
   cylinder { <0, 0, -10>, <0, 0, 10>, 200 scale <1.5, 1, 1> }  
   box { <0, -30, -10>, <300, 1, 10> }
   cylinder { <0, 0, -10>, <0, 0, -4>, 210 scale <310/210, 1, 1> }  
   box { <0, -30, -10>, <305, 1, -4> }
}
         
#macro RBBridgeStone(BSWidth, BSHeight)         
difference {
   box { <0, 0, -5>, <BSWidth, BSHeight, 0> }
   plane { x, 0 rotate <0, -45, 0> translate <0,  0, -2> }
   plane {-x, 0 rotate <0,  45, 0> translate <BSWidth,  0, -2> }
   plane { y, 0 rotate < 45, 0, 0> translate <  0,  0, -2> }
   plane {-y, 0 rotate <-45, 0, 0> translate <  0, BSHeight, -2> }
   texture { T_RBBridge }
}
#end   
   
#macro RBBridgeFaceHalf(RBBridgeLength, WaterLevel)
union{
   difference {
      object { BridgeBase }
      plane { x, 0 rotate <0, 0,  2.81035> translate <-22.0305, 199.460, 0> }
      plane { x, 0 rotate <0, -45,  2.81035> translate <-22.0305, 199.460, -2> }
      plane {-x, 0 rotate <0, 0, -2.81035> translate < 22.0305, 199.460, 0> }
      plane {-x, 0 rotate <0, 45, -2.81035> translate < 22.0305, 199.460, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -2.81035> translate < 22.0305, 199.460, 0> }
      plane { x, 0 rotate <0, -45, -2.81035> translate < 22.0305, 199.460, -2> }
      plane {-x, 0 rotate <0, 0,  -8.53251> translate < 65.8667, 195.120, 0> }
      plane {-x, 0 rotate <0, 45,  -8.53251> translate < 65.8667, 195.120, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0,  -8.53251> translate < 65.8667, 195.120, 0> }
      plane { x, 0 rotate <0, -45,  -8.53251> translate < 65.8667, 195.120, -2> }
      plane {-x, 0 rotate <0, 0, -14.57814> translate <109.0302, 186.324, 0> }
      plane {-x, 0 rotate <0, 45, -14.57814> translate <109.0302, 186.324, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -14.57814> translate <109.0302, 186.324, 0> }
      plane { x, 0 rotate <0, -45, -14.57814> translate <109.0302, 186.324, -2> }
      plane {-x, 0 rotate <0, 0, -21.21553> translate <150.9574, 172.835, 0> }
      plane {-x, 0 rotate <0, 45, -21.21553> translate <150.9574, 172.835, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -21.21553> translate <150.9574, 172.835, 0> }
      plane { x, 0 rotate <0, -45, -21.21553> translate <150.9574, 172.835, -2> }
      plane {-x, 0 rotate <0, 0, -28.80872> translate <190.9052, 154.280, 0> }
      plane {-x, 0 rotate <0, 45, -28.80872> translate <190.9052, 154.280, -2> }
      plane {-x, 0 rotate <0, 45, 0> translate <246.2965, 0, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -28.80872> translate <190.9052, 154.280, 0> }
      plane { x, 0 rotate <0, -45, -28.80872> translate <190.9052, 154.280, -2> }
      plane {-x, 0 rotate <0, 0, -37.86324> translate <227.7362, 130.190, 0> }
      plane {-x, 0 rotate <0, 45, -37.86324> translate <227.7362, 130.190, -2> }
      plane {-x, 0 rotate <0, 45, 0> translate <282.0099, 0, -2> }     
      plane {-y, 0 rotate <-45, 0, 0> translate <0, 255, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -37.86324> translate <227.7362, 130.190, 0> }
      plane { x, 0 rotate <0, -45, -37.86324> translate <227.7362, 130.190, -2> }
      plane {-x, 0 rotate <0, 0, -49.09148> translate <259.7730, 100.040, 0> }
      plane {-x, 0 rotate <0, 45, -49.09148> translate <259.7730, 100.040, -2> }
      plane {-x, 0 rotate <0, 45, 0> translate <311.6607, 0, -2> }
      plane {-y, 0 rotate <-45, 0, 0> translate <0, 200, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -49.09148> translate <259.7730, 100.040, 0> }
      plane { x, 0 rotate <0, -45, -49.09148> translate <259.7730, 100.040, -2> }
      plane {-x, 0 rotate <0, 0, -63.28423> translate <284.4171,  63.620, 0> }
      plane {-x, 0 rotate <0, 45, -63.28423> translate <284.4171,  63.620, -2> }
      plane {-x, 0 rotate <0, 45, 0> translate <336.8319, 0, -2> }
      plane {-y, 0 rotate <-45, 0, 0> translate <0, 145, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -63.28423> translate <284.4171,  63.620, 0> }
      plane { x, 0 rotate <0, -45, -63.28423> translate <284.4171,  63.620, -2> }
      plane {-x, 0 rotate <0, 0, -80.59573> translate <298.1878,  21.950, 0> }
      plane {-x, 0 rotate <0, 45, -80.59573> translate <298.1878,  21.950, -2> }
      plane {-x, 0 rotate <0, 45, 0> translate <376.98, 0, -2> }
      plane {-y, 0 rotate <-45, 0, 0> translate <0,  90, -2> }
      texture { T_RBBridge }
   }
   difference { 
      object { BridgeBase }
      plane { x, 0 rotate <0, 0, -80.59573> translate <298.1878,  21.950, 0> }
      plane { x, 0 rotate <0, -45, -80.59573> translate <298.1878,  21.950, -2> }
      plane {-x, 0 rotate <0, 45, 0> translate <400, 0, -2> }
      plane {-y, 0 rotate <-45, 0, 0> translate <0,  35, -2> }
      plane { y, 0 rotate < 45, 0, 0> translate <0, -20, -2> }
      texture { T_RBBridge }
   }
   
   //Foundation
   difference { 
      box { <300, -100, -5>, <50+RBBridgeLength/2*1.5, -20, 0> }
      box { <299, -101, -10>, <310, 1, -4> }          
      plane {-y, 0 rotate <-45, 0, 0> translate <0, -20, -2> }
      texture { T_RBBridge }
   }

   // Mortar
   difference { 
      box { <50-RBBridgeLength/2*1.5, -100, -3>, <50+RBBridgeLength/2*1.5, 290, 0> }
      cylinder { <0, 0, -10>, <0, 0, 10>, 201 scale <1.5, 1, 1> }  
      box { <-301.5, -101, -10>, <301.5, 0, 10> }
      texture { T_RBBridgeMortar }
   }
   
   //Algae
   difference { 
      box { <50-RBBridgeLength/2*1.5, -100, -5.2>, <50+RBBridgeLength/2*1.5, 290, -5.1> }
      cylinder { <0, 0, -10>, <0, 0, 10>, 201 scale <1.5, 1, 1> }  
      box { <-301.5, -101, -10>, <301.5, 0, 10> }
      texture { T_RBAlgae translate <0, 300-WaterLevel*1.5, 0> } 
      no_shadow
   }
      
   // Topedge
   union{       
      box { <-RBRailingLength/2*1.5, -20, -12>, <RBRailingLength/2*1.5, 0, 30> }
      box { <-RBRailingLength/2*1.5,  -8, -14>, <RBRailingLength/2*1.5, 1, 30> }
      cylinder { <-RBRailingLength/2*1.5, -3, -14>, <RBRailingLength/2*1.5, -3, -14>, 3 }
      texture { T_RBBridge }                           
      translate <0, 300, 0>
   }                             
         
   object { RBBridgeStone(103.7035, 45) translate <246.2965, 255, 0> }
   object { RBBridgeStone(100, 45) translate <350, 255, 0> }
   object { RBBridgeStone(117.9901, 55) translate <282.0099, 200, 0> }
   object { RBBridgeStone( 63.1681, 55) translate <336.8319, 90, 0> }
   object { RBBridgeStone( 73.02, 55) translate <376.98, 35, 0> }
  
   #local xpos = 400;
   #while (xpos < (RBBridgeLength/2*1.5))
      object { RBBridgeStone(100, 45) translate <xpos+50, 255, 0> }
      object { RBBridgeStone(100, 55) translate <xpos,    200, 0> }
      object { RBBridgeStone(100, 55) translate <xpos+50, 145, 0> }
      object { RBBridgeStone(100, 55) translate <xpos,     90, 0> }
      object { RBBridgeStone(100, 55) translate <xpos+50,  35, 0> }
      object { RBBridgeStone(100, 55) translate <xpos,    -20, 0> }
      #local xpos = xpos + 100;
   #end
      
}
#end

#declare RBBridgeYear=text {
   ttf concat(FontPath,"infr011n.ttf"), "ADMCM", 1, 0
   scale <30, 45, 5>
   texture { T_RBBridge }
}

#macro RBBridgeFaceRight(RBBridgeLength, WaterLevel)
union{
   object { RBBridgeFaceHalf(RBBridgeLength, WaterLevel) }
   difference {
   object { RBBridgeStone(138.3393, 55)  }
      box { <8, 8, -10>, <130, 47, -2.9> }    
      translate <311.6607, 145, 0> 
      texture { T_RBBridge }
   }                                      
   object { RBBridgeYear translate <321, 157, -5> }
   translate <0, -300, 0>
   scale <2/3, 2/3, 1> 
}
#end

#macro RBBridgeFaceLeft(RBBridgeLength, WaterLevel)
union{
   object { RBBridgeFaceHalf(RBBridgeLength, WaterLevel) }
   object { RBBridgeStone(138.3393, 55) translate <311.6607, 145, 0> }
   translate <0, -300, 0>
   scale <-2/3, 2/3, 1> 
}                                    
#end
      
#macro IR_RoadBridgeSide(RPBBridgeLength, RBRailingLength, WaterLevel)
union{
   object { RBBridgeFaceRight(RBBridgeLength, WaterLevel) }
   object { RBBridgeFaceLeft(RBBridgeLength, WaterLevel) }
   object { RBRailing(RBRailingLength) translate <0, 0, 30> }
}
#end

/*---------------- Assembled Bridge ------------------*/
   
#macro IR_RoadBridge
     (RBBridgeLength, RBRailingLength, RBWidth, SidewalkLevel, WaterLevel)
union {
   object { IR_RoadBridgeSide(RBBridgeLength, RBRailingLength, SidewalkLevel-WaterLevel) translate <0, SidewalkLevel, -RBWidth*0.5> }
   object { IR_RoadBridgeSide(RBBridgeLength, RBRailingLength, SidewalkLevel-WaterLevel) rotate <0, 180, 0> translate <0, SidewalkLevel, RBWidth*0.5> }
   difference {
      box { <-250, SidewalkLevel-270, -RBWidth*0.5>, <250, SidewalkLevel-50, RBWidth*0.5> }
      cylinder { <0, 0, -RBWidth>, <0, 0, RBWidth>, 400/3 scale <1.5, 1, 1> translate <0, SidewalkLevel-200, 0> }        
      box { <-600/3, SidewalkLevel-200, -RBWidth>, <600/3, SidewalkLevel-430, RBWidth> }
      texture { T_RBInside }
   }   
   difference {
      box { <-250, SidewalkLevel-270, -3-RBWidth*0.5>, <250, WaterLevel+50, 3+RBWidth*0.5> }
      cylinder { <0, 0, -RBWidth>, <0, 0, RBWidth>, 399.9/3 scale <1.5, 1, 1> translate <0, SidewalkLevel-200, 0> }        
      box { <-599.8/3, SidewalkLevel-200, -RBWidth>, <599.8/3, SidewalkLevel-430, RBWidth> }
      texture { T_RBAlgae scale <1, 2/3, 1> translate <0, WaterLevel, 0> }
   }   
   #declare IR_RoadBridge_x = RBBridgeLength;
   #declare IR_RoadBridge_y = 366;
   #declare IR_RoadBridge_z = RBWidth;
}
#end 

// ///////////////////////////////////////////////////////////////////
