// POV-Ray Push-Ball Reply File
// *****************************
   
// See "pushball.txt" for more information!
   
// Include the file with the game data.
   #include "pb_output.inc" // You can also try "match2data.inc"
   
   #macro Perpendiculize (V1,V2)
      vnormalize(vcross(vcross(V2,V1),V2))
   #end
   
   #macro FindAngle (V1,V2,About)
      degrees(acos(vdot(vnormalize(V1),vnormalize(V2))))
      *(vdot(About,vcross(V1,V2))<0?-1:1)
   #end
   
// FRAME AND DATA CONTROL
// **********************
   
// The number of frames in the animation doesn't have to be
// the same as the number of frames in the data file.
// Here we #declare which "dataframe" the current frame matches.
   #declare frame = clock*(dimension_size(pb_output,1)-1);
   
   #declare frameold = max(frame-1,0);
   
   #debug concat(" Dataframe: ",str(frame,2,2),"\n")
   
// Initialize arrays for the locations of each bot.
   #declare pb_location    = array[pb_players+1]
   #declare pb_locationold = array[pb_players+1]
   #declare pb_botpower    = array[pb_players+1]
   
// Find current location and old location for each bot using
// linear interpolation between dataframes.
   #declare C = 0;
   #while (C<=pb_players)
      
      #if (int(frame)=frame)
         #declare pb_location[C] = pb_output[frame][0][C];
      #else
         #declare pb_location[C] = (
            + (1-mod(frame,1)) * pb_output[int(frame)  ][0][C]
            + (  mod(frame,1)) * pb_output[int(frame)+1][0][C]
         );
      #end
      
      #declare pb_locationold[C] = (
         + (1-mod(frameold,1)) * pb_output[int(frameold)  ][0][C]
         + (  mod(frameold,1)) * pb_output[int(frameold)+1][0][C]
      );
      
      #if (int(frame)=frame)
         #declare pb_botpower[C] = pb_output[frame][1][C];
      #else
         #declare pb_botpower[C] = (
            + (1-mod(frame,1)) * pb_output[int(frame)  ][1][C]
            + (  mod(frame,1)) * pb_output[int(frame)+1][1][C]
         );
      #end
      
      #declare C = C+1;
   #end
   
// Include data saved in previous animation frame.
   #if (frame>1) #include "#re_data.inc" #end

// Make sure the bots always have a direction vector.
   #declare C = 1;
   #while (C<=pb_players)
      
      #if (vlength(pb_botpower[C])=0)
         #declare pb_botpower[C] = vnormalize(pb_botpowerold[C])*0.001*pb_botsize;
      #end
      
      #declare C = C+1;
   #end
   
// Make sure the ball always have a X and a Y direction.
   #if (defined(pb_ballxold)=false)
      #declare pb_ballx = z;
      #declare pb_bally = y;
   #else
      
      #declare pb_ballvector = pb_location[0]-pb_balltold;
      
      #if (vlength(pb_ballvector)=0)
         #declare pb_ballx = pb_ballxold;
         #declare pb_bally = pb_ballyold;
      #else
         #declare pb_ballx = vaxis_rotate (
            vnormalize(pb_ballxold),
            vnormalize(vrotate(pb_ballvector,90*y)),
            vlength(pb_ballvector)/pb_botsize/pi*360
         );
         #declare pb_bally = vaxis_rotate (
            vnormalize(pb_ballyold),
            vnormalize(vrotate(pb_ballvector,90*y)),
            vlength(pb_ballvector)/pb_botsize/pi*360
         );
         #declare pb_bally = Perpendiculize(pb_bally,pb_ballx);
      #end
      
   #end
   
   #declare pb_ballz = vnormalize(vcross(pb_ballx,pb_bally));
   
// CAMERA CONTROL
// **************
   
// Find out the average location of the bots:
   #declare pb_camlook = (
      #declare C = 0;
      #while (C<=pb_players)
         +pb_location[C]
         #declare C = C+1;
      #end
   ) / (pb_players+1);
   
// Find out the max difference from the average location:
   #declare pb_maxdist = 0;
   #declare C = 0;
   #while (C<=pb_players)
      #declare pb_maxdist = max(
         vlength(pb_location[C]-pb_camlook),
         pb_maxdist
      );
      #declare C = C+1;
   #end
   
   #declare pb_camloc = <0,2.0,-1.5>*(pb_fieldsize+pb_botsize);
   #declare pb_camdist = vlength(pb_camloc-pb_camlook);
   #declare pb_camang = (
      +degrees(atan2(pb_maxdist,pb_camdist))*4.0
      +degrees(atan2((pb_fieldsize+pb_botsize)/2,pb_camdist))*4.0
   )/2;
   
   camera {
      location pb_camloc
      angle pb_camang
      look_at pb_camlook
   }
   
// BOT OBJECT MACRO
// ****************
   
   #macro pb_bot (botnr)
      
      #declare botfinish = finish {phong 1}
      
      #declare botobject =
      union {
         difference {
            sphere {0, 1}
            box {<-1.1,-1.1,-1.1>, <1.1,-0.4,1.1>}
            pigment {
               gradient z
               scale 2.01
               translate -z
               color_map {
                  [0.75, color pb_color[C]]
                  [0.75, color rgb 0]
               }
            }
            finish {botfinish}
         }
         cylinder {
            -0.5*y, 0, 1
            pigment {color pb_color[C]}
            finish {botfinish}
         }
         difference {
            cylinder {
               <0,0,-0.5>, <0,0,-1.1>, 0.3
               pigment {color pb_color[C]}
               finish {botfinish}
            }
            cylinder {
               <0,0,-1.05>, <0,0,-1.15>, 0.4
               pigment {
                  spherical
                  scale 0.4
                  translate -1.1*z
                  color_map {
                     [0, color <1.0,0.0,0.0>*vlength(pb_botpower[C])/pb_power]
                     [1, color <1.5,1.5,0.0>*vlength(pb_botpower[C])/pb_power]
                  }
               }
               finish {ambient 1 diffuse 0}
            }
         }
      }
      
      #declare bot_y = vnormalize(
         +y
         +(pb_locationold[C]-pb_location[C])/3/pb_maxspeed
      );
      #declare bot_z = Perpendiculize (pb_botpower[C],bot_y);
      #declare bot_x = vnormalize(vcross(bot_y,bot_z));
      
      object {
         botobject
         matrix <
            bot_x.x,bot_x.y,bot_x.z,
            bot_y.x,bot_y.y,bot_y.z,
            bot_z.x,bot_z.y,bot_z.z,
            0,0,0
         >
         translate 1.05*y
         scale pb_botsize/2
         translate pb_location[C]
      }
      
   #end
   
// GENERAL SCENE SETTINGS
// **********************
   
   #default {finish {ambient 0.4}}
   
// BOT CONTROL
// ***********
   
// Initialize array with colors for each bot.
   #declare pb_color = array[pb_players+1]
   
// The ball is white.
   #declare pb_color[0] = color rgb 1;
   
// The bots are given colors based on a smooth color cycle that goes like:
// Red - green - blue - red.
   #declare C = 1;
   #while (C<=pb_players)
      #declare pb_color[C] = color vaxis_rotate(x,<1,1,1>,(C-1)/pb_players*360);
      #declare pb_color[C] = color <
         pow(pb_color[C].red,0.5),
         pow(pb_color[C].green,0.5),
         pow(pb_color[C].blue,0.5)
      >;
      #declare C = C+1;
   #end
   
// Place the ball in the scene.
   sphere {
      0, pb_botsize/2
      pigment {
         radial
         frequency 8
         scale 0.3
         rotate 45*x
         color_map {
            [0.3, color 0          ]
            [0.3, color pb_color[0]]
         }
      }
      matrix <
         pb_ballx.x,pb_ballx.y,pb_ballx.z,
         pb_bally.x,pb_bally.y,pb_bally.z,
         pb_ballz.x,pb_ballz.y,pb_ballz.z,
         0,0,0
      >
      translate pb_location[0]+pb_botsize*y/2
   }
   
// Place the players in the scene.
   #declare C = 1;
   #while (C<=pb_players)
      pb_bot(C)
      #declare C = C+1;
   #end
   
// THE SCENE
// *********
   
   light_source {<0,2,-1>*1000*y, color 1.0}
   
   plane {
      y, 0
      pigment {checker color rgb 1, color rgb 0.8 scale 2}
   }
   
// The inner grey patterned area.
   disc {
      0.002*y, y, pb_fieldsize
      pigment {
         onion
         frequency 1
         pigment_map {
            [0.5, color rgb 1]
            [
               0.5,
               radial
               frequency 24
               rotate 360/24/4*y
               color_map {[0.5, color rgb 1][0.5, color rgb 0.8]}
            ]
         }
      }
   }
   
// The outer area and the wall, which both have the goal colors on them.   
   union {
      disc {
         0.001, y, pb_fieldsize+pb_botsize*1.5
      }
      torus {
         pb_fieldsize+pb_botsize*1.5, pb_botsize/2
         translate pb_botsize/2*y
      }
      pigment {
         radial
         rotate (FindAngle(x,pb_goalcenter[1],y)-360/pb_players/2)*y
         color_map {
            #declare C = 1;
            #while (C<=pb_players)
               [(C-1)/pb_players, color pb_color[C]]
               [(C  )/pb_players, color pb_color[C]]
               #declare C = C+1;
            #end
         }
      }
   }
   
// SAVE DATA
// *********
   
   #fopen data "#re_data.inc" write
      
      #write (data,"#declare pb_botpowerold = array[",pb_players+1,"] {")
      #local C = 0;
      #while (C<pb_players+1)
         #write (data,pb_botpower[C],",")
         #local C = C+1;
      #end
      #write (data,"}\n\n")
      
      #write (data,"#declare pb_ballxold = ",pb_ballx,";\n")
      #write (data,"#declare pb_ballyold = ",pb_bally,";\n")
      #write (data,"#declare pb_balltold = ",pb_location[0],";\n")
      
   #fclose data
   