// POV-Ray Push-Ball Include File
// ******************************
   
// See "pushball.txt" for more information!
   
// If the data file "#pb_data.inc" doesn't exist, this is the first frame.
   #if (file_exists("#pb_data.inc")=0)
      
//    In the first frame we need to:
      
//    CREATE INITIAL SETTINGS
//    ***********************
      
//    This is the first frame...
      #declare pb_frame = 1;
      
//    In order to create random initial positions, we need a seed.
      #declare pb_seed = seed(pb_seednr);
      
//    In the first frame no goal has been scored.
      #declare pb_goal = false;
      
//    Initialize array with positions for ball and players.
      #declare pb_location = array[pb_players+1]
      
//    The ball works as if it were player nr. 0
//    The initial position of the ball is in the middle of the playing field.      
      #declare pb_location[0] = <0,0,0>;
      
//    Initialize array with old positions for ball and players.
//      #declare pb_locationold = array[pb_players+1]
      
//    Initialize array with direction vectors.
      #declare pb_botpower = array[pb_players+1]
      #declare pb_botpower[0] = <0,0,0>;
      
//    Initialize array with locations for each player's goal center.
      #declare pb_goalcenter = array[pb_players+1]
      
//    The ball doesn't have a goal, but I have just initialized it anyway.
      #declare pb_goalcenter[0] = <0,0,0>;
      
//    Random number between 1 and 100;      
      #declare pb_rot = rand(pb_seed)*100;
      
//    Make random initial positions for each player and goal:
      
      #declare C = 1; // starting at 1, not 0 which is the ball.      
      #while (C<=pb_players)
         
//       Create the initial location for the current player:
         #declare pb_location[C] = vrotate(pb_fieldsize*x,(C/pb_players+pb_rot)*360*y);
         
//       Create initial position of current player's goal:
         #declare pb_goalcenter[C] = vrotate(pb_location[C],pb_goalrot*y);
         
//       Create the direction vector of the bots in frame 1:
         #declare pb_botpower[C] = pb_location[0]-pb_location[C];
         
         #declare C = C+1;
      #end
      
//    There's no location data from before frame 1, so we copy it.
      #declare pb_locationold = pb_location
      
//    The information from each frame is stored in "pb_output.inc".
//    Here we write the options that are the same for each frame:
      #fopen Output "pb_output.inc" write
         
         #write (Output,"#declare pb_players = ",pb_players,";\n")
         #write (Output,"#declare pb_botsize = ",pb_botsize,";\n")
         #write (Output,"#declare pb_fieldsize = ",pb_fieldsize,";\n")
         #write (Output,"#declare pb_goalrot = ",pb_goalrot,";\n")
         #write (Output,"#declare pb_power = ",pb_power,";\n")
         #write (Output,"#declare pb_maxspeed = ",pb_maxspeed,";\n\n")
         
         #write (Output,"#declare pb_goalcenter = array[",pb_players+1,"] {")
         #local C = 0;
         #while (C<pb_players+1)
            #write (Output,pb_goalcenter[C],",")
            #local C = C+1;
         #end
         #write (Output,"}\n\n")
         
//       Here we write the start of the big array which will contains all
//       location info for all frames.
//       (We don't know how many frames the game will take, so we write XXX.
//       The user later must change this manually)
         #write (Output,"#declare pb_output = array[ XXX ][2][",pb_players+1,"] {\n")
         
      #fclose Output
   
   #else // In case this is not the first frame, but just a regular frame.
      
//    Include the file that knows the location data for the two previous
//    frames, what the frame number is, and if there has been scored yet.
      #include "#pb_data.inc"
      
//    This is a new frame...
      #declare pb_frame = pb_frame+1;
      
//    Create a seed that is different for each frame.
      #declare pb_seed = seed(pb_seednr+pb_frame);
      
//    Initialize the needed arrays.
      #declare pb_botpower    = array[pb_players+1] // The direction the bot wants to go
      #declare pb_target      = array[pb_players+1] // The target the bot wants to go to
      #declare pb_botinertia  = array[pb_players+1] // Inertia force of the bot
      #declare pb_botforces   = array[pb_players+1] // Forces caused by collisions.
      #declare pb_rule        = array[pb_players+1] // Forces caused by braking a "rule".
      #declare pb_botvibrate  = array[pb_players+1] // Small random force.
      #declare pb_vector      = array[pb_players+1] // All the forces added together.
      #declare pb_locationnew = array[pb_players+1] // The new location.
      
//    GET DIRECTION VECTORS FROM BOTS
//    *******************************
      
//    The ball does not move on its own.
      #declare pb_botpower[0] = <0,0,0>;
      
//    However, the players DO move on their own.
//    Here we include each player's include file.
      #declare pb_current = 1;
      #while (pb_current<=pb_players)
         
//       Include the current player.
         #include pb_name[pb_current-1]
         
//       Save his direction vector in the array.
         #declare pb_botpower[pb_current] = VECTOR*<1,0,1>;
         
//       Calculate his target, and save it in another array
//       which will be used for debugging.
         #declare pb_target[pb_current] = pb_location[pb_current]+pb_botpower[pb_current];
         
//       Hide the vector so next player won't use it by a mistake.
         #undef VECTOR
         
         #declare pb_current = pb_current+1;
      #end
      
//    DO BOT CALCULATIONS
//    *******************
      
//    Do calculations for each bot (both the ball and all the players).
      #declare C = 0;
      #while (C<=pb_players)
         
//       If bot's direction vector is longer than allowed, cut it down.
         #if (vlength(pb_botpower[C])>pb_power)
            #declare pb_botpower[C] = vnormalize(pb_botpower[C])*pb_power;
         #end
         
//       Calculate inertia for bot.
         #declare pb_botinertia[C] = (pb_location[C]-pb_locationold[C])*pb_inertia;
         
//       Calculate forces for bot caused by bumping into other bots.
         #declare pb_botforces[C] = <0,0,0>;
         #declare D = 0; // D = the bot the current bot is tested against.
         #while (D<=pb_players)
            
            #if (C!=D) // Bot shouldn't be tested against itself.
            
//             If the two bots touches. (intersects)
               #if (vlength(pb_location[C]-pb_location[D])<pb_botsize)
                  
//                Add to the force vector a new vector that pushes the current bot
//                away from the bot it is intersecting with.
                  #declare pb_botforces[C] = (
                     pb_botforces[C]
                     + (pb_botsize-vlength(pb_location[C]-pb_location[D]))
                     * vnormalize(pb_location[C]-pb_location[D])
                  );
                  
               #end
               
            #end
            
            #declare D = D+1; // Test against next bot.
         #end
         
//       Now we want to add another force if the bot hits the outer wall in the game.
         
//       If the center of the bot exceeds this radius, it will bounce.
         #declare pb_wallradius = pb_fieldsize+pb_botsize/2;
         
         #declare pb_rule[C] = <0,0,0>;
         
//       Test if the bot exceeds the radius.
         #if (vlength(pb_location[C])>pb_wallradius)
            
//          If it do, add a force that pushes it away.
            #declare pb_rule[C] = (
               +vnormalize(-pb_location[C])
               *( vlength(pb_location[C]) - pb_wallradius )
            );
            
         #end
         
//       Add a very small random force to the bot.
         #declare pb_botvibrate[C] = vrotate(x,rand(pb_seed)*36000*y)*pb_vibrate;
         
//       Now add all the forces together.
         #declare pb_vector[C] = (
            +pb_botpower[C]   // The direction the bot wants to go.
            +pb_botinertia[C] // The inertia force.
            +pb_botforces[C]  // Collision forces when bumping into other robots.
            +pb_rule[C]    // Collision force when bumping into the wall.
            +pb_botvibrate[C] // Small random force.
         )*<1,0,1>; // Keep it on the ground.
         
//       If the resulting speed is too fast, cut it down.
         #if (vlength(pb_vector[C])>pb_maxspeed)
            #declare pb_vector[C] = vnormalize(pb_vector[C])*pb_maxspeed;
         #end
         
//       The bot's new location is the current location + the vector.
         #declare pb_locationnew[C] = pb_location[C]+pb_vector[C];
         
         #declare C = C+1; // Do bot calculations for the next bot.
      #end
      
//    Update the locations.
      #declare pb_locationold = pb_location
      #declare pb_location    = pb_locationnew
      
//    See if a goal has been scored:
      #if ( (vlength(pb_location[0])>pb_fieldsize) | pb_goal=true )
         #declare pb_goal = true;
         #debug "--- GOAL SCORED ---"
      #end
      
   #end // The rest is parsed no matter if this is the first frame or not.
   
// SAVE DATA
// *********
   
   #fopen Data "#pb_data.inc" write
      
      #write (Data,"#declare pb_frame = ",pb_frame,";\n")
      
      #write (Data,"#declare pb_goal = ",pb_goal,";\n")
      
      #write (Data,"#declare pb_goalcenter = array[",pb_players+1,"] {")
      #local C = 0;
      #while (C<pb_players+1)
         #write (Data,pb_goalcenter[C],",")
         #local C = C+1;
      #end
      #write (Data,"}\n")
      
      #write (Data,"#declare pb_locationold = array[",pb_players+1,"] {")
      #local C = 0;
      #while (C<pb_players+1)
         #write (Data,pb_locationold[C],",")
         #local C = C+1;
      #end
      #write (Data,"}\n")
      
      #write (Data,"#declare pb_location    = array[",pb_players+1,"] {")
      #local C = 0;
      #while (C<pb_players+1)
         #write (Data,pb_location[C],",")
         #local C = C+1;
      #end
      #write (Data,"}\n")
      
   #fclose Data
   
   #fopen Output "pb_output.inc" append
      
      #write (Output,"{ {")
      
      #local C = 0;
      #while (C<=pb_players)
         #write (Output,pb_location[C],",")
         #local C = C+1;
      #end
      
      #write (Output,"}, // ",pb_frame,"\n{")
      
      #local C = 0;
      #while (C<=pb_players)
         #write (Output,pb_botpower[C],",")
         #local C = C+1;
      #end
      
      #write (Output,"} },\n")
      
   #fclose Output
   
// MAKE SIMPLE GAME DESIGN
// ***********************
   
   camera {
      location (pb_fieldsize*2+pb_botsize*2)*y
      look_at 0
   }
   
// The ball
   sphere {pb_location[0], pb_botsize/2 pigment {color rgb 10}}
   
// The players
   #declare C = 1;
   #while (C<=pb_players)
      
//    Player
      sphere {
         pb_location[C], pb_botsize/2
         pigment {color vaxis_rotate(10*x,<1,1,1>,(C-1)/pb_players*360)}
      }
      
//    Players' targets (used for debugging)
      #ifdef (pb_target)
         sphere {
            pb_target[C]+pb_botsize*y/2, pb_botsize/4
            pigment {color vaxis_rotate(10*x,<1,1,1>,(C-1)/pb_players*360)}
         }
      #end
      
      #declare C = C+1;
   #end
   
// Edge of the playing field.
   torus {pb_fieldsize, pb_botsize/10 pigment {color rgb 5}}
   
// Outer wall of the playing field.
   torus {pb_fieldsize+pb_botsize, pb_botsize/10 pigment {color rgb 10}}
   
// Lines to seperate the goals.
   #declare C = 1;
   #while (C<=pb_players)
      
      cylinder {
         vnormalize(pb_goalcenter[C])*pb_fieldsize,
         vnormalize(pb_goalcenter[C])*(pb_fieldsize+pb_botsize),
         pb_botsize/10
         pigment {color rgb 5}
         rotate 180/pb_players*y
      }
      
      #declare C = C+1;
   #end
   