/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* POV 3.0 include file                                                    */
/*                                                                         */
/*           STONE WALL WITH ALTERNATE STONE PLACEMENT ALGORITHM           */
/*                                                                         */
/*                            By Dave Blandston                            */
/*                                                                         */
/* This .inc file can be used to create two different types of stone       */
/* walls. Type 1 is a "standard" brick wall, and type 2 features semi-     */
/* randomly placed stones. To use this .inc file, simply include it        */
/* somewhere in your scene file, then when you need a stone wall call the  */
/* "MakeBrickWall" macro. You will need to supply the width, height, and   */
/* depth of your wall as parameters, as well as the brick size and type of */
/* stone placement you want. The wall will be created as an object called  */
/* "BrickWall," which you can then manipulate at will. The lower left      */
/* front corner of the wall will be at the origin.                         */
/*                                                                         */
/* Note: You also need to include "colors.inc" and "stones.inc" before     */
/* including "DBStones.inc."                                               */
/*                                                                         */
/* Example usage:                                                          */
/*    include "colors.inc"                                                 */
/*    include "stones.inc"                                                 */
/*    include "DBStones.inc"                                               */
/*    .                                                                    */
/*    .                                                                    */
/*    .                                                                    */
/*    #local WallWidth = 20;                                               */
/*    #local WallHeight = 12;                                              */
/*    #local WallDepth = 1;                                                */
/*    #local BrickSize = 1;                                                */
/*    #local PlacementMethod = 1; //Standard brick wall                    */
/*    MakeBrickWall (WallWidth, WallHeight, WallDepth, PlacementMethod)    */
/*    object {BrickWall} //Make the wall visible!                          */
/*                                                                         */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#declare BrickRandomizer = seed (1);

#declare Pit = height_field {
   pot "DBStones.pot" //Fractint plasma, 1024x768, graininess = 12
   smooth
   rotate 90 * x
   translate <-.5, .5, 0>
} //height_field

#declare nBrickColors = 5;

#declare BrickColor = array [nBrickColors] {
   texture {T_Stone1 finish {crand .1 phong .2 reflection 0}},
   texture {T_Stone3 finish {crand .1 phong .2 reflection 0}},
   texture {T_Stone4 finish {crand .1 phong .2 reflection 0}},
   texture {T_Stone11 finish {crand .1 phong .2 reflection 0}},
   texture {T_Stone33 finish {crand .1 phong .2 reflection 0}}
} //array

#declare MortarColor = texture {
   pigment {
      bozo
      color_map {
         [0, 1 color DarkOliveGreen color DarkWood]
      } //color_map
      scale 4
   } //pigment
   normal {granite}
} //texture

#declare nBrickWalls = 0;

#macro Brick (BrickWidth, BrickHeight, BrickDepth, BrickX, BrickY, BrickZ, BrickColor, WallWidth, WallHeight, WallDepth)

   #local PitOrientation = int (rand (BrickRandomizer) * 4) * 360;
   #local ZScale = rand (BrickRandomizer) * .4 + .3;
   #local ZDist = rand (BrickRandomizer) * ZScale * 1.2 + .3;
   #local XOffset = rand (BrickRandomizer) * (MaxDim - BrickWidth);
   #local YOffset = rand (BrickRandomizer) * (MaxDim - BrickHeight);
   #local TextureOffset = rand (BrickRandomizer) * 10000;

   //#local NewBrick = box {<0, 0, 0> <BrickWidth, BrickHeight, BrickDepth> texture {BrickColor}}
   //#local NewBrick = superellipsoid {<.15, .15> translate <1, 1, 1> scale .5 scale <BrickWidth, BrickHeight, BrickDepth> texture {BrickColor}}
   #local NewBrick = difference {
      superellipsoid {
         <.15, .15>
         translate <1, 1, 1>
         scale .5
         scale <BrickWidth, BrickHeight, BrickDepth>
         texture {
            BrickColor
            translate TextureOffset * x
            normal {granite .25}
         } //texture
      } //superellipsoid
      intersection {
         object {
            Pit
            rotate PitOrientation * z
            translate <.5, .5, 0>
            scale <MaxDim * 1.0001, MaxDim * 1.0001, 1>
            translate <-.000001, -.000001, 0>
            translate <-XOffset, -YOffset, -ZDist>
            scale <1, 1, ZScale>
            texture {
               BrickColor translate TextureOffset * x
               normal {granite 0}
               finish {phong .1}
            } //texture
         } //object
         plane {z, rand (BrickRandomizer) * .3 + .2 texture {BrickColor translate TextureOffset * x normal {granite}}}
      } //intersection
   } //difference

   object {
      intersection {
         object {NewBrick translate <BrickX, BrickY, BrickZ>}
         box {<0, 0, -.1> <WallWidth, WallHeight, WallDepth + .1> texture {MortarColor}}
      } //intersection
   } //object

#end //macro Brick

#macro FindCurHeight (WhichBrick)

   #declare CurHeight = 9999;

   #declare LeftBorder = BrickX1 [WhichBrick];
   #declare RightBorder = BrickX2 [WhichBrick];

   #local Found = false;

   #local i = WhichBrick - 1;

   #while (i > 0)
      #local StillGoing = true;
      #if (BrickX2 [i] < BrickX1 [WhichBrick] | BrickX1 [i] > BrickX2 [WhichBrick])
         #local StillGoing = false;
         #if (Found)
            #local i = 0;
         #end //if
      #end //if
      #if (StillGoing)
         #local THeight = BrickY1 [WhichBrick] - BrickY2 [i];
         #if (THeight < CurHeight & THeight > MaxGap + MinGap)
            #declare CurHeight = THeight;
         #end //if
         #if (BrickX1 [i] > BrickX1 [WhichBrick] & BrickX2 [i] < BrickX2 [WhichBrick] & THeight < CurHeight)
            #declare CurHeight = THeight;
         #end //if
         #if (Found)
            #declare X1Height = THeight;
            #declare LeftBorder = BrickX2 [i];
         #else
            #declare X2Height = THeight;
            #declare RightBorder = BrickX1 [i];
         #end //if
         #local Found = true;
      #end //if
      #local i = i - 1;
   #end //while

#end //macro FindCurHeight

#macro MakeBrickWall (WallWidth, WallHeight, WallDepth, BrickSize, PlacementMethod)

   #local MaxDim = 12; //This scale value must be larger than the largest brick dimension.
   
   #local MaxGap = .06; //Mortar gaps
   #local MinGap = .02;
   
   #local MaxBrickWidth = BrickSize;
   #switch (PlacementMethod)
      #case (1) //Standard brick wall
         #local MaxBrickHeight = MaxBrickWidth * .5;
      #break
      #case (2) //Randomly placed stones
         #local MaxBrickHeight = MaxBrickWidth * .8;
      #break
   #end //switch
   #local MaxBrickDepth = WallDepth;

   #local MaxNWide = ceil (WallWidth / (MaxBrickWidth / 1.3));
   #local MaxNHigh = ceil (WallHeight / (MaxBrickHeight / 1.3));
   #local MaxArray = MaxNWide * MaxNHigh;

   #declare BrickX1 = array [MaxArray]
   #declare BrickY1 = array [MaxArray]
   #declare BrickX2 = array [MaxArray]
   #declare BrickY2 = array [MaxArray]

   #declare nBricks = 0;

   #switch (PlacementMethod)
      #case (1) //Standard brick wall
         #local LastStartX = -MaxBrickWidth * 1.5 + rand (BrickRandomizer) * MaxBrickWidth;
         #local CurY = 0;
         #local Bricks = object {
            union {
               #while (CurY < WallHeight)
                  #local CurX = LastStartX + rand (BrickRandomizer) * (MaxBrickWidth * .5);
                  #local LastStartX = CurX;
                  #if (CurX > -MaxBrickWidth * .5)
                     #local LastStartX = -MaxBrickWidth * 1.5 + rand (BrickRandomizer) * MaxBrickWidth;
                  #end //if
                  #local CurHeight = rand (BrickRandomizer) * (MaxBrickHeight * .25) + MaxBrickHeight * .75;
                  #while (CurX < WallWidth)
                     #local CurWidth = rand (BrickRandomizer) * (MaxBrickWidth * .33) + MaxBrickWidth * .67;
                     Brick (CurWidth, CurHeight, MaxBrickDepth, CurX, CurY, 0, BrickColor [int (rand (BrickRandomizer) * nBrickColors)], WallWidth, WallHeight, WallDepth)
                     #declare nBricks = nBricks + 1;
                     #declare BrickX1 [nBricks] = CurX;
                     #declare BrickY1 [nBricks] = CurY;
                     #declare BrickX2 [nBricks] = CurX + CurWidth;
                     #declare BrickY2 [nBricks] = CurY + CurHeight;
                     #local CurX = CurX + CurWidth + rand (BrickRandomizer) * MaxGap + MinGap;
                  #end //while
                  #local CurY = CurY + CurHeight + rand (BrickRandomizer) * MaxGap + MinGap;
               #end //while
            } //union
         } //object Bricks
      #break
      #case (2) //Randomly placed stones
         #local MinUsablePct = .28; //Fraction of brick size that can be filled in
         #declare CurY = -999999;
         #local Bricks = object {
            union {
               #while (CurY < WallHeight)
                  #declare CurX = -rand (BrickRandomizer) * MaxBrickWidth;
                  #while (CurX <= WallWidth)
                     #declare CurWidth = rand (BrickRandomizer) * MaxBrickWidth / 1.3 + MaxBrickWidth / 1.3;
                     #declare CurHeight = rand (BrickRandomizer) * MaxBrickHeight / 1.3 + MaxBrickHeight / 1.3;
                     #local Highest = -rand (BrickRandomizer) * MaxBrickHeight;
                     #local i = 1;
                     #while (i <= nBricks)
                        #if ((CurX >= BrickX1 [i] & CurX <= BrickX2 [i]) | (CurX + CurWidth >= BrickX1 [i] & CurX + CurWidth <= BrickX2 [i]) | (CurX <= BrickX1 [i] & CurX + CurWidth >= BrickX2 [i]))
                           #if (BrickY2 [i] > Highest)
                              #local Highest = BrickY2 [i];
                           #end //if
                        #end //if
                        #local i = i + 1;
                     #end //while
                     #declare CurY = Highest + rand (BrickRandomizer) * MaxGap + MinGap;
                     #if (CurY <= WallHeight)
                        Brick (CurWidth, CurHeight, MaxBrickDepth, CurX, CurY, 0, BrickColor [int (rand (BrickRandomizer) * nBrickColors)], WallWidth, WallHeight, WallDepth)
                        #declare nBricks = nBricks + 1;
                        #declare BrickX1 [nBricks] = CurX;
                        #declare BrickY1 [nBricks] = CurY;
                        #declare BrickX2 [nBricks] = CurX + CurWidth;
                        #declare BrickY2 [nBricks] = CurY + CurHeight;
                     #end //if
                     #declare CurX = CurX + CurWidth + rand (BrickRandomizer) * MaxGap + MinGap;
                  #end //while
               #end //while
               #local i = 1;
               #local OldNBricks = nBricks;
               #while (i <= OldNBricks)
                  FindCurHeight (i)
                  #if (CurHeight = 9999)
                     #declare CurHeight = 0;
                  #end //if
                  #if (CurHeight > MaxBrickHeight * MinUsablePct)
                     #if (X1Height > MaxBrickHeight * MinUsablePct)
                        #declare LeftBorder = BrickX1 [i];
                     #end //if
                     #if (X2Height > MaxBrickHeight * MinUsablePct)
                        #declare RightBorder = BrickX2 [i];
                     #end //if
                     #declare CurWidth = RightBorder - LeftBorder;
                     #if (CurWidth > MaxBrickWidth * MinUsablePct)
                        #declare LeftBorder = LeftBorder + (rand (BrickRandomizer) * MaxGap + MinGap);
                        #declare RightBorder = RightBorder - (rand (BrickRandomizer) * MaxGap + MinGap);
                        #declare CurWidth = RightBorder - LeftBorder;
                        #local TopGap = rand (BrickRandomizer) * MaxGap + MinGap;
                        #local BotGap = rand (BrickRandomizer) * MaxGap + MinGap;
                        #declare CurHeight = CurHeight - TopGap - BotGap;
                        #declare CurY = BrickY1 [i] - CurHeight - TopGap;
                        Brick (CurWidth, CurHeight, MaxBrickDepth, LeftBorder, CurY, 0, BrickColor [int (rand (BrickRandomizer) * nBrickColors)], WallWidth, WallHeight, WallDepth)
                        #declare nBricks = nBricks + 1;
                        #declare BrickX1 [nBricks] = LeftBorder;
                        #declare BrickY1 [nBricks] = CurY;
                        #declare BrickX2 [nBricks] = RightBorder;
                        #declare BrickY2 [nBricks] = CurY + CurHeight;
                     #end //if
                  #end //if
                  #local i = i + 1;
               #end //while
            } //union
         } //object Bricks
      #break
   #end //switch

   #local Mortar = object {
      difference {
         box {<0, 0, .2> <WallWidth, WallHeight, WallDepth - .2>}
         #local i = 1;
         #while (i <= nBricks)
            superellipsoid {
               <.15, .15>
               translate <1, 1, 1>
               scale .5
               scale <BrickX2 [i] - BrickX1 [i], BrickY2 [i] - BrickY1 [i], WallDepth + .2>
               translate <BrickX1 [i], BrickY1 [i], -.1>
            } //superellipsoid
            #local i = i + 1;
         #end //while
      } //difference
      texture {MortarColor}
   } //object Mortar

   #declare BrickWall = object {
      union {
         object {Bricks}
         object {Mortar}
      } //union
   } //object BrickWall

   #declare nBrickWalls = nBrickWalls + 1;
   
   #if (nBrickWalls = 1)
      #debug concat("\r\n")
   #end //if
   
   #debug concat("Stone wall number: ", str (nBrickWalls, 0, 0), "\r\n")
   #debug concat("Placement method: ", str (PlacementMethod, 0, 0), "\r\n")
   #debug concat("Number of bricks used: ", str (nBricks, 0, 0), "\r\n")
   #debug concat("Maximum number of bricks allowable: ", str (MaxArray, 0, 0), "\r\n")

   #if (nBricks > MaxArray)
      #debug concat("*******************************************************************************\r\n")
      #debug concat("* Error: Exceeded maximum number of bricks allowed. Increase MaxArray to fix. *\r\n")
      #debug concat("*******************************************************************************\r\n")
   #end //if

   #debug concat("\r\n")

   #undef BrickX1
   #undef BrickY1
   #undef BrickX2
   #undef BrickY2

#end //macro MakeBrickWall
