//////////////////////////////////////////////////////////////////////////////
//
// ruined_wall_macro.inc
//
// Copyright Ian Shumsky / OuterArm 2000
//
//////////////////////////////////////////////////////////////////////////////
//
// This include file builds a crumbled wall. The wall starts at <0, 0, 0> and
// extends in the +x axis. All measurements are in meters.
//
// marco parameters:
//
// rw_wall_length      : The maximum length of the wall
// rw_wall_height      : The maximum height of the wall
// rw_max_bricks_z     : The number of bricks thick the wall is
// rw_brick_height     : The y dimension of the brick object
// rw_brick_width      : The z dimension of the brick object 
// rw_brick_length     : The x dimension of the brick object 
// rw_mortar_thickness : How much space is between bricks 
// rw_mortar_depth     : How far below the surface of the bricks the mortar
//                       starts  
// rw_brick_type       : The type of brick. Valid values are 0 for a box
//                       brick, any other value for a superelipsoid 
// rw_seed_value       : A randow seed value.
//
//////////////////////////////////////////////////////////////////////////////

#include "stones.inc"

//
// Define a random seed.
//

#declare rw_rand1 = seed (0);

//
// Create some textures for the bricks and mortar.
//

#declare rw_brick_texture_01 = texture
{
	T_Grnt14
	normal
	{
		granite
	}
	scale .5
}

#declare rw_brick_texture_02 = texture
{
	T_Grnt14
	normal
	{
		granite
	}
	scale .5
}
texture
{
	pigment
	{
		granite
		color_map
		{
			[0.0 Clear]
			[0.9 Gold filter 0.5]
			[1.0 DarkTan filter 0.75]
		}
	}
	normal
	{
		crackle 10
		turbulence 0.5
		lambda 3
	}
	scale 3.5
}

#declare rw_mortar_texture = texture
{
	pigment
	{
		colour rgb <0.9, 0.89, 0.85>
	}
	normal
	{
		granite
	}
}

//
// A simple macro to pick one of the two brick testures and
// 'jiggle' it.
//

#macro get_brick_texture ()
		#if (rand (rw_rand1) < 0.4)
			rw_brick_texture_02
		#else
			rw_brick_texture_01
		#end

		rotate <360*rand(rw_rand1), 360*rand(rw_rand1), 360*rand(rw_rand1)>
		translate <100*rand(rw_rand1), 100*rand(rw_rand1), 100*rand(rw_rand1)>
		scale 0.08
#end

//
// The main macro.
//

#macro gen_ruined_wall (rw_wall_length, rw_wall_height, rw_max_bricks_z, rw_brick_height, rw_brick_width, rw_brick_length, rw_mortar_thickness, rw_mortar_depth, rw_brick_type, rw_seed_value)
	//
	// These give the profile for the wall. Try changing these
	// values to have different effects.
	//
              
	#declare rw_length_delta_bound1 = 0.1;
	#declare rw_length_delta_bound2 = 0.35;
	#declare rw_length_delta_bound3 = 0.5;
	#declare rw_missing_brick_bound = 0.5;

	//
	// Create the brick object.
	//

	#if (rw_brick_type = 0)
		#declare rw_brick_object = box
		{
			<-(rw_brick_length / 2), -(rw_brick_height / 2), -(rw_brick_width / 2)>, 
			<rw_brick_length / 2, rw_brick_height / 2, rw_brick_width / 2>
		}
	#else
		#declare rw_brick_object = superellipsoid
		{
			<0.1, 0.1>
			scale <rw_brick_length / 2, rw_brick_height / 2, rw_brick_width / 2>
		}
	#end

	//
	// The wall is a union of *a lot* of bricks and some mortar...
	//

	union
	{
		//
		// Set the random values seeds. These need to be set so we can be
		// consistent in building multidepth walls.
		//

		#local rw_rand = seed (rw_seed_value);

		#local no_bricks_rand_seed = rand (rw_rand) * 100; //50;

		#declare rw_missing_brick_rand_seed=array[rw_max_bricks_z]
		#declare rw_current_brick_z = 0;
		#while (rw_current_brick_z < rw_max_bricks_z)
			#declare rw_missing_brick_rand_seed[rw_current_brick_z] = rand (rw_rand) * 100;
			#declare rw_current_brick_z = rw_current_brick_z + 1;
		#end

		//
		// First, we have to place the bricks.
		//

		#declare rw_current_brick_z = 0;

		#while (rw_current_brick_z < rw_max_bricks_z)
			#declare no_bricks_rand = seed (no_bricks_rand_seed);
			#declare rw_max_bricks_x = rw_wall_length / (rw_brick_length + rw_mortar_thickness);
			#declare rw_max_bricks_y = rw_wall_height / (rw_brick_height + rw_mortar_thickness);
			#declare rw_current_brick_y = 0;
			#declare rw_wall_offset = 0;
			#declare rw_wall_offset_flag = 0;

			//
			// Define the depth offset for each wall
			//

			#declare brick_z_offset = rw_current_brick_z * (rw_brick_width + rw_mortar_thickness);
			#declare rw_missing_brick_rand = seed (rw_missing_brick_rand_seed[rw_current_brick_z ]);

			//
			// Add a row of bricks.
			//

			#while (rw_current_brick_y < rw_max_bricks_y)
				//
				// Define an offset for alternate rows of bricks.
				//

				#if (rw_wall_offset_flag = 0)
					#declare rw_wall_offset_flag = 1;
					#declare rw_wall_offset = 0;
				#else
					#declare rw_wall_offset_flag = 0;
					#declare rw_wall_offset = (rw_brick_length + rw_mortar_thickness) / 2;
				#end

				#if (rand (rw_missing_brick_rand) < rw_missing_brick_bound)
					#declare rw_temp_max_bricks_x = rw_max_bricks_x - 1;
				#else
					#declare rw_temp_max_bricks_x = rw_max_bricks_x;
				#end

				#declare rw_current_brick_x = 0;

				#while (rw_current_brick_x < rw_temp_max_bricks_x)
					object
					{
						rw_brick_object

						//
						// Slightly resize the brick
						//

						scale <0.99 + (rand (rw_rand1) / 50),0.99 + (rand (rw_rand1) / 50),0.99 + (rand (rw_rand1) / 50)>

						texture
						{
							get_brick_texture ()
						}

						//
						// Randomly rotate the brick.
						//

						rotate <-0.5 + rand (rw_rand1),-0.5 + rand (rw_rand1),-0.5 + rand (rw_rand1)>

						//
						// Move the brick to its correct location.
						//

						translate <rw_wall_offset + (rw_current_brick_x * (rw_brick_length + rw_mortar_thickness)), rw_current_brick_y * (rw_brick_height + rw_mortar_thickness), brick_z_offset>

						//
						// Slightly 'jiggle' the brick.
						//

						translate <-0.005 + (rand (rw_rand1) / 100),-0.005 + (rand (rw_rand1) / 100),-0.005 + (rand (rw_rand1) / 100)>
					}

					#declare rw_current_brick_x = rw_current_brick_x + 1;
				#end

				//
				// Alter the length of the next level of the wall.
				//

				#declare rw_length_delta = rand (no_bricks_rand);

				#if (rw_length_delta < rw_length_delta_bound1)
					#if (rw_wall_offset_flag = 0)
						#declare rw_max_bricks_x = rw_max_bricks_x + 1;
					#else
						#declare rw_max_bricks_x = rw_max_bricks_x;
					#end
				#else
					#if (rw_length_delta < rw_length_delta_bound2)
						#declare rw_max_bricks_x = rw_max_bricks_x - 1;
					#else
						#if (rw_length_delta < rw_length_delta_bound3)
							#declare rw_max_bricks_x = rw_max_bricks_x - 2;
						#else
							#declare rw_max_bricks_x = rw_max_bricks_x;
						#end
					#end
				#end

				#declare rw_current_brick_y = rw_current_brick_y + 1;
			#end

			#declare rw_current_brick_z = rw_current_brick_z + 1;
		#end

		//
		// OK... We have now placed all the bricks in the wall, but we now need
		// to create the mortar inbetween the bricks.
		//

		union
		{
			#declare rw_current_brick_z = 0;

			#while (rw_current_brick_z < rw_max_bricks_z)
				#declare no_bricks_rand = seed (no_bricks_rand_seed);
				#declare rw_max_bricks_y = rw_wall_height / (rw_brick_height + rw_mortar_thickness);
				#declare rw_max_bricks_x = rw_wall_length / (rw_brick_length + rw_mortar_thickness);
				#declare rw_current_brick_y = 0;
				#declare rw_wall_offset = 0;
				#declare rw_wall_offset_flag = 0;

				//
				// Define the from and to thickness of the mortar. Do the start
				// first.
				//

				#if (rw_current_brick_z = 0)
					#declare rw_mortar_from = -((rw_brick_width / 2) - rw_mortar_depth);
				#else
					#declare rw_mortar_from = (rw_current_brick_z - 0.5) * (rw_brick_width + rw_mortar_thickness);
				#end

				//
				// now do the end
				//

				#if (rw_current_brick_z = (rw_max_bricks_z - 1))
					#declare rw_mortar_to = (((rw_current_brick_z + 0.5) * rw_brick_width) + (rw_current_brick_z * rw_mortar_thickness)) - rw_mortar_depth;
				#else
					#declare rw_mortar_to = (rw_current_brick_z + 0.5) * (rw_brick_width + rw_mortar_thickness);
				#end

				#declare rw_missing_brick_rand = seed (rw_missing_brick_rand_seed[rw_current_brick_z]);

				//
				// Add a row of moratr.
				//

				#while (rw_current_brick_y < rw_max_bricks_y)

					//
					// Define an offset for alternate rows of bricks.
					//

					#if (rw_wall_offset_flag = 0)
						#declare rw_wall_offset_flag = 1;
						#declare rw_wall_offset = 0;
					#else
						#declare rw_wall_offset_flag = 0;
						#declare rw_wall_offset = (rw_brick_length + rw_mortar_thickness) / 2;
					#end

					#if (rand (rw_missing_brick_rand) < rw_missing_brick_bound)
						#declare rw_temp_max_bricks_x = rw_max_bricks_x - 1;
					#else
						#declare rw_temp_max_bricks_x = rw_max_bricks_x;
					#end

					#if (rw_temp_max_bricks_x > 0)
						#if (rw_wall_offset_flag = 1) // these bricks are not offset
							box
							{
								<0 - (rand (rw_rand1) * ((rw_brick_length / 2) + (rw_mortar_thickness / 2))), (rw_current_brick_y * (rw_brick_height + rw_mortar_thickness)) - ((rw_brick_height + rw_mortar_thickness) / 2), rw_mortar_from>,
								<((rw_max_bricks_x - 1) * (rw_brick_length + rw_mortar_thickness)) + (rand (rw_rand1) * ((rw_brick_length / 2) + (rw_mortar_thickness / 2))), (rw_current_brick_y * (rw_brick_height + rw_mortar_thickness)) + ((rw_brick_height + rw_mortar_thickness) / 2), rw_mortar_to>
							}
						#else // these are
							box
							{
								<(rw_brick_length / 2) - (rand (rw_rand1) * ((rw_brick_length / 2) + (rw_mortar_thickness / 2))), (rw_current_brick_y * (rw_brick_height + rw_mortar_thickness)) - ((rw_brick_height + rw_mortar_thickness) / 2), rw_mortar_from>,
								<(rw_max_bricks_x * (rw_brick_length + rw_mortar_thickness)) - (rw_mortar_thickness / 2) - (rand (rw_rand1) * ((rw_brick_length / 2) + (rw_mortar_thickness / 2))), (rw_current_brick_y * (rw_brick_height + rw_mortar_thickness)) + ((rw_brick_height + rw_mortar_thickness) / 2), rw_mortar_to>
							}
						#end
					#end

					//
					// Alter the length of the next level of the wall.
					//

					#declare rw_length_delta = rand (no_bricks_rand);

					#if (rw_length_delta < rw_length_delta_bound1)
						#if (rw_wall_offset_flag = 0)
							#declare rw_max_bricks_x = rw_max_bricks_x + 1;
						#else
							#declare rw_max_bricks_x = rw_max_bricks_x;
						#end
					#else
						#if (rw_length_delta < rw_length_delta_bound2)
							#declare rw_max_bricks_x = rw_max_bricks_x - 1;
						#else
							#if (rw_length_delta < rw_length_delta_bound3)
								#declare rw_max_bricks_x = rw_max_bricks_x - 2;
							#else
								#declare rw_max_bricks_x = rw_max_bricks_x;
							#end
						#end
					#end

					#declare rw_current_brick_y = rw_current_brick_y + 1;
				#end

				#declare rw_current_brick_z = rw_current_brick_z + 1;
			#end

			texture
			{
				rw_mortar_texture
			}
		} // end of mortar union

		//
		// Now translate the wall so it is centred on z axis
		//

		translate <0, 0, rw_max_bricks_z * -((rw_brick_width + rw_mortar_thickness) / 2)>
	} // end of whole union
#end // end of macro
