#version unofficial MegaPov 0.4;

#declare TS_CullPath = "culling.tmp"
#declare TS_VarBias = 3.0; //seems to work fairly well as a default...
#declare TS_FillLevel = off; //By default, no fill scanning. Adjust with
                             //the macro ScanLevel()..see readme file for
                             //details...

   
////////////////////TRISCAN SUPPORT FUNCTIONS///////////////////////////////

#macro Triscan_Setup()
/*Just a quick little something to keep POV from complaining there's nothing
  actually in the scene. If you don't have crystal.ttf, use a ttf you do have,
  or, optionally, replace the whole text object with a simple sphere or
  whatever. All the action takes place during parsing anyway..  */
text { ttf "crystal.ttf" "Scan complete" 1,0 align_center pigment { rgb<1,0,0> } finish { ambient 1 } translate<0,-.25,0> scale <1,.5,1> }
camera { orthographic location <0,0,-1000> look_at<0,0,0> up<0,1,0> right<7.5,0,0>}
#end//end setup macro

#macro ScanLevel(fvX)
   #declare TS_FillLevel = fvX;
   //but value of <= 1 makes no sense, of course...so default
   #if (fvX > 0 & fvX <= 1) #declare TS_FillLevel= 2; #end
#end

#macro IsNullVector(vecX)
/*Check a vector to see if its <0,0,0> */
     #if ( vecX.x = 0 & vecX.y = 0 & vecX.z = 0 )
		#local Result = true;
     #else #local Result = false;
     #end

     Result
#end

#macro AreEqualVectors(vecX,vecY)
/* Compare two vectors to see if they are equivalent */
     #if ( vecX.x = vecY.x & vecX.y = vecY.y & vecX.z = vecY.z )
		#local Result = true;
     #else #local Result = false;
     #end

     Result
#end

#macro ExceedsVariance(V1,V2,V3)
/* Used to determine if triangles are likely to be false due to 
   near miss trace errors */
    #local Result = no;
    #if ( vlength(V1-V2) > TS_Variance ) #local Result = yes; #end
    #if ( vlength(V2-V3) > TS_Variance ) #local Result = yes; #end
    #if ( vlength(V3-V1) > TS_Variance ) #local Result = yes; #end

    Result
#end

#macro ExceedsDifference(V1,V2)
/* Used as a threshold during overlap checks due to variance in mesh to 
   actual surface */
    #local Result = no;
    #if ( vlength(V1-V2) > TS_Variance/TS_VarBias ) #local Result = yes; #end

    Result
#end

#macro AngleBetween(V1,V2)
/* Calculates the angle between two vectors using the law of cosines -
   A good multipurpose function...
*/
    #local SA = vlength(V1);
    #local SB = vlength(V2);
    #local SC = vlength(V2-V1);
    
    #if ( SA>0 & SB>0 )
       #local T = degrees( acos ( (SA*SA+SB*SB-SC*SC)/(2*SA*SB) ));
    #else
       #local T = -1; //generic return meaning failure of function
    #end

    T
#end//end macro AngleBetween

#macro IsColinear(V1,V2,V3)
/* Calls on AngleBetween to determine if any three given vectors lie along a
   straight line */
    //make local relative to V3...
    #local V1 = V1 - V3; #local V2 = V2 - V3;
    #local X = AngleBetween(V1,V2);
    //return after value is obtained...
    #local V1 = V1 + V3; #local V2 = V2 + V3; 
    #if ( X = 0 | X = 180 )
        #local Result = yes;
    #else 
        #local Result = no;
    #end

    Result
#end //end IsColinear


#macro OppositeDirections(V1,V2,V3,Na,Nb,Nc)
/* Determine if any normals lie on opposite sides of the plane containing
   the triangle's vertices -- one of the conditions of a degenerate normal
*/
   //localize the triangle relative to V3
   #local V2 = V2 - V3; #local V1 = V1 - V3;
   #local N = vcross(V1,V2);
   /*return after cross product stuff is done.*/
   #local V2 = V2 + V3; #local V1 = V1 + V3;

   #local Xa = AngleBetween(Na,N); 
      #if (Xa<90) #local Xs = -1; #end
      #if (Xa=90) #local Xs = 0; #end
      #if (Xa>90) #local Xs = 1; #end
   #local Ya = AngleBetween(Nb,N);
      #if (Ya<90) #local Ys = -1; #end
      #if (Ya=90) #local Ys = 0; #end
      #if (Ya>90) #local Ys = 1; #end
   #local Za = AngleBetween(Nc,N);
      #if (Za<90) #local Zs = -1; #end
      #if (Za=90) #local Zs = 0; #end
      #if (Za>90) #local Zs = 1; #end

   #if ( Xa != 90 & Ya != 90 & Za != 90 & Xs = Ys & Ys = Zs )
       #local Result = no;
   #else
       #local Result = yes;
   #end

   Result
#end //end macro OppositeDirections

#macro OverlappedTriangle(At,Bt,Ct,Au,Bu,Cu,)
/* Determines if a triangle found during a scan other than the first
   lies completely within a previously scanned region... helps to cull
   some unnecessary duplication
*/
    #local tmpNormal = <0,0,0>; 
    #local MidPoint = ((Au+Cu)/2 + Bu)/2;
    #local AuIP = trace(Primescan,Au,TS_ScanDir,tmpNormal);
    #local BuIP = trace(Primescan,Bu,TS_ScanDir,tmpNormal);
    #local CuIP = trace(Primescan,Cu,TS_ScanDir,tmpNormal);
    #local MPIP1 = trace(objTarget,MidPoint,TS_ScanDir,tmpNormal);
    #local MPIP2 = trace(Primescan,MidPoint,TS_ScanDir,tmpNormal);

    #if ( !ExceedsDifference(At,AuIP) & 
          !ExceedsDifference(Bt,BuIP) &
          !ExceedsDifference(Ct,CuIP) &
          !IsNullVector(tmpNormal) &
          !ExceedsDifference(MPIP1,MPIP2)
         )
       #local Result = yes;
    #else #local Result = no;
    #end //if

    Result 
#end


//////////////// TRISCAN SCANNING OPERATIONS /////////////////////////////

#macro ForeScan(DensityMax,Density,TrackDone)
//prep for forward side scan...
   #local tmpNormal = <0,0,0>; 
   #local IP_A = array[DensityMax+1]
   #local IP_B = array[DensityMax+1]
   #local Normal_A = array[DensityMax+1]
   #local Normal_B = array[DensityMax+1]
   #local vecA = array[DensityMax+1] 
   #local vecB = array[DensityMax+1] 
   #local BoxHeight = max_extent(objTarget).y - min_extent(objTarget).y;
   #local BoxWidth = max_extent(objTarget).x - min_extent(objTarget).x;
   #local SegmentWidth = BoxWidth / Density.u;
   #local SegmentHeight = BoxHeight / Density.v; 
   #local TS_Variance = SegmentWidth - SegmentHeight;
   #if(TS_Variance<0) #declare TS_Variance = SegmentHeight*TS_VarBias;
   #else #declare TS_Variance = SegmentWidth*TS_VarBias; #end

//Primary scan loops...
   #declare TS_ScanDir = z;
   #declare TS_uLoop = 0; #declare TS_vLoop= 0; 
   #while ( TS_uLoop < Density.u  )
//First, fill the IP and Normal arrays...
      #while ( TS_vLoop < Density.v )
         #local vecA[TS_vLoop] = < TS_uLoop*SegmentWidth+min_extent(objTarget).x ,
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                  min_extent(objTarget).z -.1 >;
         #local vecB[TS_vLoop] = < (TS_uLoop+1)*SegmentWidth+min_extent(objTarget).x ,
                                   TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                   min_extent(objTarget).z -.1 >;
         #local IP_A[TS_vLoop] = trace(objTarget,vecA[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_A[TS_vLoop] = tmpNormal; 
         #local IP_B[TS_vLoop] = trace(objTarget,vecB[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_B[TS_vLoop] = tmpNormal; 
         #declare TS_vLoop = TS_vLoop + 1;
      #end //TS_vLoop

      #declare TS_vLoop = 0;
//Then resolve potential triangles based on the results
      #while ( TS_vLoop < Density.v-1 )
         #switch(TS_Recode)
            #case(0) //not a previous IP...
               Evaluate( IP_A[TS_vLoop], IP_B[TS_vLoop], 
                         IP_A[TS_vLoop+1],IP_B[TS_vLoop+1],
                         Normal_A[TS_vLoop], Normal_B[TS_vLoop],
                         Normal_A[TS_vLoop+1], Normal_B[TS_vLoop+1],
                         vecA[TS_vLoop], vecB[TS_vLoop],
                         vecA[TS_vLoop+1], vecB[TS_vLoop+1])
               #break
             #case(1)// recurrent testing... 
               #local IPRecodeA = trace(objTarget,TS_RecodeA,TS_ScanDir,tmpNormal);
               #local NormalRecodeA = tmpNormal;
               #local IPRecodeB = trace(objTarget,TS_RecodeB,TS_ScanDir,tmpNormal);
               #local NormalRecodeB = tmpNormal;
               #local IPRecodeC = trace(objTarget,TS_RecodeC,TS_ScanDir,tmpNormal);
               #local NormalRecodeC = tmpNormal;
               #local IPRecodeD = trace(objTarget,TS_RecodeD,TS_ScanDir,tmpNormal);
               #local NormalRecodeD = tmpNormal;

               Evaluate(IPRecodeA, IPRecodeB, IPRecodeC, IPRecodeD,
                        NormalRecodeA, NormalRecodeB, NormalRecodeC, NormalRecodeD,
                        TS_RecodeA, TS_RecodeB, TS_RecodeC, TS_RecodeD)
         #end //end Switch Recode

         #if (TS_Recode = 0) #declare TS_vLoop = TS_vLoop + 1; #end
      #end //TS_vLoop

       #declare TS_uLoop = TS_uLoop + 1; #declare TS_vLoop = 0;
   #end //TS_uLoop
#end //end Forescan


#macro BackScan(DensityMax,Density,TrackDone)
//prep for back face scan...
   #local IP_A = array[DensityMax+1]
   #local IP_B = array[DensityMax+1]
   #local Normal_A = array[DensityMax+1]
   #local Normal_B = array[DensityMax+1]
   #local vecA = array[DensityMax+1] 
   #local vecB = array[DensityMax+1] 
   #local tmpNormal = <0,0,0>; 
   #local BoxHeight = max_extent(objTarget).y - min_extent(objTarget).y;
   #local BoxWidth = max_extent(objTarget).x - min_extent(objTarget).x;
   #local SegmentWidth = BoxWidth / Density.u;
   #local SegmentHeight = BoxHeight / Density.v; 
   #declare TS_Variance = SegmentWidth - SegmentHeight;
   #if(TS_Variance<0) #declare TS_Variance = SegmentHeight*TS_VarBias;
      #else #declare TS_Variance = SegmentWidth*TS_VarBias; #end
   #fclose TS_CullFile
   #declare Primescan=  #include "culling.tmp" }
   #fopen TS_CullFile TS_CullPath append
   #declare TS_ScanDir = -z;
//Primary scan loops...
   #declare TS_uLoop = 0; #declare TS_vLoop= 0; 
   #while ( TS_uLoop < Density.u  )
//First, fill the IP and Normal arrays...
      #while ( TS_vLoop < Density.v )
         #local vecA[TS_vLoop] = < max_extent(objTarget).x-TS_uLoop*SegmentWidth ,
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                  max_extent(objTarget).z +.1 >;
         #local vecB[TS_vLoop] = < max_extent(objTarget).x-(TS_uLoop-1)*SegmentWidth ,
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                  max_extent(objTarget).z +.1>;
         #local IP_A[TS_vLoop] = trace(objTarget,vecA[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_A[TS_vLoop] = tmpNormal; 
         #local IP_B[TS_vLoop] = trace(objTarget,vecB[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_B[TS_vLoop] = tmpNormal; 
         #declare TS_vLoop = TS_vLoop + 1;
      #end //TS_vLoop

      #declare TS_vLoop = 0;
//Then resolve potential triangles based on the results
      #while ( TS_vLoop < Density.v-1 )
         #switch(TS_Recode)
         #case(0) //not a previous point...
            Evaluate( IP_A[TS_vLoop], IP_B[TS_vLoop], 
                      IP_A[TS_vLoop+1],IP_B[TS_vLoop+1],
                      Normal_A[TS_vLoop], Normal_B[TS_vLoop],
                      Normal_A[TS_vLoop+1], Normal_B[TS_vLoop+1],
                      vecA[TS_vLoop], vecB[TS_vLoop],
                      vecA[TS_vLoop+1], vecB[TS_vLoop+1])
            #break
         #case(1)// recurrent testing... 
           
            #local IPRecodeA = trace(objTarget,TS_RecodeA,TS_ScanDir,tmpNormal);
            #local NormalRecodeA = tmpNormal;
            #local IPRecodeB = trace(objTarget,TS_RecodeB,TS_ScanDir,tmpNormal);
            #local NormalRecodeB = tmpNormal;
            #local IPRecodeC = trace(objTarget,TS_RecodeC,TS_ScanDir,tmpNormal);
            #local NormalRecodeC = tmpNormal;
            #local IPRecodeD = trace(objTarget,TS_RecodeD,TS_ScanDir,tmpNormal);
            #local NormalRecodeD = tmpNormal;
            Evaluate(IPRecodeA, IPRecodeB, IPRecodeC, IPRecodeD,
                     NormalRecodeA, NormalRecodeB, NormalRecodeC, NormalRecodeD,
                     TS_RecodeA, TS_RecodeB, TS_RecodeC, TS_RecodeD)
         #end //end Switch TS_Recode


         #if (TS_Recode = 0) #declare TS_vLoop = TS_vLoop + 1; #end
     #end //TS_vLoop

     #declare TS_uLoop = TS_uLoop + 1; #declare TS_vLoop = 0;
   #end //TS_uLoop
#end//macro BackScan


#macro LeftScan(DensityMax,Density,TrackDone)
//prep for left side scan...
   #local IP_A = array[DensityMax+1]
   #local IP_B = array[DensityMax+1]
   #local Normal_A = array[DensityMax+1]
   #local Normal_B = array[DensityMax+1]
   #local vecA = array[DensityMax+1] 
   #local vecB = array[DensityMax+1] 
   #local tmpNormal = <0,0,0>; 
   #local BoxHeight = max_extent(objTarget).y - min_extent(objTarget).y;
   #local BoxWidth = max_extent(objTarget).z - min_extent(objTarget).z;
   #local SegmentWidth = BoxWidth / Density.u;
   #local SegmentHeight = BoxHeight / Density.v; 
   #declare TS_Variance = SegmentWidth - SegmentHeight;
   #if(TS_Variance<0) 
      #declare TS_Variance = SegmentHeight*TS_VarBias;
   #else 
      #declare TS_Variance = SegmentWidth*TS_VarBias; 
   #end//if
   #declare NextPass = 1;
   #fclose TS_CullFile
   #declare Primescan=  #include "culling.tmp" }
   #fopen TS_CullFile TS_CullPath append
//Primary scan loops...
   #declare TS_ScanDir = x;
   #declare TS_uLoop = 0; #declare TS_vLoop= 0; 
   #while ( TS_uLoop < Density.u  )
//First, fill the IP and Normal arrays...
      #while ( TS_vLoop < Density.v )
         #local vecA[TS_vLoop] = < min_extent(objTarget).x - .1 ,
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                  max_extent(objTarget).z-TS_uLoop*SegmentWidth>;
         #local vecB[TS_vLoop] = < min_extent(objTarget).x - .1 ,
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                  max_extent(objTarget).z-(TS_uLoop+1)*SegmentWidth>;
         #local IP_A[TS_vLoop] = trace(objTarget,vecA[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_A[TS_vLoop] = tmpNormal; 
         #local IP_B[TS_vLoop] = trace(objTarget,vecB[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_B[TS_vLoop] = tmpNormal; 
         #declare TS_vLoop = TS_vLoop + 1;
     #end //TS_vLoop
     #declare TS_vLoop = 0;
//Then resolve potential triangles based on the results
     #while ( TS_vLoop < Density.v-1 )
        #switch(TS_Recode)
           #case(0) //not a previous point...
              Evaluate( IP_A[TS_vLoop], IP_B[TS_vLoop], 
                        IP_A[TS_vLoop+1],IP_B[TS_vLoop+1],
                        Normal_A[TS_vLoop], Normal_B[TS_vLoop],
                        Normal_A[TS_vLoop+1], Normal_B[TS_vLoop+1],
                        vecA[TS_vLoop], vecB[TS_vLoop],
                        vecA[TS_vLoop+1], vecB[TS_vLoop+1])
              #break
           #case(1)// recurrent testing... 
           #local IPRecodeA = trace(objTarget,TS_RecodeA,TS_ScanDir,tmpNormal);
           #local NormalRecodeA = tmpNormal;
           #local IPRecodeB = trace(objTarget,TS_RecodeB,TS_ScanDir,tmpNormal);
           #local NormalRecodeB = tmpNormal;
           #local IPRecodeC = trace(objTarget,TS_RecodeC,TS_ScanDir,tmpNormal);
           #local NormalRecodeC = tmpNormal;
           #local IPRecodeD = trace(objTarget,TS_RecodeD,TS_ScanDir,tmpNormal);
           #local NormalRecodeD = tmpNormal;
           Evaluate(IPRecodeA, IPRecodeB, IPRecodeC, IPRecodeD,
                    NormalRecodeA, NormalRecodeB, NormalRecodeC, NormalRecodeD,
                    TS_RecodeA, TS_RecodeB, TS_RecodeC, TS_RecodeD)
       #end //end Switch TS_Recode
       #if (TS_Recode = 0) #declare TS_vLoop = TS_vLoop + 1; #end
       #end //TS_vLoop
       #declare TS_uLoop = TS_uLoop + 1; #declare TS_vLoop = 0;
     #end //TS_uLoop
#end //end LeftScan


#macro RightScan(DensityMax,Density,TrackDone)
//prep for right side scan...
   #local IP_A = array[DensityMax+1]
   #local IP_B = array[DensityMax+1]
   #local Normal_A = array[DensityMax+1]
   #local Normal_B = array[DensityMax+1]
   #local vecA = array[DensityMax+1] 
   #local vecB = array[DensityMax+1] 
   #local tmpNormal = <0,0,0>; 
   #local BoxHeight = max_extent(objTarget).y - min_extent(objTarget).y;
   #local BoxWidth = max_extent(objTarget).z - min_extent(objTarget).z;
   #local SegmentWidth = BoxWidth / Density.u;
   #local SegmentHeight = BoxHeight / Density.v; 
   #declare TS_Variance = SegmentWidth - SegmentHeight;
   #if(TS_Variance<0) 
      #declare TS_Variance = SegmentHeight*TS_VarBias;
   #else 
      #declare TS_Variance = SegmentWidth*TS_VarBias; 
   #end//if
   #fclose TS_CullFile
   #declare Primescan=  #include "culling.tmp" }
   #fopen TS_CullFile TS_CullPath append
//Primary scan loops...
   #declare TS_ScanDir = -x;
   #declare TS_uLoop = 0; #declare TS_vLoop= 0; 
   #while ( TS_uLoop < Density.u )
//First, fill the IP and Normal arrays...
      #while ( TS_vLoop < Density.v )
          #local vecA[TS_vLoop] = < max_extent(objTarget).x + .1 ,
                                   TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                   max_extent(objTarget).z-TS_uLoop*SegmentWidth>;
          #local vecB[TS_vLoop] = < max_extent(objTarget).x + .1 ,
                                   TS_vLoop*SegmentHeight+min_extent(objTarget).y ,
                                   max_extent(objTarget).z-(TS_uLoop-1)*SegmentWidth>;
          #local IP_A[TS_vLoop] = trace(objTarget,vecA[TS_vLoop],TS_ScanDir,tmpNormal);
          #local Normal_A[TS_vLoop] = tmpNormal; 
          #local IP_B[TS_vLoop] = trace(objTarget,vecB[TS_vLoop],TS_ScanDir,tmpNormal);
          #local Normal_B[TS_vLoop] = tmpNormal; 
          #declare TS_vLoop = TS_vLoop + 1;
      #end //TS_vLoop
      #declare TS_vLoop = 0;
//Then resolve potential triangles based on the results
      #while ( TS_vLoop < Density.v-1 )
         #switch(TS_Recode)
            #case(0) //not a previous point...
               Evaluate( IP_A[TS_vLoop], IP_B[TS_vLoop], 
                         IP_A[TS_vLoop+1],IP_B[TS_vLoop+1],
                         Normal_A[TS_vLoop], Normal_B[TS_vLoop],
                         Normal_A[TS_vLoop+1], Normal_B[TS_vLoop+1],
                         vecA[TS_vLoop], vecB[TS_vLoop],
                         vecA[TS_vLoop+1], vecB[TS_vLoop+1])
               #break
            #case(1)// recurrent testing... 
               #local IPRecodeA = trace(objTarget,TS_RecodeA,TS_ScanDir,tmpNormal);
               #local NormalRecodeA = tmpNormal;
               #local IPRecodeB = trace(objTarget,TS_RecodeB,TS_ScanDir,tmpNormal);
               #local NormalRecodeB = tmpNormal;
               #local IPRecodeC = trace(objTarget,TS_RecodeC,TS_ScanDir,tmpNormal);
               #local NormalRecodeC = tmpNormal;
               #local IPRecodeD = trace(objTarget,TS_RecodeD,TS_ScanDir,tmpNormal);
               #local NormalRecodeD = tmpNormal;
               Evaluate(IPRecodeA, IPRecodeB, IPRecodeC, IPRecodeD,
                        NormalRecodeA, NormalRecodeB, NormalRecodeC, NormalRecodeD,
                        TS_RecodeA, TS_RecodeB, TS_RecodeC, TS_RecodeD)
         #end //end Switch TS_Recode
         #if (TS_Recode = 0) #declare TS_vLoop = TS_vLoop + 1; #end
      #end //TS_vLoop

      #declare TS_uLoop = TS_uLoop + 1; #declare TS_vLoop = 0;
   #end //TS_uLoop
#end //end RightScan


#macro TopScan(DensityMax,Density,TrackDone)
//prep for top side scan...
   #local IP_A = array[DensityMax+1]
   #local IP_B = array[DensityMax+1]
   #local Normal_A = array[DensityMax+1]
   #local Normal_B = array[DensityMax+1]
   #local vecA = array[DensityMax+1] 
   #local vecB = array[DensityMax+1] 
   #local tmpNormal = <0,0,0>; 
   #local BoxHeight = max_extent(objTarget).z - min_extent(objTarget).z;
   #local BoxWidth = max_extent(objTarget).x - min_extent(objTarget).x;
   #local SegmentWidth = BoxWidth / Density.u;
   #local SegmentHeight = BoxHeight / Density.v; 
   #declare TS_Variance = SegmentWidth - SegmentHeight;
   #if(TS_Variance<0) 
      #declare TS_Variance = SegmentHeight*TS_VarBias;
   #else  
      #declare TS_Variance = SegmentWidth*TS_VarBias; 
   #end//if
   #fclose TS_CullFile
   #declare Primescan=  #include "culling.tmp" }
   #fopen TS_CullFile TS_CullPath append
//Primary scan loops...
   #declare TS_ScanDir = -y;
   #declare TS_uLoop = 0; #declare TS_vLoop= 0; 
   #while ( TS_uLoop < Density.u  )
//First, fill the IP and Normal arrays...
      #while ( TS_vLoop < Density.v )
         #local vecA[TS_vLoop] = <TS_uLoop*SegmentWidth+min_extent(objTarget).x ,
                                 max_extent(objTarget).y + .1 , 
                                 TS_vLoop*SegmentHeight+min_extent(objTarget).z>;
         #local vecB[TS_vLoop] = <(TS_uLoop+1)*SegmentWidth+min_extent(objTarget).x ,
                                  max_extent(objTarget).y + .1 ,
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).z>;
         #local IP_A[TS_vLoop] = trace(objTarget,vecA[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_A[TS_vLoop] = tmpNormal; 
         #local IP_B[TS_vLoop] = trace(objTarget,vecB[TS_vLoop],TS_ScanDir,tmpNormal);
         #local Normal_B[TS_vLoop] = tmpNormal; 
         #declare TS_vLoop = TS_vLoop + 1;
      #end //TS_vLoop
      #declare TS_vLoop = 0;
//Then resolve potential triangles based on the results
      #while ( TS_vLoop < Density.v-1 )
         #switch(TS_Recode)
            #case(0) //not a previous point...
               Evaluate( IP_A[TS_vLoop], IP_B[TS_vLoop], 
                         IP_A[TS_vLoop+1],IP_B[TS_vLoop+1],
                         Normal_A[TS_vLoop], Normal_B[TS_vLoop],
                         Normal_A[TS_vLoop+1], Normal_B[TS_vLoop+1],
                         vecA[TS_vLoop], vecB[TS_vLoop],
                         vecA[TS_vLoop+1], vecB[TS_vLoop+1])
               #break
            #case(1)// recurrent testing... 
            #local IPRecodeA = trace(objTarget,TS_RecodeA,TS_ScanDir,tmpNormal);
            #local NormalRecodeA = tmpNormal;
            #local IPRecodeB = trace(objTarget,TS_RecodeB,TS_ScanDir,tmpNormal);
            #local NormalRecodeB = tmpNormal;
            #local IPRecodeC = trace(objTarget,TS_RecodeC,TS_ScanDir,tmpNormal);
            #local NormalRecodeC = tmpNormal;
            #local IPRecodeD = trace(objTarget,TS_RecodeD,TS_ScanDir,tmpNormal);
            #local NormalRecodeD = tmpNormal;
            Evaluate(IPRecodeA, IPRecodeB, IPRecodeC, IPRecodeD,
                     NormalRecodeA, NormalRecodeB, NormalRecodeC, NormalRecodeD,
                     TS_RecodeA, TS_RecodeB, TS_RecodeC, TS_RecodeD)
         #end //end Switch TS_Recode
         #if (TS_Recode = 0) #declare TS_vLoop = TS_vLoop + 1; #end
     #end //TS_vLoop
     #declare TS_uLoop = TS_uLoop + 1; #declare TS_vLoop = 0;
   #end //TS_uLoop
#end //end Topscan


#macro BottomScan(DensityMax,Density,TrackDone)
//prep for bottom face scan...
   #local IP_A = array[DensityMax+1]
   #local IP_B = array[DensityMax+1]
   #local Normal_A = array[DensityMax+1]
   #local Normal_B = array[DensityMax+1]
   #local vecA = array[DensityMax+1] 
   #local vecB = array[DensityMax+1] 
   #local tmpNormal = <0,0,0>; 
   #local BoxHeight = max_extent(objTarget).z - min_extent(objTarget).z;
   #local BoxWidth = max_extent(objTarget).x - min_extent(objTarget).x;
   #local SegmentWidth = BoxWidth / Density.u;
   #local SegmentHeight = BoxHeight / Density.v; 
   #declare TS_Variance = SegmentWidth - SegmentHeight;
   #if(TS_Variance<0) 
      #declare TS_Variance = SegmentHeight*TS_VarBias;
   #else 
      #declare TS_Variance = SegmentWidth*TS_VarBias; 
   #end//if
   #fclose TS_CullFile
   #declare Primescan=  #include "culling.tmp" }
   #fopen TS_CullFile TS_CullPath append
//primary scan loops
   #declare TS_ScanDir = y;
   #declare TS_uLoop = 0; #declare TS_vLoop= 0; 
   #while ( TS_uLoop < Density.u )
//First, fill the IP and Normal arrays...
      #while ( TS_vLoop < Density.v )
          #local vecA[TS_vLoop] = <max_extent(objTarget).x-TS_uLoop*SegmentWidth,
                                  min_extent(objTarget).y - .1 , 
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).z>;
          #local vecB[TS_vLoop] = <max_extent(objTarget).x-(TS_uLoop-1)*SegmentWidth,
                                  min_extent(objTarget).y - .1 ,
                                  TS_vLoop*SegmentHeight+min_extent(objTarget).z>;
          #local IP_A[TS_vLoop] = trace(objTarget,vecA[TS_vLoop],TS_ScanDir,tmpNormal);
          #local Normal_A[TS_vLoop] = tmpNormal; 
          #local IP_B[TS_vLoop] = trace(objTarget,vecB[TS_vLoop],TS_ScanDir,tmpNormal);
          #local Normal_B[TS_vLoop] = tmpNormal; 
          #declare TS_vLoop = TS_vLoop + 1;
      #end //TS_vLoop
      #declare TS_vLoop = 0;
//Then resolve potential triangles based on the results
      #while ( TS_vLoop < Density.v-1 )
         #switch(TS_Recode)
            #case(0) //not a previous point...
               Evaluate( IP_A[TS_vLoop], IP_B[TS_vLoop], 
                         IP_A[TS_vLoop+1],IP_B[TS_vLoop+1],
                         Normal_A[TS_vLoop], Normal_B[TS_vLoop],
                         Normal_A[TS_vLoop+1], Normal_B[TS_vLoop+1],
                         vecA[TS_vLoop], vecB[TS_vLoop],
                         vecA[TS_vLoop+1], vecB[TS_vLoop+1])
               #break
         #case(1)// recurrent testing... 
               #local IPRecodeA = trace(objTarget,TS_RecodeA,TS_ScanDir,tmpNormal);
               #local NormalRecodeA = tmpNormal;
               #local IPRecodeB = trace(objTarget,TS_RecodeB,TS_ScanDir,tmpNormal);
               #local NormalRecodeB = tmpNormal;
               #local IPRecodeC = trace(objTarget,TS_RecodeC,TS_ScanDir,tmpNormal);
               #local NormalRecodeC = tmpNormal;
               #local IPRecodeD = trace(objTarget,TS_RecodeD,TS_ScanDir,tmpNormal);
               #local NormalRecodeD = tmpNormal;
               Evaluate(IPRecodeA, IPRecodeB, IPRecodeC, IPRecodeD,
                        NormalRecodeA, NormalRecodeB, NormalRecodeC, NormalRecodeD,
                        TS_RecodeA, TS_RecodeB, TS_RecodeC, TS_RecodeD)
         #end //end Switch Recode

         #if (TS_Recode = 0) #declare TS_vLoop = TS_vLoop + 1; #end
         #end //TS_vLoop

         #declare TS_uLoop = TS_uLoop +1; #declare TS_vLoop = 0;
    #end //TS_uLoop
#end //end Bottomscan

/////////////////// TRISCAN OUTPUT CONTROL OPERATIONS ////////////////////

#macro OpenFile(FileName,ObjectName,Type)
         #fopen TS_CullFile TS_CullPath write
         #write (TS_CullFile,"union {\n")
         #switch(Type)
            #case(0) //RAW output setup...
                #fopen TS_OutFile FileName write
                #write (TS_OutFile,ObjectName,"\n")
                #break
            #case(1) //LSL output setup....
                #fopen TS_OutFile FileName write
                #write (TS_OutFile,"// ",ObjectName,"\n")
                #break
         #end//switch
#end//end OpenFile

#macro CloseFile()
   #fclose TS_CullFile
   #fopen TS_CullFile TS_CullPath write
   #write(TS_CullFile,"Scan is complete, this temp file can be deleted if desired.")
   #fclose TS_CullFile
   #fclose TS_OutFile
#end//end Closefile
      
#macro DumpRAWTriangle(V1,V2,V3,Na,Nb,Nc)
   //localize the triangle relative to C
   #local V2 = V2 - V3; #local V1 = V1 - V3;
   #local N = vcross(V1,V2);
   //then put it back when the cross product stuff is done...
   #local V1 = V1 + V3; #local V2 = V2 + V3;

   #if (AngleBetween(N,Nc)>90)
      #write( TS_OutFile,
              str(V3.x,7,7), " ", str(V3.y,7,7), " ", str(V3.z,7,7), "  ",
              str(V2.x,7,7), " ", str(V2.y,7,7), " ", str(V2.z,7,7), "  ",
              str(V1.x,7,7), " ", str(V1.y,7,7), " ", str(V1.z,7,7), "   ",
              str(Nc.x,7,7)," ", str(Nc.y,7,7)," ", str(Nc.z,7,7),"  ",
              str(Nb.x,7,7)," ", str(Nb.y,7,7)," ", str(Nb.z,7,7),"  ",
              str(Na.x,7,7)," ", str(Na.y,7,7)," ", str(Na.z,7,7),"\n")
   #else
      #write( TS_OutFile,
              str(V1.x,7,7), " ", str(V1.y,7,7), " ", str(V1.z,7,7), "  ",
              str(V2.x,7,7), " ", str(V2.y,7,7), " ", str(V2.z,7,7), "  ",
              str(V3.x,7,7), " ", str(V3.y,7,7), " ", str(V3.z,7,7), "   ",
              str(Na.x,7,7)," ", str(Na.y,7,7)," ", str(Na.z,7,7),"  ",
              str(Nb.x,7,7)," ", str(Nb.y,7,7)," ", str(Nb.z,7,7),"  ",
              str(Nc.x,7,7)," ", str(Nc.y,7,7)," ", str(Nc.z,7,7),"\n")

   #end//end Output
#end//end macro DumpRAWTriangle...

#macro DumpLSLTriangle(V1,V2,V3)
   #write ( TS_OutFile,
            "triangle { color 0,0,1 p1 ",
            str(V1.x,7,7), ",", str(V1.y,7,7), ",", str(V1.z,7,7), " p2 ",
            str(V2.x,7,7), ",", str(V2.y,7,7), ",", str(V2.z,7,7), " p3 ",
            str(V3.x,7,7), ",", str(V3.y,7,7), ",", str(V3.z,7,7), " }\n"
          )
#end

////////////// POTENTIAL TRIANGLE EVALUATION ////////////////////////


#macro Evaluate (A,B,C,D,An,Bn,Cn,Dn,W,X,Y,Z)
/* Evaluates whether any triangles occured, given four corner points and their
   normals, outputs if required. Note, this macro may be called recursively
   where necessary
*/
//INITIALIZE...
   #local TriASSERT1 = true; #local TriASSERT2 = true; 


//FALSE HIT/DEGENERATE TRIANGLE CHECKS...
   #if ( ExceedsVariance(A,B,C) )
       #local TriASSERT1 = false; 
   #end //variance test for Triangle 1

   #if ( AreEqualVectors(A,B) | AreEqualVectors(B,C) | AreEqualVectors(A,C) | 
         IsColinear(A,B,C) )
       #local TriASSERT1 = false; 
   #end //degenerate vertex test, triangle 1

   #if ( AngleBetween(An,Bn) >= 90 | AngleBetween(Bn,Cn) >= 90 | 
         AngleBetween(An,Cn) >= 90 | OppositeDirections(A,B,C,An,Bn,Cn)  ) 
      #local TriASSERT1 = false; 
   #end //degenerate normal test, triangle 1

   #if ( IsNullVector(An) | IsNullVector(Bn) | IsNullVector(Cn) )
       #local TriASSERT1 = false;
   #end //failed hits, triangle 1

   #if ( ExceedsVariance(B,C,D))
       #local TriASSERT2 = false;
   #end //variance test, Triangle 2

   #if ( AreEqualVectors(B,C) | AreEqualVectors(C,D) | AreEqualVectors(B,D) |
         IsColinear(B,C,D) )
       #local TriASSERT2 = false;
   #end //degenerate vertex test, triangle 2

   #if ( AngleBetween(Bn,Cn) >= 90 | AngleBetween(Cn,Dn) >= 90 | 
         AngleBetween(Bn,Dn) >= 90 | OppositeDirections(B,C,D,Bn,Cn,Dn)  )
       #local TriASSERT2 = false;
   #end //degenerate normal test, triangle 2

   #if ( IsNullVector(Bn) | IsNullVector(Cn) | IsNullVector(Dn) )
      #local TriASSERT2 = false; 
   #end //failed hits, triangle 2

   #if ( TracksDone > 0 )
      #if ( OverlappedTriangle(A,B,C,W,X,Y) )
         #local TriASSERT1 = false;
      #end //test 1
      #if ( OverlappedTriangle(B,C,D,X,Y,Z) )
         #local TriASSERT2 = false;
      #end //test2
   #end //culling anything that doesn't add something new...

//CALCULATE WHETHER THERE'S A POINT IN RECURRENT SEARCH...
   #declare TS_Recode = false; //initially...
   #if (TriASSERT1)
      #declare TS_Recode = true; 
      #declare TS_RecodeA = A; #declare TS_RecodeB = B; #declare TS_RecodeC = C;
   #else 
      #declare TS_RecodeA = W; #declare TS_RecodeB = X; #declare TS_RecodeC = Y;
   #end//if

   #if (TriASSERT2)
       #declare TS_Recode = true; 
       #declare TS_RecodeB = B; #declare TS_RecodeC = C; #declare TS_RecodeD = D;
   #else 
       #declare TS_RecodeB = X; #declare TS_RecodeC = Y; #declare TS_RecodeD = Z;
   #end//if

//STATISTICS
#local NumberTriangles = TriASSERT1 + TriASSERT2;
#declare TS_TotalTriangles = TS_TotalTriangles + NumberTriangles;
#local Percent = (TS_vLoop+TS_uLoop*Density.v+TracksDone)*100/TS_ScanTotal;
#debug concat("Percent complete:",str(Percent,4,0),"% ",
              " :: Total Triangles found... ",str(TS_TotalTriangles,5,0),"\r")


//OUTPUT IF VALID TRIANGLES
   #if ( TriASSERT1 )
      #write (TS_CullFile,"smooth_triangle { ", A, " , ", An, ",", B, ",", Bn,
                       ",", C, ",", Cn, "}\n")
      #switch(Type)
         #case(0)
            DumpRAWTriangle(A,B,C,An,Bn,Cn)
            #break
         #case(1)
            DumpLSLTriangle(A,B,C)
           #break
      #end//switch
   #end //end TriASSERT1 check

   #if ( TriASSERT2 )
      #write (TS_CullFile,"smooth_triangle { ", B, " , ", Bn, ",", C, ",", Cn,
                       ",", D, ",", Dn, "}\n")
      #switch(Type)
          #case(0)
             DumpRAWTriangle(B,C,D,Bn,Cn,Dn)
             #break
          #case(1)
             DumpLSLTriangle(B,C,D)
             #break
          #end//switch
   #end//end TriASSERT2 check

#end// macro Evaluate

//////////////////// THE MAIN SCAN LOOP //////////////////////////////

#macro Triscan(objTarget,ObjectName,FileName,DensityMult)

  Triscan_Setup() 

// Basic variables needed through remainder of macro...
   #local DX = (max_extent(objTarget).x - min_extent(objTarget).x)*DensityMult;
   #local DY = (max_extent(objTarget).y - min_extent(objTarget).y)*DensityMult;
   #local DZ = (max_extent(objTarget).z - min_extent(objTarget).z)*DensityMult;
   #local Density3 = <DX,DY,DZ>;
   #local DensityMax = DX;
     #if(DensityMax < DY) #local DensityMax = DY; #end
     #if(DensityMax < DZ) #local DensityMax = DZ; #end
   #local TypeString = substr(FileName,strlen(FileName)-3,4)
   #if (strcmp(strupr(TypeString),".LSL")=0) 
      #local Type = 1;
   #else 
      #local Type = 0;
   #end

   #declare TS_Recode = 0;
   #declare TS_TotalTriangles = 0;
   #declare TS_ScanTotal = Density3.x*Density3.y*2 + 
                           Density3.y*Density3.z*2 +
                           Density3.x*Density3.z*2;



//Prepare the output file...

   OpenFile(FileName,ObjectName,Type)
   #debug concat("\n\nTriscan of ",ObjectName," in progress\n\n\n")

   #local Density = <Density3.x,Density3.y>;
   #local TracksDone = 0;
   ForeScan(DensityMax,Density,TracksDone)

   #local Density = <Density3.z,Density3.y>;
   #local TracksDone = Density3.x*Density3.y;
   LeftScan(DensityMax,Density,TracksDone)

   #local Density = <Density3.x,Density3.z>;
   #local TracksDone = TracksDone + Density3.y*Density3.z;
   TopScan(DensityMax,Density,TracksDone)

   #local Density = <Density3.x,Density3.y>;
   #local TracksDone = TracksDone + Density3.x*Density3.z;
   BackScan(DensityMax,Density,TracksDone)

   #local Density = <Density3.z,Density3.y>;
   #local TracksDone = TracksDone + Density3.x*Density3.y;
   RightScan(DensityMax,Density,TracksDone)

   #local Density = <Density3.x,Density3.z>;
   #local TracksDone = TracksDone + Density3.y*Density3.z;
   BottomScan(DensityMax,Density,TracksDone)

//fill scans, if desired...
   #if (TS_FillLevel)
      #local Density3 = Density3*TS_FillLevel;
      #local DensityMax = DensityMax*TS_FillLevel;
      #declare TS_ScanTotal = Density3.x*Density3.y*2 + 
                              Density3.y*Density3.z*2 +
                              Density3.x*Density3.z*2;
      #fclose TS_CullFile
      #declare Primescan=  #include "culling.tmp" }
      #fopen TS_CullFile TS_CullPath append
      #debug concat("\n\nFill scans progress...\n\n\n")

      #local Density = <Density3.x,Density3.y>;
      #local TracksDone = 0;
      ForeScan(DensityMax,Density,TracksDone)

      #local Density = <Density3.z,Density3.y>;
      #local TracksDone = Density3.x*Density3.y;
      LeftScan(DensityMax,Density,TracksDone)

      #local Density = <Density3.x,Density3.z>;
      #local TracksDone = TracksDone + Density3.y*Density3.z;
      TopScan(DensityMax,Density,TracksDone)

      #local Density = <Density3.x,Density3.y>;
      #local TracksDone = TracksDone + Density3.x*Density3.z;
      BackScan(DensityMax,Density,TracksDone)

      #local Density = <Density3.z,Density3.y>;
      #local TracksDone = TracksDone + Density3.x*Density3.y;
      RightScan(DensityMax,Density,TracksDone)

      #local Density = <Density3.x,Density3.z>;
      #local TracksDone = TracksDone + Density3.y*Density3.z;
      BottomScan(DensityMax,Density,TracksDone)
   #end//end Fill scanning...

//and close out the files in final form...
   CloseFile()
   #debug concat("\n\nTriscan completed.", str(TS_TotalTriangles,8,0),
                 " triangles exported to file ",FileName,"\n")

#end//macro Triscan    

