//Fur generator include file (version 2.0)
//Created by Margus Ramst (margus@peak.edu.ee)
//January 23, 2000

#include "admacros.inc"


/*              THE Fur MACRO               */

//Main macro, samples the object and places the hairs
//
//Parameters:
//
//Object:
//    #declared object to be made furry
//DivIn:
//    sampling rate along each axis (float or vector)
//PosJ:
//    randomness in the position of hairs;
//    PosJ=1 => max_displacement = object_size / DivIn / 2
//NumSegm
//    number of segments (cones) in each hair
//Len:
//    length of hairs (in POV units)
//LenJ:
//    randomness in hair length (fraction of total length)
//RadIn:
//    radius of hair at the base (in POV units)
//Wind:
//    direcion of wind (or any directional force),
//    longer vector => stronger force
//BendExp:
//    how the hair is bent by Wind:
//    1 bends uniformly from start to end;
//    N>1 bends more towards end;
//    N<1 bends more towards start;
//TwistIn:
//    random twisting of hair (fraction of 90 degrees rotation);
//    it is independent of NumSegm:
//    twist per segment is calculated as TwistIn/NumSegm
//HairFile:
//    if true, outputs cones to FileName,
//    #declaring a CSG union object 'Hairs'
//HairFile_Name:
//    name of the output file (in double quotes, e.g. "filename.inc")
//    You need to put something (e.g. '0') here even if FileO=false
//PointFile:
//    if true, outputs hair start points
//    and surface normals to PointFile_Name
//PointFile_Name:
//    name of the output file (in double quotes, e.g. "filename.inc")
//    You need to put something (e.g. '0') here even if PointFile=false
//Culling:
//    if true, enables culling, i.e. removal of hairs that do not
//    affect the image (does not work with reflections)
//TestPos:
//    array identifier; points from which the hair must be visible
//    to be created (when culling is enabled).
//    First entry is considered to be the camera position, the rest light sources.
//    Culling works differently for the two.
//MaxAngle:
//    How much the base of a  hair must be below the horizon
//    before the hair can be culled.
//    The object horizon is at 90 degrees.
//    If MaxAngle = 0 all hairs are discarded.
//    If MaxAngle = 180 no culling occurs.
//    Longer hairs require greater MaxAngle. A good value is around 95 - 110.
//Seed:
//    random number seed to be used

//NB! I have added manual bounding to the hairs to save memory;
//this generates warnings. You should either disable the warning
//stream with -GW or comment out the bounded_by statement in worm.inc

#macro Fur(Object, DivIn, PosJ,
           NumSegm, Len, LenJ, RadIn,
           Wind, BendExp, TwistIn,
           HairFile, HairFile_Name,
           PointFile, PointFile_Name,
           Culling, TestPos, MaxAngle,
           Seed)

    #debug "\nFur macro called\n"
    
    union{

    #local Seed=seed(Seed);

    #local HairNum=0;
    #local CullFlag=0;
    #local Culled=0;
    #if(Culling!=0)
        #debug "Culling: on\n"
        #local CullTests=dimension_size(TestPos,1);
        #if(CullTests>1)
            #local LightSw=1;
        #else #local LightSw=0;
        #end
    #else #debug "Culling: off\n"
    #end
    #local MaxAngle=radians(MaxAngle);

    #local Divisions=DivIn+<0,0,0>;
    #local Jitter=PosJ+<0,0,0>;
    #local LenJ=LenJ*Len;
    #local Twist=TwistIn*90/NumSegm;
    #local PosMin=min_extent(Object);
    #local PosMax=max_extent(Object);
    #local Dim=PosMax-PosMin;
    #local JumpInit=Dim/Divisions;
    #local Start=PosMin;
    #local XYZ=1;

    #if(HairFile!=0)
        #debug "Objets to file: on\n"
        #fopen OFile HairFile_Name write
        #write(OFile,"#declare Hairs=union{\n")
        #local FileO=1;
    #else #debug "Objets to file: off\n"
    #end
    #if(PointFile!=0)
        #debug "Points to file: on\n"
        #fopen PFile PointFile_Name write
    #else #debug "Points to file: off\n"
    #end       

    #debug "Calculating points...\n"
    #while (XYZ<=3)
        #local Count1=1;
        #local Count2=1;
        #switch(XYZ)
            #case(1) #local PosC=<PosMin.x+.5*JumpInit.x,PosMin.y+.5*JumpInit.y,PosMin.z-.1*JumpInit.z>;
                     #local NDir=z;
                     #local CountDim1=Divisions.x;
                     #local CountDim2=Divisions.y;
                     #local Jump1=JumpInit*x;
                     #local Jump2=JumpInit*y;
            #break
            #case(2) #local PosC=<PosMin.x-.1*JumpInit.x,PosMin.y+.5*JumpInit.y,PosMin.z+.5*JumpInit.z>;
                     #local NDir=x;
                     #local CountDim1=Divisions.z;
                     #local CountDim1=Divisions.y;
                     #local Jump1=JumpInit*z;
                     #local Jump2=JumpInit*y;
            #break
            #case(3) #local PosC=<PosMin.x+.5*JumpInit.x,PosMin.y-.1*JumpInit.y,PosMin.z+.5*JumpInit.z>;
                     #local NDir=y;
                     #local CountDim1=Divisions.x;
                     #local CountDim2=Divisions.z;
                     #local Jump1=JumpInit*x;
                     #local Jump2=JumpInit*z;
            #break
        #end

        #local Pos=PosC;
        #while(Count2<=CountDim2)
            #local SeedC=Count1+Count2;
            #local Flag=1;
            #local PosT=Pos+(1-NDir)*v_rand_ext(0,JumpInit/2*Jitter,Seed);
            #while(Flag=1)
                #local Rad=RadIn;
                #local N = <0,0,0>;
                #local Start = trace(Object, PosT, NDir, N);
    
                #if (N.x!=0 | N.y!=0 | N.z!=0)
    
                    #if(PointFile!=0)
                        #write(PFile,Start,",",N,",\n")
                    #end
    
                    //If culling is enabled
                    #if(Culling!=0)
                        #local CullFlag=1;
                        
                        //Test for camera (first test point)
                        #local CamDir=vnormalize(TestPos[0]-Start);
                        #if(acos(vdot(CamDir,N))<MaxAngle)
                            #local CullFlag=0;
                        #end
                        
                        //Test for light sources (any points after camera)
                        #if(LightSw!=0&CullFlag=1)
                            #local LC=1;
                            #while(LC<CullTests&CullFlag=1)
                                #local LightDir=vnormalize(TestPos[LC]-Start);
                                #if(abs(acos(vdot(LightDir,N))-pi/2)<abs(MaxAngle-pi/2))
                                    #local CullFlag=0;
                                #end
                                #local LC=LC+1;
                            #end
                        #end
                    
                    #end //End culling
    
                    //Call Worm macro to create hairs
                    #if(CullFlag=0)
                        Worm(NumSegm,rand_ext(Len,LenJ,Seed),Rad,
                             Start,vrotate(N,v_rand_ext(0,90*TwistIn/NumSegm,Seed)),
                             Wind,BendExp,
                             Twist,Seed,
                             HairFile)
                        #local HairNum=HairNum+1;
                    #else #local Culled=Culled+1;
                    #end
    
                    #local PosT=Pos+NDir*(Start-Pos)+(1-NDir)*v_rand_ext(0,JumpInit/2*Jitter,Seed);
    
                #else #local Flag=0;
                #end
            #end
            
            #if(Count1>=CountDim1)
                #local Pos=PosC+Count2*Jump2;
                #local Count1=1;
                //#debug concat("Dim ",str(XYZ,0,0)," Block ",str(Count2,0,0)," of ",str(CountDim2,0,0),"\n")
                #local Count2=Count2+1;
            #else
                #local Pos=PosC+Count1*Jump1+(Count2-1)*Jump2;
                #local Count1=Count1+1;
            #end
        
        #end
        #local XYZ=XYZ+1;
    #end
    #if(HairFile!=0)
        #write (OFile,"}")
        #fclose OFile
    #end
    #if(PointFile!=0)
        #fclose PFile
    #end
    
    //Report stats
    #debug concat("Created ",str(HairNum,0,0)," hairs")
    #if(Culling!=0)
        #debug concat(" (discarded ",str(Culled,0,0),")\n")
    #else #debug "\n"
    #end
    }
#end



/*              THE Fur_PF MACRO              */

//Fur_PF creates fur based on a pre-generated coordinate file
//(generated using PointFile and PointFile_Name options in fur macro)
//
//Most parameters are identical to those in 'Fur' (when present)
//The following are exceptions:
//
//InputFile:
//    name of the file containing coordinate data
//ScaleIn:
//    scale (vector or float identifier) that you wish to apply to the fur

#macro Fur_PF(InputFile, ScaleIn,
              NumSegm, Len, LenJ, RadIn,
              Wind, BendExp, TwistIn,
              HairFile, HairFile_Name,
              Culling,TestPos, MaxAngle,
              Seed)
    
    #debug "\nFur_PF macro called\n"

    union{

    #local Seed=seed(Seed);

    #local LenJ=LenJ*Len;
    #local Twist=TwistIn*90/NumSegm;

    #local HairNum=0;
    #local CullFlag=0;
    #local Culled=0;
    #if(Culling!=0)
        #debug "Culling: on\n"
        #local CullTests=dimension_size(TestPos,1);
        #if(CullTests>1)
            #local LightSw=1;
        #else #local LightSw=0;
        #end
    #else #debug "Culling: off\n"
    #end
    #local MaxAngle=radians(MaxAngle);

    #if(HairFile!=0)
        #debug "Objets to file: on\n"
        #fopen OFile HairFile_Name write
        #write(OFile,"#declare Hairs=union{\n")
        #local FileO=1;
    #else #debug "Objets to file: off\n"
    #end       

    #local Scale=ScaleIn*<1,1,1>;
    #if(Scale.x=0)
        #local Scale=Scale+<1,0,0>;
        #debug "Warning: cannot scale X by 0; scale X set to 1\n"
    #end
    #if(Scale.y=0)
        #local Scale=Scale+<0,1,0>;
        #debug "Warning: cannot scale Y by 0; scale Y set to 1\n"
    #end
    #if(Scale.z=0)
        #local Scale=Scale+<0,0,1>;
        #debug "Warning: cannot scale Z by 0; scale Z set to 1\n"
        
    #end
    #fopen IF InputFile read
    #while(defined(IF))
        #read(IF,Point,Normal)
        #local Start=Point*Scale;
        #local N=Normal*(1/Scale);
        
        //If culling is enabled
        #if(Culling!=0)
            #local CullFlag=1;
            
            //Test for camera (first test point)
            #local CamDir=vnormalize(TestPos[0]-Start);
            #if(acos(vdot(CamDir,N))<MaxAngle)
                #local CullFlag=0;
            #end
            
            //Test for light sources (any points after camera)
            #if(LightSw!=0&CullFlag=1)
                #local LC=1;
                #while(LC<CullTests&CullFlag=1)
                    #local LightDir=vnormalize(TestPos[LC]-Start);
                    #if(abs(acos(vdot(LightDir,N))-pi/2)<abs(MaxAngle-pi/2))
                        #local CullFlag=0;
                    #end
                    #local LC=LC+1;
                #end
            #end
        
        #end //End culling

        //Call Worm macro to create hairs
        #if(CullFlag=0)
            #local Rad=RadIn;
            Worm(NumSegm,rand_ext(Len,LenJ,Seed),Rad,
                 Start,vrotate(N,v_rand_ext(0,90*TwistIn/NumSegm,Seed)),
                 Wind,BendExp,
                 Twist,Seed,
                 HairFile)
            #local HairNum=HairNum+1;
        #else #local Culled=Culled+1;
        #end

    #end

    #if(HairFile!=0)
        #write (OFile,"}")
        #fclose OFile
    #end
    
    //Report stats
    #debug concat("Created ",str(HairNum,0,0)," hairs")
    #if(Culling!=0)
        #debug concat(" (discarded ",str(Culled,0,0),")\n")
    #else #debug "\n"
    #end
    }
#end


/*              THE Worm MACRO              */

//Generates random "Worm" or hair strand
//
//NumSegm:
//    number of segments
//Len:
//    length of worm
//RadStart:
//    worm start radius
//PStart:
//    start point
//DirStart:
//    initial direction
//Wind:
//    directional force applied to worm (vector)
//BendExp:
//    1 => Wind bends the worm uniformly; >1 => bends more towards end
//Random:
//    randomness in X,Y and Z directions (vector)
//Seed:
//    random number seed
//OutF:
//    1 => generate output file (works with fur macro, otherwise leave 0)

#macro Worm(NumSegm,Len,RadStart,PStart,DirStart,Wind,BendExp,Random,Seed,OutF)
        #local CurSegm = 0;
        #local LenSegm = Len/NumSegm;
        #local PointStart = <1,1,1>*PStart;
        #local PointEnd = PointStart+vnormalize(DirStart)*LenSegm;
        #local VDir=vnormalize(vnormalize(PointEnd-PointStart)+Interpolate(CurSegm+1,0,NumSegm,0,Wind,BendExp));
        #local RadConst = RadStart;
        union{
            #if(OutF!=0)
            #write(OFile,"union{\n")
            #end
            #while (CurSegm < NumSegm)
                    #local RadEnd=(NumSegm-CurSegm-1)/NumSegm*RadConst;
                    #local PointEnd=PointStart+vrotate(LenSegm*VDir,v_rand_ext(0,Random,Seed));
                    cone {PointStart, RadStart, PointEnd, RadEnd}
                    #if(OutF!=0)
                    #write(OFile,"cone{",PointStart,",",RadStart,",",PointEnd,",",RadEnd,"}\n")
                    #end
                    #local CurSegm=CurSegm+1;
                    #local VDir=vnormalize(vnormalize(PointEnd-PointStart)+Interpolate(CurSegm+1,0,NumSegm,0,Wind,BendExp));
                    #local PointStart=PointEnd;
                    #local RadStart=RadEnd;
            #end
            bounded_by{sphere{PStart,Len}} //Comment out for slightly faster render & larger memory consumption
        }
        #if(OutF!=0)
        #write(OFile,"bounded_by{sphere{",PStart,",",Len,"}}}\n")
        #end
#end
