/* 
 *	MANDEL3.C
 *		A program to generate a POV-Ray include file containing spheres approximating
 *	a 3D mandelbrot set.
 */

#include <stdio.h>
#include <stdlib.h> 
#include <math.h>
#include <string.h>
#include <ctype.h>

#define BLOB 1
#define SPHERE 2

 
void main(int argc, char *argv[])
{
	FILE *outfile;
	long maxiter, iter;
	double X,Y,Z,r,i,j;
	double div;
	double rn,in;
	char buffer[255];
	int type = SPHERE; 
	size_t bufsize;
	 
	if( argc < 4 || argc > 5 || strcmp(argv[1], "?") == 0)
	{
		printf("Generates a POV include file that approximates the 3D M-set\n");
		printf("\tUsage:\tMandel3 File Iterations Division [B|S]\n");
		printf("\tParameters:\n");
		printf("\tFile:       Name of file to create\n");
		printf("\tIterations: Number of iterations to run\n");
		printf("\tDivision:   Size of a voxel in units\n");
		printf("\t[B|S]:      (B)lob or (S)phere output\n");
	}
	outfile = fopen(argv[1], "w");
	if(argc == 5)
	{
		if((char)toupper(argv[4][0]) == 'B')
		{
			type = BLOB;
			printf("Blobs used %c", argv[4][0]);
		}
		else
		{
			type = SPHERE;
			printf("Spheres used %c", argv[4][0]);
		}
	}
	if(outfile == NULL)
	{
		printf("Error creating file\n");
		exit(1);
	}
	div = atof(argv[3]);
	if(div <= 0)
	{
		printf("Voxel size must be greater than zero\n");
		exit(1);
	}
	maxiter = atol(argv[2]);
	if(maxiter <= 0)
	{
		printf("Minimum number of iterations is 1\n");
		exit(1);
	}
	
	
	/* Calculation loops */
    for(X = -2;X <= 1;X+=div)
    {
    	for(Y = -1;Y <= 1; Y+=div)
    	{
    		for(Z = -1;Z <= 1;Z+=div)
    		{
    			iter = 0;
    			r = X; i = Y; j = Z;
    			while(iter < maxiter && ((r*r)+(i+i)+(j*j)) <= 4)
    			{
    				rn = ((r*r)-(i*i))+X;
    				in = ((2*r*i)-(j*j))+Y;
    				j  = ((2*i*j)+(2*r*j))+Z;
    				r  = rn;
    				i  = in;
    				iter++;
    			}
    			if(iter == maxiter)
    			{
    				/* Point is in set so far, write to file */
    				switch(type)
    				{
    					case SPHERE:
    						bufsize = sprintf(buffer, "sphere{<%f,%f,%f>,%f}\n",X,Y,Z,div);
    						break;
    					case BLOB:
    						bufsize = sprintf(buffer, "sphere{<%f,%f,%f>,%f,1}\n",X,Y,Z,(div*2));
    						break;
    					default:
    						printf("Unknown object type %d!", type);
    						exit(1);
    				}
    				
    				fwrite(buffer, bufsize, 1, outfile);
    				
    			}
    		}
    	}
    }
    fclose(outfile);
}