// SurfaceMacro.inc - Defines macro DrawSurf() which creates a mesh surface defined by a function definedin
//                    Macro SurfaceFunction. The mesh is comprised of smooth triangles.

/*

Surface is drawn as x-z plane with height in y direction.

SurfPos - Position of centre of surface.
SurfSize - Size in x and z directions, y coordinate is vertical scale.
SurfPatches - Number of patches in x and z direction. Each patch is made up of 2 triangles.
SurfFunction - Function number. Use this to pass to the surface function to define the function to use
SurfTexture - Texture applied to surface.
*/

// Define surface function macro

// FuncNum is passed from the call to DrawSurf
// FuncAction exists incase you need to do some initialization. The macro is called once with this set to 0
// before it is called to get height with this set to 1.
// Pos is the position at which to calculate a height.
// Macro returns Y value at Pos.

// Enclose each function in a #case..#break wrapper like those below.
// Change the number to suit.

#macro SurfaceFunction(FuncNum,FuncAction,Pos)
        #local Y = Pos.y;
        #switch (FuncNum)
        #case (0) // Simple Sine wave
                #if (FuncAction=0) // Initialize
                #else // Calculate Y value from X and Z        
                        #local D=sqrt((X*X) + (Z*Z)); // Calculate distance
                        #local Y = sin(((D/4)-(clock*6))*2*pi);
                #end
        #break       
        #case (1) // Average surface through array of control points, requires arrays ContPoint and ContSize.
                #local Y = 0;
                #local I = 0;
                #local NumContPoints = dimension_size(ContPoint,1);
                #while (I<NumContPoints)
                        #local D=sqrt(((Pos.x-ContPoint[I].x)*(Pos.x-ContPoint[I].x))+
                                          ((Pos.z-ContPoint[I].z)*(Pos.z-ContPoint[I].z)));
                        #if (D<ContSize[I])
                                #local Y = Y + (ContPoint[I].y*(1-pow(D/ContSize[I],2)));
                        #end
                        #local I = I + 1;
                #end
        #break       
        #end
        ( Y ) // Return height
#end
        
#macro DrawSurf(SurfPos, SurfSize, SurfPatchs, SurfFuncNum, SurfTexture)
        #local Heights=array[SurfPatchs.x+3][2]
        #local Normals=array[SurfPatchs.x+3]
        #local NewNormal = <1,1,1>;
        
        // Initialize function
        #local SurfFuncAction = 0;
        #local Y = SurfaceFunction(SurfFuncNum,SurfFuncAction,<0,0,0>);
        #local SurfFuncAction = 1;
        
        #local SurfStep = <(SurfSize.x/SurfPatchs.x),1,(SurfSize.z/SurfPatchs.z)>;
        
        // Prime Heights array
        #local X = (SurfPos.x-(SurfSize.x/2))+(SurfStep.x*-1);
        #local Z = (SurfPos.z-(SurfSize.z/2))+(SurfStep.z*-1);
        
        #local XI = 0;
        #while (XI<SurfPatchs.x+3)
                #local Heights[XI][0] = SurfaceFunction(SurfFuncNum,SurfFuncAction,<X,SurfPos.y,Z>); 
                #local XI = XI + 1;
                #local X = X + SurfStep.x;
        #end

        #local Z = (SurfPos.z-(SurfSize.z/2));
        #local X = (SurfPos.x-(SurfSize.x/2))+(SurfStep.x*-1);
        #local XI = 0;
        
        mesh{
        
        #while (XI<SurfPatchs.x+3)
                #local Heights[XI][1]=SurfaceFunction(SurfFuncNum,SurfFuncAction,<X,SurfPos.y,Z>); 
                #local XI = XI + 1;
                #local X = X + SurfStep.x;
        #end
        
        #local ZI = 2;
        #local Z = (SurfPos.z-(SurfSize.z/2))+SurfStep.z;
        #while (ZI<SurfPatchs.z+3)
                #local ZPrev = Z-SurfStep.z;
                #local ZPrevPrev = Z-(SurfStep.z*2);
                #local XI = 0;
                #local X = (SurfPos.x-(SurfSize.x/2))+(SurfStep.x*-1);
                #while (XI<SurfPatchs.x+3)
                        #local XPrev = X-SurfStep.x;
                        #local XNext = X+SurfStep.x;
                        #local Y = SurfaceFunction(SurfFuncNum,SurfFuncAction,<X,SurfPos.y,Z>);
                        #if(XI>0)
                                #if (XI<SurfPatchs.x+2)
                                        #local V0 = vnormalize(vcross(<XPrev,Heights[XI-1][1],ZPrev>-<X,Heights[XI][1],ZPrev>,
                                                                      <X,Heights[XI][0],ZPrevPrev>-<X,Heights[XI][1],ZPrev>));
                                        #local V1 = vnormalize(vcross(<X,Heights[XI][0],ZPrevPrev>-<X,Heights[XI][1],ZPrev>,
                                                                      <XNext,Heights[XI+1][1],ZPrev>-<X,Heights[XI][1],ZPrev>));
                                        #local V2 = vnormalize(vcross(<XNext,Heights[XI+1][1],ZPrev>-<X,Heights[XI][1],ZPrev>,
                                                                      <X,Y,Z>-<X,Heights[XI][1],ZPrev>));
                                        #local V3 = vnormalize(vcross(<X,Y,Z>-<X,Heights[XI][1],ZPrev>,
                                                                      <XPrev,Heights[XI-1][1],ZPrev>-<X,Heights[XI][1],ZPrev>));
                                        #local NewNormal = vnormalize(V0+V1+V2+V3);
                                #end        
                        #end
                        #if (ZI>2)
                            #if (XI>1)
                                #if (XI<SurfPatchs.x+2)
                                        smooth_triangle {
                                                <X,Heights[XI][1],ZPrev>, NewNormal,
                                                <XPrev,Heights[XI-1][0],ZPrev>, Normals[XI-1],
                                                <X,Heights[XI][0],ZPrevPrev>, Normals[XI]
                                                texture { SurfTexture }
                                                }
                                #end                
                            #end
                            #if (XI>0)
                                #if (XI<SurfPatchs.x+1)
                                        smooth_triangle {
                                                <X,Heights[XI][0],ZPrevPrev>, Normals[XI],
                                                <XNext,Heights[XI+1][0],ZPrevPrev>, Normals[XI+1],
                                                <X,Heights[XI][1],ZPrev>, NewNormal
                                                texture { SurfTexture }
                                                }
                                #end        
                            #end
                        #end
                        #local Heights[XI][0] = Heights[XI][1];
                        #local Heights[XI][1] = Y;
                        #local Normals[XI] = NewNormal;
                        #local XI = XI + 1;
                        #local X = XNext;
                #end
                #local ZI = ZI + 1;
                #local Z = Z + SurfStep.z;
        #end
        scale <1,SurfSize.y,1>
        }
#end
