/*******************************************************************************
 * isosurf.cpp
 *
 * This module implements the iso surface shapetype.
 *
 * This module was written by R.Suzuki.
 * Ported to POV-Ray 3.5 by Thorsten Froehlich.
 *
 * from Persistence of Vision Ray Tracer ('POV-Ray') version 3.7.
 * Copyright 1991-2003 Persistence of Vision Team
 * Copyright 2003-2008 Persistence of Vision Raytracer Pty. Ltd.
 * ---------------------------------------------------------------------------
 * NOTICE: This source code file is provided so that users may experiment
 * with enhancements to POV-Ray and to port the software to platforms other
 * than those supported by the POV-Ray developers. There are strict rules
 * regarding how you are permitted to use this file. These rules are contained
 * in the distribution and derivative versions licenses which should have been
 * provided with this file.
 *
 * These licences may be found online, linked from the end-user license
 * agreement that is located at http://www.povray.org/povlegal.html
 * ---------------------------------------------------------------------------
 * POV-Ray is based on the popular DKB raytracer version 2.12.
 * DKBTrace was originally written by David K. Buck.
 * DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
 * ---------------------------------------------------------------------------
 * $File: //depot/povray/smp/source/backend/shape/isosurf.cpp $
 * $Revision: #35 $
 * $Change: 4528 $
 * $DateTime: 2008/02/04 08:36:09 $
 * $Author: chrisc $
 *******************************************************************************/

/*********************************************************************************
 * NOTICE
 *
 * This file is part of a BETA-TEST version of POV-Ray version 3.7. It is not
 * final code. Use of this source file is governed by both the standard POV-Ray
 * licences referred to in the copyright header block above this notice, and the
 * following additional restrictions numbered 1 through 4 below:
 *
 *   1. This source file may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd.
 *
 *   2. This notice may not be altered or removed.
 *   
 *   3. Binaries generated from this source file by individuals for their own
 *      personal use may not be re-distributed without the written permission
 *      of Persistence of Vision Raytracer Pty. Ltd. Such personal-use binaries
 *      are not required to have a timeout, and thus permission is granted in
 *      these circumstances only to disable the timeout code contained within
 *      the beta software.
 *   
 *   4. Binaries generated from this source file for use within an organizational
 *   	unit (such as, but not limited to, a company or university) may not be
 *      distributed beyond the local organizational unit in which they were made,
 *      unless written permission is obtained from Persistence of Vision Raytracer
 *      Pty. Ltd. Additionally, the timeout code implemented within the beta may
 *      not be disabled or otherwise bypassed in any manner.
 *
 * The following text is not part of the above conditions and is provided for
 * informational purposes only.
 *
 * The purpose of the no-redistribution clause is to attempt to keep the
 * circulating copies of the beta source fresh. The only authorized distribution
 * point for the source code is the POV-Ray website and Perforce server, where
 * the code will be kept up to date with recent fixes. Additionally the beta
 * timeout code mentioned above has been a standard part of POV-Ray betas since
 * version 1.0, and is intended to reduce bug reports from old betas as well as
 * keep any circulating beta binaries relatively fresh.
 *
 * All said, however, the POV-Ray developers are open to any reasonable request
 * for variations to the above conditions and will consider them on a case-by-case
 * basis.
 *
 * Additionally, the developers request your co-operation in fixing bugs and
 * generally improving the program. If submitting a bug-fix, please ensure that
 * you quote the revision number of the file shown above in the copyright header
 * (see the '$Revision:' field). This ensures that it is possible to determine
 * what specific copy of the file you are working with. The developers also would
 * like to make it known that until POV-Ray 3.7 is out of beta, they would prefer
 * to emphasize the provision of bug fixes over the addition of new features.
 *
 * Persons wishing to enhance this source are requested to take the above into
 * account. It is also strongly suggested that such enhancements are started with
 * a recent copy of the source.
 *
 * The source code page (see http://www.povray.org/beta/source/) sets out the
 * conditions under which the developers are willing to accept contributions back
 * into the primary source tree. Please refer to those conditions prior to making
 * any changes to this source, if you wish to submit those changes for inclusion
 * with POV-Ray.
 *
 *********************************************************************************/

#include <limits.h>
#include <algorithm>

#include "backend/frame.h"
#include "backend/scene/objects.h"
#include "backend/scene/scene.h"
#include "backend/scene/threaddata.h"
#include "backend/shape/isosurf.h"
#include "backend/shape/spheres.h" // TODO - Move sphere intersection function to math code! [trf]
#include "backend/shape/boxes.h" // TODO - Move box intersection function to math code! [trf]
#include "backend/vm/fnpovfpu.h"

// this must be the last file included
#include "base/povdebug.h"

namespace pov
{

using namespace std;

/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

/* Side hit. */
const int SIDE_X_0 = 1;
const int SIDE_X_1 = 2;
const int SIDE_Y_0 = 3;
const int SIDE_Y_1 = 4;
const int SIDE_Z_0 = 5;
const int SIDE_Z_1 = 6;

/*****************************************************************************
*
* FUNCTION
*
*   All_IsoSurface_Intersections
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

bool IsoSurface::All_Intersections(Ray& ray, IStack& Depth_Stack, const TraceThreadData *Thread)
{
  Thread->Stats[Ray_IsoSurface_Bound_Tests]++;
  
  VECTOR P, D;
  DBL len;
  
  if(Trans != NULL)
  {
    //transform ray into isosurface space
    MInvTransPoint(P, ray.Origin, Trans);
	  MInvTransDirection(D, ray.Direction, Trans);
  }
  else
  {
    Assign_Vector(P, ray.Origin);
    Assign_Vector(D, ray.Direction);
  }
  
  //normalise D and keep length for later
  VLength(len, D);
  DBL scaled_accuracy = accuracy * len;
  len = 1/len;
	VScaleEq(D, len);
	
	DBL t1, t2;
	int side_hit1, side_hit2;
  
  
  Ray temp_ray;
  Assign_Vector(temp_ray.Origin, P);
  Assign_Vector(temp_ray.Direction, D);
  switch(container_shape)
  {
    case true: //sphere
    {
			if(!Sphere::Intersect(temp_ray, container.sphere.center, 
	                         (container.sphere.radius) * (container.sphere.radius),
	                          &t1, &t2) )
	    { return false; }
	    Thread->Stats[Ray_Sphere_Tests]--;
			Thread->Stats[Ray_Sphere_Tests_Succeeded]--;
    }
    break;
    
    case false: //box
    {
      if(!Box::Intersect(temp_ray, NULL, container.box.corner1, container.box.corner2,
		                     &t1, &t2, &side_hit1, &side_hit2))
		  { return false; }
    }
    break;
  }
  
  //the container shape was hit!
  Thread->Stats[Ray_IsoSurface_Bound_Tests_Succeeded]++;
  t1 = max(t1, 0.0);
  
  int hit_count = 0;
  
  RayContext rc(P, D, scaled_accuracy, *this, Thread);
  
  //check for visible container intersections
  if(closed)
  {
    DBL f = rc(t1);
    //not sure if this is really necessary! [lwi]
    /*if(t1 < accuracy)
    {
      if(f < max_gradient * accuracy * 4)
      {
        t1 = accuracy * 5; //move the interval boundary from the surface
        f = rc(t1);
      }
    }*/
    if(t1 > scaled_accuracy)
    {
      //is there a true container intersection?
      if(f < 0)
      {
        DBL depth = t1 * len;
        VECTOR point;
        VEvaluateRay(point, ray.Origin, depth, ray.Direction);
        if(Clip.empty() || Point_In_Clip(point, Clip, Thread))
				{
					Depth_Stack->push(Intersection(depth, point, this, side_hit1));
					hit_count++;
				}
      }
    }
    
    //now check t2
    f = rc(t2);
    if(f < 0)
    {
      DBL depth = t2 * len;
      VECTOR point;
      VEvaluateRay(point, ray.Origin, depth, ray.Direction);
      if(Clip.empty() || Point_In_Clip(point, Clip, Thread))
			{
				Depth_Stack->push(Intersection(depth, point, this, side_hit1));
			}
    }
  }
  
  if(hit_count >= max_trace)
  { return true; }
  
  Thread->Stats[Ray_IsoSurface_Tests]++;
  
  if(fabs(rc(t1)) < max_gradient * scaled_accuracy * 4)
  {
    t1 += scaled_accuracy * 5; //move the interval boundary from the surface
  } 
  
  t2 = min(t2, BOUND_HUGE);
  t1 = min(t1, t2);
  if(t2 - t1 < scaled_accuracy)
  { return false; /*really? false? or hit_count > 0? [lwi]*/ }
  
  VECTOR start_point, end_point;
  VEvaluateRay(end_point, P, t2, D);
  
  for(; hit_count < max_trace; hit_count++)
  {
    VEvaluateRay(start_point, P, t1, D);
    Thread->Stats[Ray_IsoSurface_Find_Root]++; //clone 3.6.1 behaviour ignoring cache hits!
    
    if(cache.check(start_point, end_point, max_gradient, Thread))
    { break; }
  
    DBL found_t = rc.Bisect(t1, t2);
    //TODO: update max_gradient statistics
    
    //no intersection found?
    if(found_t == BOUND_HUGE)
    {
      //note: we assume that the transformation does not change between store and check calls!
	    cache.store(start_point, end_point, rc.fmax);
      break;
    }
    
    
    DBL depth = found_t * len;
    VECTOR point;
    VEvaluateRay(point, ray.Origin, depth, ray.Direction);
    if(Clip.empty() || Point_In_Clip(point, Clip, Thread))
		{
			Depth_Stack->push(Intersection(depth, point, this, 0));
		}
		
		t1 = scaled_accuracy * 5 + found_t;
		if(t2 - t1 < scaled_accuracy)
		{ break; }
  }
  
  if(eval)
  {
    DBL temp_max_gradient = max_gradient; // TODO FIXME - works around nasty gcc (found using 4.0.1) bug failing to honor casting away of volatile on pass by value on template argument lookup [trf]
    max_gradient = max(temp_max_gradient, rc.max_gradient); // TODO FIXME - This is not thread-safe but should be!!! [trf]
  }
  
  if(hit_count > 0)
  {
    Thread->Stats[Ray_IsoSurface_Tests_Succeeded]++;
    return true;
  }
  else
  { return false; }
}


/*****************************************************************************
*
* FUNCTION
*
*   Inside_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

bool IsoSurface::Inside(VECTOR IPoint, const TraceThreadData *Thread)
{
	VECTOR Origin_To_Center;
	VECTOR New_Point;
	DBL OCSquared;

	/* Transform the point into box space. */
	if(Trans != NULL)
		MInvTransPoint(New_Point, IPoint, Trans);
	else
		Assign_Vector(New_Point, IPoint);

	if(container_shape != 0)
	{
		/* Use ellipse method. */
		VSub(Origin_To_Center, container.sphere.center, New_Point);
		VDot(OCSquared, Origin_To_Center, Origin_To_Center);

		if(OCSquared > Sqr(container.sphere.radius))
			return (Test_Flag(this, INVERTED_FLAG));
	}
	else
	{
		/* Test to see if we are outside the box. */
		if((New_Point[X] < container.box.corner1[X]) || (New_Point[X] > container.box.corner2[X]))
			return (Test_Flag(this, INVERTED_FLAG));

		if((New_Point[Y] < container.box.corner1[Y]) || (New_Point[Y] > container.box.corner2[Y]))
			return (Test_Flag(this, INVERTED_FLAG));

		if((New_Point[Z] < container.box.corner1[Z]) || (New_Point[Z] > container.box.corner2[Z]))
			return (Test_Flag(this, INVERTED_FLAG));
	}

	if(Evaluate_Function(Thread->functionContext, New_Point) > 0)
		return (Test_Flag(this, INVERTED_FLAG));

	/* Inside the box. */
	return (!Test_Flag(this, INVERTED_FLAG));
}



/*****************************************************************************
*
* FUNCTION
*
*   IsoSurface_Normal
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void IsoSurface::Normal(VECTOR Result, Intersection *Inter, const TraceThreadData *Thread)
{
	VECTOR New_Point, TPoint;
	DBL funct;

	switch (Inter->i1)
	{
		case SIDE_X_0:
			Make_Vector(Result, -1.0, 0.0, 0.0);
			break;
		case SIDE_X_1:
			Make_Vector(Result, 1.0, 0.0, 0.0);
			break;
		case SIDE_Y_0:
			Make_Vector(Result, 0.0, -1.0, 0.0);
			break;
		case SIDE_Y_1:
			Make_Vector(Result, 0.0, 1.0, 0.0);
			break;
		case SIDE_Z_0:
			Make_Vector(Result, 0.0, 0.0, -1.0);
			break;
		case SIDE_Z_1:
			Make_Vector(Result, 0.0, 0.0, 1.0);
			break;

		default:

			/* Transform the point into the isosurface space */
			if(Trans != NULL)
				MInvTransPoint(New_Point, Inter->IPoint, Trans);
			else
				Assign_Vector(New_Point, Inter->IPoint);

			if(container_shape)
			{
				VSub(Result, New_Point, container.sphere.center);
				VLength(funct, Result);
				if(fabs(funct - container.sphere.radius) < EPSILON)
				{
					VInverseScaleEq(Result, container.sphere.radius);
					break;
				}
			}

			Assign_Vector(TPoint, New_Point);
			funct = Evaluate_Function(Thread->functionContext, TPoint);
			Assign_Vector(TPoint, New_Point);
			TPoint[X] += accuracy;
			Result[X] = Evaluate_Function(Thread->functionContext, TPoint) - funct;
			Assign_Vector(TPoint, New_Point);
			TPoint[Y] += accuracy;
			Result[Y] = Evaluate_Function(Thread->functionContext, TPoint) - funct;
			Assign_Vector(TPoint, New_Point);
			TPoint[Z] += accuracy;
			Result[Z] = Evaluate_Function(Thread->functionContext, TPoint) - funct;

			if((Result[X] == 0) && (Result[Y] == 0) && (Result[Z] == 0))
				Result[X] = 1.0;
			VNormalize(Result, Result);
	}


	/* Transform the point into the boxes space. */

	if(Trans != NULL)
	{
		MTransNormal(Result, Result, Trans);

		VNormalize(Result, Result);
	}
}



/*****************************************************************************
*
* FUNCTION
*
*   Translate_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void IsoSurface::Translate(VECTOR, TRANSFORM* tr)
{
	Transform(tr);
}



/*****************************************************************************
*
* FUNCTION
*
*   Rotate_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void IsoSurface::Rotate(VECTOR, TRANSFORM* tr)
{
	Transform(tr);
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void IsoSurface::Scale(VECTOR, TRANSFORM* tr)
{
	Transform(tr);
}



/*****************************************************************************
*
* FUNCTION
*
*   Invert_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void IsoSurface::Invert()
{
	Invert_Flag(this, INVERTED_FLAG);
}



/*****************************************************************************
*
* FUNCTION
*
*   Transform_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void IsoSurface::Transform(TRANSFORM* tr)
{
	if(Trans == NULL)
		Trans = Create_Transform();

	Compose_Transforms(Trans, tr);

	Compute_BBox();
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

IsoSurface::IsoSurface() 
  :Function(NULL), accuracy(0.001), max_trace(1), eval(false), closed(true), container_shape(0), //box
   max_gradient(1.1), gradient(0.0), threshold(0.0)
{
	INIT_OBJECT_FIELDS(this, ISOSURFACE_OBJECT)

	Make_Vector(container.box.corner1, -1.0, -1.0, -1.0);
	Make_Vector(container.box.corner2, 1.0, 1.0, 1.0);

	Make_BBox(BBox, -1.0, -1.0, -1.0, 2.0, 2.0, 2.0);

	Trans = Create_Transform();

	eval_param[0] = 0.0; // 1.1; // not necessary
	eval_param[1] = 0.0; // 1.4; // not necessary
	eval_param[2] = 0.0; // 0.99; // not necessary

	mginfo = (ISO_Max_Gradient *)POV_MALLOC(sizeof(ISO_Max_Gradient), "isosurface max_gradient info");
	mginfo->refcnt = 1;
	mginfo->max_gradient = 0.0;
	mginfo->gradient = 0.0; // not really necessary yet [trf]
	mginfo->eval_max = 0.0;
	mginfo->eval_cnt = 0.0;
	mginfo->eval_gradient_sum = 0.0;
}



/*****************************************************************************
*
* FUNCTION
*
*   Copy_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

ObjectPtr IsoSurface::Copy()
{
	IsoSurface *New = new IsoSurface();
	Destroy_Transform(New->Trans);
	*New = *this;

	New->Function = Parser::Copy_Function(vm, Function);
	New->Trans = Copy_Transform(Trans);

	New->mginfo = mginfo;
	New->mginfo->refcnt++;

	return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Destroy_IsoSurface
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   R. Suzuki
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

IsoSurface::~IsoSurface()
{
	DBL temp_max_gradient = max_gradient; // TODO FIXME - works around nasty gcc (found using 4.0.1) bug failing to honor casting away of volatile on pass by value on template argument lookup [trf]

	mginfo->refcnt--;
	mginfo->gradient = max(gradient, mginfo->gradient);
	mginfo->max_gradient = max((DBL)temp_max_gradient, mginfo->max_gradient);

	if(/*(Stage == STAGE_SHUTDOWN) && */(mginfo->refcnt == 0)) // TODO
	{
		FunctionCode *fn = vm->GetFunction(*(Function));

		if(fn != NULL)
		{
			if(eval == false)
			{
				// Only show the warning if necessary!
				// BTW, not being too picky here is a feature and not a bug ;-)  [trf]
				if((mginfo->gradient > EPSILON) && (mginfo->max_gradient > EPSILON))
				{
					DBL diff = mginfo->max_gradient - mginfo->gradient;
					DBL prop = fabs(mginfo->max_gradient / mginfo->gradient);

					if(((prop <= 0.9) && (diff <= -0.5)) ||
					   (((prop <= 0.95) || (diff <= -0.1)) && (mginfo->max_gradient < 10.0)))
					{
;// TODO MESSAGE						WarningAt(0, fn->filename, fn->filepos.lineno, fn->filepos.offset,
//						          "The maximum gradient found was %0.3f, but max_gradient of the\n"
//						          "isosurface was set to %0.3f. The isosurface may contain holes!\n"
//						          "Adjust max_gradient to get a proper rendering of the isosurface.",
//						          (float)(mginfo->gradient),
//						          (float)(mginfo->max_gradient));
					}
					else if((diff >= 10.0) || ((prop >= 1.1) && (diff >= 0.5)))
					{
;// TODO MESSAGE						WarningAt(0, fn->filename, fn->filepos.lineno, fn->filepos.offset,
//						          "The maximum gradient found was %0.3f, but max_gradient of\n"
//						          "the isosurface was set to %0.3f. Adjust max_gradient to\n"
//						          "get a faster rendering of the isosurface.",
//						          (float)(mginfo->gradient),
//						          (float)(mginfo->max_gradient));
					}
				}
			}
			else
			{
				DBL diff = (mginfo->eval_max / max(mginfo->eval_max - mginfo->eval_var, EPSILON));

				if((eval_param[0] > mginfo->eval_max) ||
				   (eval_param[1] > diff))
				{
					mginfo->eval_cnt = max(mginfo->eval_cnt, 1.0); // make sure it won't be zero

;// TODO MESSAGE					WarningAt(0, fn->filename, fn->filepos.lineno, fn->filepos.offset,
//					          "Evaluate found a maximum gradient of %0.3f and an average\n"
//					          "gradient of %0.3f. The maximum gradient variation was %0.3f.\n",
//					          (float)(mginfo->eval_max),
//					          (float)(mginfo->eval_gradient_sum / mginfo->eval_cnt),
//					          (float)(mginfo->eval_var));

// TODO FIXME
#if 0
					if(opts.Options & VERBOSE)
					{
						diff = max(diff, 1.0); // prevent contradicting output

						Debug_Info("It is recommended to adjust the parameters of 'evaluate' to:\n"
					               "First parameter less than %0.3f\n"
					               "Second parameter less than %0.3f and greater than 1.0\n"
					               "Third parameter greater than %0.3f and less than 1.0\n",
						          (float)(mginfo->eval_max),
						          (float)(diff),
						          (float)(1.0 / diff));
					}
#endif
				}
			}
		}
	}

	if(mginfo->refcnt == 0)
		POV_FREE(mginfo);

	Parser::Destroy_Function(vm, Function);
	Destroy_Transform(Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Compute_IsoSurface_BBox
*
* INPUT
*
*   ISOSURFACE - IsoSurface
*
* OUTPUT
*
*   ISOSURFACE
*
* RETURNS
*
* AUTHOR
*
* DESCRIPTION
*
*   Calculate the bounding box of an Isosurface.
*
* CHANGES
*
******************************************************************************/

void IsoSurface::Compute_BBox()
{
	if(container_shape != 0)
	{
		Make_BBox(BBox,
		          container.sphere.center[X] - container.sphere.radius,
		          container.sphere.center[Y] - container.sphere.radius,
		          container.sphere.center[Z] - container.sphere.radius,
		          container.sphere.radius * 2,
		          container.sphere.radius * 2,
		          container.sphere.radius * 2);
	}
	else
	{
		// [ABX 20.01.2004] Low_Left introduced to hide BCC 5.5 bug
		BBOX_VECT& Low_Left = BBox.Lower_Left;

		Assign_BBox_Vect(Low_Left, container.box.corner1);
		VSub(BBox.Lengths, container.box.corner2, container.box.corner1);
	}

	if(Trans != NULL)
	{
		Recompute_BBox(&BBox, Trans);
	}
}


/*****************************************************************************
 *
 * FUNCTION
 *
 *   Evaluate_Function
 *
 * INPUT
 *
 * OUTPUT
 *
 * RETURNS
 *
 * AUTHOR
 *
 * DESCRIPTION
 *
 *   -
 *
 * CHANGES
 *   
 *   -
 *
 ******************************************************************************/

DBL IsoSurface::Evaluate_Function(FPUContext *ctx, VECTOR fnvec)
{
	ctx->SetLocal(X, fnvec[X]);
	ctx->SetLocal(Y, fnvec[Y]);
	ctx->SetLocal(Z, fnvec[Z]);

	return POVFPU_Run(ctx, *Function) - threshold;
}

/*****************************************************************************
New code introduced by Lukas Winter.
******************************************************************************/

DBL IsoSurface::RayContext::operator()(DBL t) const
{
  //RayContext acts like a function (sampling the isosurface along the ray)
  VECTOR point;
  VEvaluateRay(point, p, t, d);
  return isosurf.Evaluate_Function(thread_data->functionContext, point);
}

//note: this class assumes that d is normalised!
IsoSurface::RayContext::RayContext(const VECTOR ip, const VECTOR id, DBL a,
                                   IsoSurface& i, const TraceThreadData* td)
                             :isosurf(i), thread_data(td), accuracy(a), max_gradient(i.max_gradient), fmax(BOUND_HUGE)
{
  Assign_Vector(p, ip);
  Assign_Vector(d, id);
}

/*the maximum number of entries in the isosurface cache per object. This really should be something like
1 (camera rays) + number of light sources (shadow tests) + number of reflective texture layers (reflection rays)
[lwi]*/
const int IsoSurface::Cache::max_size = 2;

void IsoSurface::Cache::store(VECTOR iA, VECTOR iB, DBL ifmax)
{
  if(data.get() == NULL)
  {
    data.reset(new list<Data>);
  }
  Data new_data;
  Assign_Vector(new_data.A, iA);
  Assign_Vector(new_data.B, iB);
  new_data.fmax = ifmax; //note: fmax > 0 must be true!
  data->push_front(new_data);
  if(data->size() > max_size) { data->pop_back(); } //TODO: better strategy of deleting cache entries [lwi]
}

bool IsoSurface::Cache::check(VECTOR A2, VECTOR B2, DBL max_gradient, const TraceThreadData* td)
{
  if(data.get() == NULL)
  { return false; }
  
  for(list<Data>::iterator i = data->begin(); i != data->end(); i++)
  {
	  td->Stats[Ray_IsoSurface_Cache]++;
	  
	  //it is safe to assume that fmax > 0
	  VECTOR T;
	  VSub(T, i->A, A2);
	  DBL length1 = VSumSqr(T);
	  VSub(T, i->B, B2);
	  DBL length2 = VSumSqr(T);
	  
	  if(Sqr(i->fmax) > Sqr(max_gradient) * max(length1, length2))
	  {
	    td->Stats[Ray_IsoSurface_Cache_Succeeded]++;
	    return true;
	  }
	}
	
	return false;
}
/*****************************************************************************
 *
 * FUNCTION
 *
 *   Bisect
 *
 * INPUT
 *
 * OUTPUT
 *
 * RETURNS
 *
 * AUTHOR
 *
 *   Lukas Winter
 *
 * DESCRIPTION
 *
 *   Finds the first root of an isosurface by bisecting the function along a ray
 *
 * CHANGES
 *   
 *   February 27, 2008: Initial creation. [lwi]
 *
 ******************************************************************************/
DBL IsoSurface::RayContext::Bisect(DBL t1, DBL t2)
{ 
  Pair sample1, sample2;
  sample1.t = t1;
  sample2.t = t2;
  
  sample1.f = (*this)(t1);
  
  /*if both values are negative, this will ensure they are now positive and
  btw it will ensure that we always look for a transition from positive to
  negative [lwi]*/
  DBL invert = (sample1.f<0)?-1:1;
  sample1.f *= invert;
  sample2.f = (*this)(t2) * invert;
  
  fmax = min(sample1.f, sample2.f);
  
  DBL delta_t = t2 - t1;
  DBL found;
  
  if(isosurf.eval && max_gradient > isosurf.eval_param[0])
  { max_gradient *= isosurf.eval_param[2]; }
  
  if(Bisect_Recurse(sample1, sample2, delta_t, invert, &found, false))
  {
    return found;
  }
  else
  {
    return BOUND_HUGE;
  }
}

bool IsoSurface::RayContext::Bisect_Recurse(const Pair& min_sample, const Pair& max_sample,
                                            DBL delta_t, const DBL& invert, DBL *found, bool skip)
{
  if(skip)
    return false;
  
  //if the interval is short enough
  if(delta_t < accuracy)
  {
    //have we bracketed a root?
    if(max_sample.f < 0)
    {
      //linear interpolation
      *found = min_sample.f * delta_t / (min_sample.f - max_sample.f) + min_sample.t;
      return true;
    }
    else
    {
      return false;
    }
  }
  
  DBL gradient = fabs((min_sample.f - max_sample.f) / delta_t);
  if(isosurf.eval && max_gradient < gradient * isosurf.eval_param[1])
  {
    max_gradient = gradient * isosurf.eval_param[1] * isosurf.eval_param[1];
  }
  
  //we are not close enough, so we have to carry on lookin'
  //can there be a root between the sample points (according to max_gradient)?
  if( min_sample.f + max_sample.f < delta_t * max_gradient)
  {
    bool skip_test1 = false;
    bool skip_test2 = false;
    DBL old_delta_t = delta_t;
    delta_t *= 0.474;
    /* The following code tries to move the split point to a location that leads
    to the narrowest possible interval while the other one cannot contain an intersection
    to make the approximation converge more quickly.
    This seems to be too expensive and has only a marginal effect.[lwi]
    */
    /*if(gradient < max_gradient)
    {
      DBL temp_delta_t = min_sample.f / max_gradient;
      if(temp_delta_t > delta_t)
      {
        skip_test1 = true;
        delta_t = temp_delta_t;
      }
      
      temp_delta_t = old_delta_t - fabs(max_sample.f) / max_gradient;
      if(temp_delta_t < delta_t)
      {
        skip_test2 = true;
        delta_t = temp_delta_t;
      }
    }*/
  
  	Pair mid_sample;
    mid_sample.t = min_sample.t + delta_t;
    mid_sample.f = (*this)(mid_sample.t) * invert;
    delta_t = mid_sample.t - min_sample.t;
    
    fmax = min(fmax, mid_sample.f);
    
    if(!Bisect_Recurse(min_sample, mid_sample, delta_t, invert, found, skip_test1))
    {
      
      //but I still haven't found what I'm looking for
      delta_t = max_sample.t - mid_sample.t;
      return Bisect_Recurse(mid_sample, max_sample, delta_t, invert, found, skip_test2);
    }
    else
    { return true; }
  }
  else
  { return false; }
}

}
