// $Id: cvector.cc,v 1.4 2001/08/05 22:30:19 peter Exp $

#ifndef CVECTOR_CC_
#define CVECTOR_CC_ 

#include "ctransf.hh"
#include "cvector.hh"
#include "cmatrix.hh"

int CVector::Count = 0;

CVector::CVector () {
	x = y = z = length = xnorm = ynorm = znorm = 0.0;
	Update();
	Number = ++Count;
}

CVector::CVector ( const CVector &V1 ) {
	x = V1.x; 
	y = V1.y; 
	z = V1.z;
	Update();
	Number = ++Count;
}
		
CVector::CVector ( double X1, double Y1, double Z1 ) { 
	x = X1; 
	y = Y1; 
	z = Z1;
	Update(); 
	Number = ++Count;
}
           
CVector::CVector ( double D1 ) { 
	x = y = z = D1;
	Update();
	Number = ++Count;
}

CVector::~CVector() {
	Count--;
}

void CVector::Update() {
	length = sqrt ( x*x + y*y + z*z);
	Normalize();
}

const double CVector::Length () const {
	return length;
}

const double CVector::X() const { 
	return x; 
}
		
const double CVector::Y() const { 
	return y; 
}

const double CVector::Z() const { 
	return z; 
}
		
const CVector CVector::Norm() const {
	return CVector(xnorm, ynorm, znorm);
}

void CVector::Set (double a, double b, double c) {
	x = a;
	y = b;
	z = c;
	Update();
}

void CVector::SetX (double a) {
	x = a;
	Update();
}

void CVector::SetY (double b) {
	y = b;
	Update();
}

void CVector::SetZ (double c) {
	z = c;
	Update();
}

void CVector::Normalize() { 
	if (fabs(length)>1e-12) { 
		xnorm = x/length;
		ynorm = y/length;
		znorm = z/length;
	}
}

CVector CVector::VCross ( const CVector &V1 ) {
	return CVector (y * V1.Z() - z * V1.Y(), 
	                z * V1.X() - x * V1.Z(),
	                x * V1.Y() - y * V1.X());
}
		
double CVector::VDot ( const CVector& V1) {
	return x*V1.X() + y*V1.Y() + z*V1.Z();
}

void CVector::VTransform ( const CTransform &T1 ) {
	CMatrix TempMatrix;

	TempMatrix = T1.Transform;

	x = x * TempMatrix.Elements[0][0] +
       y * TempMatrix.Elements[1][0] +
       z * TempMatrix.Elements[2][0] + 
           TempMatrix.Elements[3][0];

	y = x * TempMatrix.Elements[0][1] +
       y * TempMatrix.Elements[1][1] +
       z * TempMatrix.Elements[2][1] + 
           TempMatrix.Elements[3][1];

	z = x * TempMatrix.Elements[0][2] +
       y * TempMatrix.Elements[1][2] +
       z * TempMatrix.Elements[2][2] + 
           TempMatrix.Elements[3][2];

	Update();
}

void CVector::VInverseTransform ( const CTransform &T1 ) {
	CMatrix TempMatrix;

	TempMatrix = T1.Inverse;

	x = x * TempMatrix.Elements[0][0] +
       y * TempMatrix.Elements[1][0] +
       z * TempMatrix.Elements[2][0] + 
           TempMatrix.Elements[3][0];

	y = x * TempMatrix.Elements[0][1] +
       y * TempMatrix.Elements[1][1] +
       z * TempMatrix.Elements[2][1] + 
           TempMatrix.Elements[3][1];

	z = x * TempMatrix.Elements[0][2] +
       y * TempMatrix.Elements[1][2] +
       z * TempMatrix.Elements[2][2] + 
           TempMatrix.Elements[3][2];
	
	Update();
}
	
int CVector::Id() const {
	return Number;
}

void CVector::Print() const { 
/*
	cout << "Vector No. " << Number << ": <" 
	     << x << ", " << y << ", " << z << ">" << "\n"; 
*/
	cout.precision(8);
	cout << "<" << x << ", " << y << ", " << z << ">";
		
}

CVector CVector::operator * ( const double D1 ) const { 
	return CVector ( D1 * X(), D1 * Y(), D1 * Z() ); 
}
	
CVector CVector::operator / ( const double D1 ) { 
	return CVector ( X() / D1, Y() / D1, Z() / D1); 
}

CVector CVector::operator - () const {
	return CVector ( -X(), -Y(), -Z() );
}

double CVector::operator [] (const unsigned char i) const {
	switch (i) {
		case 0: return x;
		case 1: return y;
		case 2: return z;
		default:
			cout << "\n\aCVector::operator [] (const unsigned char) :\n"
			     << "    index out of range [0-2]";
			exit(1);
	}
}

CVector& CVector::operator = ( const CVector &V1 ) {
	x = V1.X();
	y = V1.Y();
	z = V1.Z();
	Update();
	return *this;
}	

CVector& CVector::operator *= ( const double D1) {
	x *= D1;
	y *= D1;
	z *= D1;
	Update();
	return *this;
}

CVector& CVector::operator *= ( const CVector &V1 ) {
	VCross (V1);
	Update();
	return *this;
}

CVector& CVector::operator /= ( const double D1 ) {
	x /= D1;
	y /= D1;
	z /= D1;
	Update();
	return *this;
}

CVector& CVector::operator += ( const CVector &V1 ) {
	x += V1.X();
	y += V1.Y();
	z += V1.Z();
	Update();
	return *this;
}

CVector& CVector::operator -= ( const CVector &V1 ) {
	x -= V1.X();
	y -= V1.Y();
	z -= V1.Z();
	Update();
	return *this;
}

CVector operator + ( const CVector &V1, const CVector &V2 ) { 
	return CVector ( V1.X() + V2.X(), V1.Y() + V2.Y(), V1.Z() + V2.Z() );
}

CVector operator - ( const CVector &V1, const CVector &V2 ) { 
	return CVector ( V1.X() - V2.X(), V1.Y() - V2.Y(), V1.Z() - V2.Z() ); 
}
		
CVector operator * ( const double D1, const CVector &V1 ) { 
	return CVector ( D1 * V1.X(), D1 * V1.Y(), D1 * V1.Z() ); 
}

CVector VCross ( const CVector &V1, const CVector &V2 ) {
	return CVector (V1.Y() * V2.Z() - V1.Z() * V2.Y(), 
			          V1.Z() * V2.X() - V1.X() * V2.Z(),
			          V1.X() * V2.Y() - V1.Y() * V2.X());
}
		
double VDot ( const CVector& V1, const CVector &V2) {
	return V1.X()*V2.X() + V1.Y()*V2.Y() + V1.Z()*V2.Z();
}

CVector VTransform ( const CVector &V1, const CTransform &T1 ) {
	CVector Result;
	CMatrix TempMatrix;

	TempMatrix = T1.Transform;

	Result.SetX( V1.X() * TempMatrix.Elements[0][0] +
                V1.Y() * TempMatrix.Elements[1][0] +
                V1.Z() * TempMatrix.Elements[2][0] + 
                         TempMatrix.Elements[3][0] );

	Result.SetY( V1.X() * TempMatrix.Elements[0][1] +
                V1.Y() * TempMatrix.Elements[1][1] +
                V1.Z() * TempMatrix.Elements[2][1] + 
                         TempMatrix.Elements[3][1] );

	Result.SetZ( V1.X() * TempMatrix.Elements[0][2] +
                V1.Y() * TempMatrix.Elements[1][2] +
                V1.Z() * TempMatrix.Elements[2][2] + 
                         TempMatrix.Elements[3][2] );

	return Result;
}

CVector VInverseTransform ( const CVector &V1, const CTransform &T1 ) {
	CVector Result;
	CMatrix TempMatrix;

	TempMatrix = T1.Inverse;

	Result.SetX( V1.X() * TempMatrix.Elements[0][0] +
                V1.Y() * TempMatrix.Elements[1][0] +
                V1.Z() * TempMatrix.Elements[2][0] + 
                         TempMatrix.Elements[3][0] );

	Result.SetY( V1.X() * TempMatrix.Elements[0][1] +
                V1.Y() * TempMatrix.Elements[1][1] +
                V1.Z() * TempMatrix.Elements[2][1] + 
                         TempMatrix.Elements[3][1] );

	Result.SetZ( V1.X() * TempMatrix.Elements[0][2] +
                V1.Y() * TempMatrix.Elements[1][2] +
                V1.Z() * TempMatrix.Elements[2][2] + 
                         TempMatrix.Elements[3][2] );
	return Result;
}
		
#endif

// $Log: cvector.cc,v $
// Revision 1.4  2001/08/05 22:30:19  peter
// Added indexing operator [] (read-only)
// Fixed typo in VTransform
//
// Revision 1.3  2001/07/31 18:42:03  peter
// Finally added some comments. Finalized CVector (I think...)
//
// Revision 1.2  2001/07/29 06:17:28  peter
// Finally moved all member and non-member functions outside the header file.
//
// Revision 1.1  2001/07/27 12:51:50  peter
// VCross and VDot are no longer membre functions of class CVector. It was a
// bug.
//
