// $Id: cmesh.hh,v 1.5 2001/08/05 22:27:49 peter Exp $

#ifndef CMESH_HH_
#define CMESH_HH_

#include "cvector.hh"
#include "cforce.hh"
#include "cray.hh"
#include "cobject.hh"
#include "cnode.hh"
#include "clink.hh"
#include <vector>
#include <stack>
#include <list>

// The mesh is the main class of interest. It contains all the information and
// functionality to calculate an iteration.
// All nodes, links, external forces and constraints are stored in a mesh and
// only pointers of them are sent to links, nodes and elements. This is due to
// the fact that there's a lot of sharing of the former among the latter and
// there's no simple way to implement this sharing other than pointers. The
// mesh will take care of the memory its nodes, links, external forces and
// constraints take.
// Note that a mesh doesn't actually contain the objects that form the environment.
// Rather, this is up to the frame to think of, since several meshes may exist
// in the same environment.
class CMesh
{
	private:

		// All nodes, links, external,  forces and constraints, as well as 
		// pointers to all objects in the environment.
		// I may later decide to use a container different than a vector later
		vector <CNode> Nodes;
		vector <CLink> Links;
		vector <CForce> External;
		vector <CForce> Constraints;
		list <CObject *> Objects;

		// Add a link, node, constraint or external force to the mesh data.
		// These are private as they only make sense when used by member functions.
		void AddLink (const CLink &);
		void AddNode (const CNode &);
		void AddConstraint (const CForce &);
		void AddExternal (const CForce &);
		
		// Attach a link, constraint or external force to a node. Duplication is 
		// avoided.
		void AttachLink (CNode, CLink);
		void AttachConstraint (CNode, CForce);
		void AttachExternal (CNode, CForce);

		// These are used for debugging as well as for identification.
		static int Count;
		int Number;
	public:
		CMesh ();

		~CMesh();

		// Thanks go to Warp for the template idea. Passing pointers around as
		// if I was working in C was a real nightmare!
		template <typename TObject>
		void AddObject (const TObject &Object) {
			CObject *tmp = new TObject(Object);
			Objects.push_back(tmp);
		}

		// Ray-shape intersection with the list of objects available to the mesh.
		// This will be used in the collision detection routine later.
		CIntersection Trace ( const CRay & ) const ;

		// These are used for debugging as well as for identification.
		int Id();
		void Print();
};

#endif

// $Log: cmesh.hh,v $
// Revision 1.5  2001/08/05 22:27:49  peter
// The destructor now detroys all objects the mesh uses
//
// Revision 1.4  2001/07/31 18:42:03  peter
// Finally added some comments. Finalized CVector (I think...)
//
// Revision 1.3  2001/07/29 06:19:01  peter
// Added an object counter. object ID and print method to each class. Fixed
// various annoying little bugs.
//
// Revision 1.2  2001/07/23 07:33:24  peter
// Added CVS information
//
// Revision 1.1  2001/07/23 07:11:51  peter
// Created
