// $Id: cvector.hh,v 1.9 2001/08/05 22:30:19 peter Exp $

#ifndef CVECTOR_HH_
#define CVECTOR_HH_ 

#include <iostream>

class CTransform;

class CVector {
	protected:
		double x, y, z;

		// Store the length and normalized direction to avoid calculating it every time
		double xnorm, ynorm, znorm, length;

		// The following to variables are used mostly for debugging
		// and optimisation purposes - keep track of object creation
		static int Count;
		int Number;

		// Called on creation or change to update xnorm, ynorm, znorm and length
		void Update();

	public:

		// Create a zero vector
		CVector ();

		// Copy constructor needed to update counter
		CVector ( const CVector & );

		// The most commonly used constructor
		CVector ( double, double, double ); 

		// Duplicate the way POV-Ray casts a double to a vector.
		// It is not matheatically correct but saves typing.
		// CVector (a) is equivalent to CVector (a, a, a)
		CVector ( double ); 
		
		// Keep track of the object count
		~CVector();

		// If these aren't straight forward you should be doing something else :)
		const double Length() const;
		const double X() const;
		const double Y() const;
		const double Z() const;
		const CVector Norm() const;
		
		// A quicker way than using 'operator =' (I think)
		void Set (double a, double b, double c);

		// Shortcuts if you only want one coordinate changed
		void SetX (double a);
		void SetY (double b);
		void SetZ (double c);

		// Bet you'll never figure this one out...
		void Normalize(); 
	
		// The following four functions are available as both member functions and regular 
		// functions depending on which is faster in a specific situation.

		// Compute the dot (scalar) product of two vectors
		double VDot ( const CVector & );

		// Compute the cross (vector) product of two vectors
		CVector VCross ( const CVector & );

		// Transform a vector using a matrix transformation
		// See the comments in ctransf.hh
		void VTransform ( const CTransform & );
		void VInverseTransform ( const CTransform & );
	
		// Need to be friends with CVector because the lefthand parameter is not 'this'.
		// Instead, it is a double, a CVector or a double casted to a CVector
		// via the CVector (double) constructor (read comments there)
		friend CVector operator + ( const CVector &, const CVector & ); 
		friend CVector operator - ( const CVector &, const CVector & ); 
		friend CVector operator * ( const double, const CVector & ); 
		
		// Vector scaling
		CVector operator * ( const double ) const; 

		// Vector inverse scaling
		CVector operator / ( const double ); 

		// Negation 
		CVector operator - () const;

		// Indexing
		double operator [] (const unsigned char) const;

		CVector& operator = ( const CVector &);

		// Vector scaling
		CVector& operator *= ( const double );

		// Vector inverse scaling
		CVector& operator /= ( const double );

		// Vector addition and subtraction. See comments to CVector ( double ).
		CVector& operator += ( const CVector & );
		CVector& operator -= ( const CVector & );

		// Cross product
		CVector& operator *= ( const CVector & );
		
		// These are mostly used for debugging
		int Id() const;
		void Print() const; 
};

// The following four functions are available as both member functions and regular 
// functions depending on which is faster in a specific situation

// Compute the dot (scalar) product of two vectors
double VDot ( const CVector &, const CVector &);

// Compute the cross (vector) product of two vectors
CVector VCross ( const CVector &, const CVector & );

// Transform a vector using a matrix transformation
// See the comments in ctransf.hh
CVector VTransform ( const CVector &, const CTransform & );
CVector VInverseTransform ( const CVector &, const CTransform & );

#endif

// $Log: cvector.hh,v $
// Revision 1.9  2001/08/05 22:30:19  peter
// Added indexing operator [] (read-only)
// Fixed typo in VTransform
//
// Revision 1.8  2001/07/31 18:42:03  peter
// Finally added some comments. Finalized CVector (I think...)
//
// Revision 1.7  2001/07/29 06:17:28  peter
// Finally moved all member and non-member functions outside the header file.
//
// Revision 1.6  2001/07/27 13:20:39  peter
// Fixed some CVS-introduced ambiguities.
//
// Revision 1.5  2001/07/27 12:53:07  peter
// Took VCross and VDot out of CVector.
// FIXME: All overloaded operators of class CVector should be defined in
// cvector.cc
//
// Revision 1.4  2001/07/23 07:11:51  peter
// Major changes:
//   * Extracted each class into its own header and source
//
// Minor changes:
//
// Revision 1.3  2001/07/22 23:06:50  peter
// Added a real CForce class. Tested for interchangeability with CVector.
// Seems to work OK. Removed the ugly #define from cnode.hh
//
// Revision 1.2  2001/07/22 20:59:01  peter
// Tested CVector - seems to work
//
// Revision 1.1  2001/07/22 20:44:12  peter
// Initial version in CVS
//
