// $Id: cbox.cc,v 1.1 2001/08/05 22:24:58 peter Exp $

#ifndef CBOX_CC_
#define CBOX_CC_

#include "consts.hh"
#include "cvector.hh"
#include "cray.hh"
#include "cobject.hh"
#include "cintrsct.hh"
#include "cbox.hh"
#include "ctransf.hh"
#include <iostream>

CBox::CBox () {
	lower = 0;
	upper = 0;
}

CBox::CBox (const CVector &V1, const CVector &V2) {
	double v1x, v1y, v1z, v2x, v2y, v2z;
	double minx, miny, minz, maxx, maxy, maxz;
	
	v1x = V1.X(); v1y = V1.Y(); v1z = V1.Z();
	v2x = V2.X(); v2y = V2.Y(); v2z = V2.Z();

	if ( v1x < v2x ) {
		minx = v1x; maxx = v2x;
	}
	else {
		minx = v2x; maxx = v1x;
	}

	if ( v1y < v2y ) {
		miny = v1y; maxy = v2y;
	}
	else {
		miny = v2y; maxy = v1y;
	}

	if ( v1z < v2z ) {
		minz = v1z; maxz = v2z;
	}
	else {
		minz = v2z; maxz = v1z;
	}

	lower = CVector (minx, miny, minz);
	upper = CVector (maxx, maxy, maxz);
}

CBox::~CBox () {
	// Make the compiler happy
}

bool CBox::Inside (const CVector &V1) const {
	CVector tmp = VInverseTransform(V1, transform);
	double x = tmp.X(), y = tmp.Y(), z = tmp.Z();
	double lx = lower.X(), ly = lower.Y(), lz = lower.Z();
	double ux = upper.X(), uy = upper.Y(), uz = upper.Z();
	
	return ( (x >= lx) && (x <= ux) && (y >= ly) && (y <= uy) && (z >= lz) && (z <= uz) );
}

CVector CBox::NormalAt (const CVector &V1) const {
	CVector middle = 0.5 * (lower + upper);
	CVector offset = VInverseTransform(V1 - middle, transform);
	CVector size = upper - lower;

	CVector Result;
	
	if ( ( offset.X() / size.X() ) > ( offset.Y() / size.Y() ) )
		if ( ( offset.X() / size.X() ) > ( offset.Z() / size.Z() ) )
			if ( offset.X() > 0 )
				Result = CVector (1,0,0);
			else
				Result = CVector (-1,0,0);
		else
			if ( offset.Z() > 0 )
				Result = CVector (0,0,1);
			else
				Result = CVector (0,0,-1);
	else
		if ( ( offset.Y() / size.Y() ) > ( offset.Z() / size.Z() ) )
			if ( offset.Y() > 0 )
				Result = CVector (0,1,0);
			else
				Result = CVector (0,-1,0);
		else
			if ( offset.Z() > 0 )
				Result = CVector (0,0,1);
			else
				Result = CVector (0,0,-1);

	Result.VTransform(transform);
	return Result.Norm();
}

IStack CBox::Intersect (const CRay &R1) const {
	double current, min = 0.0, max = SOMETHING_BIG;	
	unsigned char i;

	CVector O = VInverseTransform(R1.Origin(),transform);
	CVector D = VInverseTransform(R1.Direction(),transform);

	IStack Intersections;

	// The tests are identical for each pair of opposing sides
	for (i = 0; i < 3; i++) {
		if (D[i] < -SOMETHING_SMALL) {
			current = (lower[i] - O[i]) / D[i];

			// Preliminary test if the ray points in the general direction of the box
			if (current < min)
				return Intersections;

			// Check if the intersection with the more distant of the two opposing sides
			// lies closer along the ray than the currently found most distant one
			if (current <= max)
				max = current;

			current = (upper[i] - O[i]) / D[i];

			if (current >= min) {
				if (current > max) 
					return Intersections;
				min = current;
			}
		}

		// D[i] >= SOMETHING_SMALL
		else 
			if (D[i] > SOMETHING_SMALL) {
				current = (upper[i] - O[i]) / D[i];

				// Preliminary test if the ray points in the general direction of the box
				if (current < min)
					return Intersections;

				if (current <= max)
					max = current;

				current = (lower[i] - O[i]) / D[i];

				if (current >= min) {
					if (current > max)
						return Intersections;
					min = current;
				}
			}
			
			// D[i] is within [-SOMETHING_SMALL; SOMETHING_SMALL], i.e. The ray is parallel
			// to the i-tieth pair of sides
			else
				if ( (O[i] < lower[i]) || (O[i] > upper[i] ) )
					// The origin of the ray is outside the box, no intersection possible
					return Intersections;
	}

	if (min > SOMETHING_SMALL) {
		Intersections.push( CIntersection( min, this ) );
	}	
	
	if (max < SOMETHING_BIG) {
		Intersections.push( CIntersection( max, this ) );
	}

	return Intersections;
}

void CBox::Print() const {
	cout << "box { ";
	lower.Print();
	cout << " , ";
	upper.Print();
	cout << " }\n";
}

#endif

// $Log: cbox.cc,v $
// Revision 1.1  2001/08/05 22:24:58  peter
// Added transformation related member functions to CObject and derived.
// Added plane object ( class CPlane : public CObject )
// Intersect, Inside and NormalAt member functions take into account
// object transformations. FIXME: Not all objects do this yet.
//
