/****************************************************************************
*                mechsim.cpp
*
*  Author: Christoph Hormann <chris_hormann@gmx.de>
*
*  Mechanics simulation
*
*  Copyright 2002 Christoph Hormann
*
*  MechSim version 0.2.0   December 2002
*
*****************************************************************************
*
*  from Persistence of Vision(tm) Ray Tracer
*  Copyright 1996-2002 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file.
*  If POVLEGAL.DOC is not available it may be found online at -
*
*    http://www.povray.org/povlegal.html.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
*###################################################################
*  This file is part of MegaPOV, which is a modified and unofficial version of POV-Ray
*
* $RCSfile: mechsim.cpp,v $
* $Revision: 1.12 $
* $Author: chris $
* $Log: mechsim.cpp,v $
* Revision 1.12  2002/12/20 10:23:04  chris
* fixed problem in the Triangle_Distribute() function
*
* Revision 1.11  2002/12/16 14:09:35  chris
* fixed bug in format 3 file writing (start time was the same as when reading)
* fixed bug in gradient descent collisions (forces on both masses were identical
*
* Revision 1.10  2002/12/08 17:49:58  chris
* added instability check
*
* Revision 1.9  2002/12/07 16:48:43  smellenbergh
* Adds listed pattern, displacement warp
* Header CVS keywords fix
*
* Revision 1.8  2002/12/06 11:39:23  abx
* "no warning" corrections
*
* Revision 1.7  2002/12/05 20:57:25  chris
* merging the mechsim changes after version 0.1.0:
* attachments, interactions, fields and various other changes
* removed unused variables
*
* Revision 1.6  2002/12/03 08:50:13  smellenbergh
* syntax corrections
*
*
* Revision 1.2  2002/12/02 14:41:09  abx
* clothray and mechsim adjusted to 'no warning' version,
* frame_step is working switch and script build-in token but not influence animation rendering,
* CLASSNAMEPREFIX in windows code is MegaPOV now
*
* Revision 1.1  2002/12/02 00:03:27  smellenbergh
* Adds Mechanics Simulation Patch and Clothray Patch
*
*****************************************************************************/

#include "frame.h"
#include "povray.h"
#include "vector.h"
#include "povproto.h"
#include "function.h"
#include "fnpovfpu.h"
#include "file_pov.h"
#include "mesh.h"
#include "ray.h"
#include "objects.h"

#include "mechsim.h"

#include <algorithm>

#ifdef MECHANICS_SIMULATION_PATCH            /* Christoph Hormann August 2002 */

/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

#define MAX_STR 255

#define READ_INTEGER(var) \
if ( fd->getline (line, 99).eof () ) { \
  Warning(0, "\nError reading mechanics simulation data (wrong file format)"); \
  return false; \
} \
if (sscanf(line, "%d,\n", &var) != 1) { \
  Warning(0, "\nError reading mechanics simulation data (wrong file format)"); \
  return false; \
}


#define MECHSIM_GRADIENT_ACCURACY 0.001

#define MECHSIM_EPSILON 0.0001

/*****************************************************************************
* Local functions
******************************************************************************/

void MechSimFree_Rec(MECHSIM_ENVIRONMENT *PEnv);

void Function_Gradient(FUNCTION funct, VECTOR Pos, VECTOR grad);
int func_triangle(VECTOR Point, VECTOR CPoint, VECTOR Grad, VECTOR P1, VECTOR P2, VECTOR P3, DBL *Ratio);
void Triangle_Distribute(VECTOR Point, VECTOR P1, VECTOR P2, VECTOR P3, DBL *Weight1, DBL *Weight2, DBL *Weight3);

void Mechsim_Impact_Collide_Environment(int Index, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_ENVIRONMENT *Env);
void Mechsim_Gradient_Collide_Environment(int Index, VECTOR Gradient, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_ENVIRONMENT *Env);
void Mechsim_Force_Collide_Environment(int Index, VECTOR Accel,VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_ENVIRONMENT *Env);
void Mechsim_Gradient_Collide_Dynamic(VECTOR *Gradient, VECTOR *Pos_Array, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
  bounding_hashtable, int bounding_hashtable_size, int time_stamp);
void Mechsim_Force_Collide_Dynamic(VECTOR *Accel, VECTOR *Pos_Array, VECTOR *Vel_Array, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
  bounding_hashtable, int bounding_hashtable_size, int time_stamp);
void Mechsim_Gradient_Connect(VECTOR *Gradient, VECTOR *Pos_Array);
void Mechsim_Force_Connect(VECTOR *Accel, VECTOR *Pos_Array, VECTOR *Vel_Array);
void Mechsim_Gradient_Interactions(VECTOR *Accel, VECTOR *Pos_Array);
void Mechsim_Force_Interactions(VECTOR *Accel, VECTOR *Pos_Array);
void Mechsim_Force_Fields(int Index, VECTOR Accel, VECTOR *Pos_Array);
void Mechsim_Gradient_Fields(int Index, VECTOR Accel, VECTOR *Pos_Array);
void Mechsim_Gradients(VECTOR *Gradient, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
  bounding_hashtable, int bounding_hashtable_size, int time_stamp);
void Mechsim_Forces(VECTOR *Accel, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
  bounding_hashtable, int bounding_hashtable_size, int time_stamp);
void Mechsim_Check_Col(VECTOR Pos, VECTOR Old_Pos);
void Mechsim_Check_Instability(VECTOR Pos, VECTOR Vel, VECTOR Accel);
void Mechsim_Create_Groups(int Count);
void Mechsim_Prepare_Data(void);
bool Mechsim_save_file (void);

char *Read_Vector(char *buffer, VECTOR Vect);
char *Read_Float(char *buffer, double *Value);
char *Read_Int(char *buffer, int *Value);

static DBL calc_faces_bounding_box(VECTOR* Pos_Array, MECHSIM_BBOX* Faces_BBox_Array);
static bool intersect_sphere_box(VECTOR Box_Min, VECTOR Box_Max, VECTOR Center, DBL Radius);
static void calc_faces_bounding_hashtable(BOUNDING_HASHTABLE_ITEM* bounding_hashtable, DBL Unit, int bounding_hashtable_size, int time_stamp,    
  MECHSIM_BBOX* Faces_BBox_Array);
static void insert_index_bounding_hastable(BOUNDING_HASHTABLE_ITEM* bounding_hashtable, unsigned int hash, int index, int time_stamp);
static void intersect_sphere_bounding_hashtable(BOUNDING_HASHTABLE_ITEM* bounding_hashtable, VECTOR Center, DBL Radius, 
  DBL Unit, int** list, int* size, int* index, int bounding_hashtable_size, int time_stamp);
static int CDECL compare_int (const void *in_a, const void *in_b);
static unsigned int bounding_hash_func(long int x, long int y, long int z, long int n);

/* ------------------------------------------------------ */
/* global variables */
/* ------------------------------------------------------ */

MECHSIM_OPTIONS MechsimOptions;

/* ------------------------------------------------------ */
/* external variables */
/* ------------------------------------------------------ */

/* ------------------------------------------------------ */
/* static functions */
/* ------------------------------------------------------ */

/* ------------------------------------------------------ */
/* static variables */
/* ------------------------------------------------------ */

/*****************************************************************************
*
* FUNCTION
*
*   Initialize_MechSim
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Set default values of simulation data structure
*
* CHANGES
*
*   --- Aug 2002 : Creation
*
******************************************************************************/
void Initialize_MechSim(void)
{
  /* Initialize 'MechsimOptions' fields */

  MechsimOptions.Method = 1;
  MechsimOptions.Step_Count = 10;
  MechsimOptions.Time_Step = 0.1;
  MechsimOptions.Start_Time = 0.0;
  Make_Vector(MechsimOptions.gravity, 0.0, 0.0, 0.0);
  MechsimOptions.Damping = 0.0;
  MechsimOptions.Environment = NULL;
  MechsimOptions.Collision_Stiffness = 0.0;
  MechsimOptions.Collision_Damping = 0.0;
  MechsimOptions.Calc_Collision = MECHSIM_COLLISION_NONE;
  MechsimOptions.Calc_Collision_Faces = MECHSIM_COLLISION_NONE;
  MechsimOptions.Load_File_Name = NULL;
  MechsimOptions.Save_File_Name = NULL;
  MechsimOptions.Save_File = false;
  MechsimOptions.Save_File_Type = 3;
  MechsimOptions.Data.mass_count = 0;
  MechsimOptions.Data.connect_count = 0;
  MechsimOptions.Data.faces_count = 0;
  MechsimOptions.Data.group_count = 0;

  MechsimOptions.Data.max_masses = 0;
  MechsimOptions.Data.max_connects = 0;
  MechsimOptions.Data.max_faces = 0;
  MechsimOptions.Data.max_groups = 0;

  MechsimOptions.Enabled = false;

  MechsimOptions.Interactions.Function = NULL;
  MechsimOptions.Interactions.count = 0;
  MechsimOptions.Interactions.max_count = 0;

  MechsimOptions.Attachments.Function = NULL;
  MechsimOptions.Attachments.count = 0;
  MechsimOptions.Attachments.max_count = 0;

  MechsimOptions.Fields.Function = NULL;
  MechsimOptions.Fields.count = 0;
  MechsimOptions.Fields.max_count = 0;
  MechsimOptions.Bounding = MECHSIM_FACE_BOUNDING_NO;
}

/*****************************************************************************
*
* FUNCTION
*
*   MechSimFree_Rec
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Recursive function to free Environment data fields
*
* CHANGES
*
*   --- Oct 2002 : Creation
*
******************************************************************************/
void MechSimFree_Rec(MECHSIM_ENVIRONMENT *PEnv)
{
  MECHSIM_ENVIRONMENT *Env;

  Env = PEnv->Next;

  if (Env != NULL)
  {
    MechSimFree_Rec(Env);

    POV_FREE(Env);
    PEnv->Next = NULL;
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Deinitialize_MechSim
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Free all simulation data
*
* CHANGES
*
*   --- Aug 2002 : Creation

******************************************************************************/
void Deinitialize_MechSim(void)
{
  int i;

  if (MechsimOptions.Enabled)
  {
    if (MechsimOptions.Data.max_masses>0) POV_FREE(MechsimOptions.Data.Masses);
    if (MechsimOptions.Data.max_connects>0) POV_FREE(MechsimOptions.Data.Connects);
    if (MechsimOptions.Data.max_faces>0) POV_FREE(MechsimOptions.Data.Faces);
    if (MechsimOptions.Data.max_groups>0)
    {
      for(i=0; i<MechsimOptions.Data.max_groups; i++)
      {
        POV_FREE(MechsimOptions.Data.Groups[i].Masses);
        POV_FREE(MechsimOptions.Data.Groups[i].Connects);
        POV_FREE(MechsimOptions.Data.Groups[i].Faces);
      }

      POV_FREE(MechsimOptions.Data.Groups);
    }

    MechsimOptions.Data.max_masses = 0;
    MechsimOptions.Data.max_connects = 0;
    MechsimOptions.Data.max_faces = 0;
    MechsimOptions.Data.max_groups = 0;

    if (MechsimOptions.Interactions.max_count>0) POV_FREE(MechsimOptions.Interactions.Function);
    if (MechsimOptions.Attachments.max_count>0) POV_FREE(MechsimOptions.Attachments.Function);
    if (MechsimOptions.Fields.max_count>0) POV_FREE(MechsimOptions.Fields.Function);

    MechsimOptions.Attachments.max_count = 0;
    MechsimOptions.Attachments.count = 0;
    MechsimOptions.Interactions.max_count = 0;
    MechsimOptions.Interactions.count = 0;
    MechsimOptions.Fields.max_count = 0;
    MechsimOptions.Fields.count = 0;

    MechsimOptions.Enabled = false;
  }

  /* Environments */
  if (MechsimOptions.Environment != NULL)
  {
    MechSimFree_Rec(MechsimOptions.Environment);
    POV_FREE(MechsimOptions.Environment);
    MechsimOptions.Environment = NULL;
  }

}

/*****************************************************************************
*
* FUNCTION
*
*   Function_Gradient
*
* INPUT
*
*   funct - function to calculate gradient from
*   Pos - evaluation point
*
* OUTPUT
*
*   the gradient is returned via 'grad'
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates the gradient of a function at a specified point.
*
*   Works just like the corresponding macro in 'math.inc'
*
* CHANGES
*
*   --- Aug 2002 : Creation
*
******************************************************************************/

void Function_Gradient(FUNCTION funct, VECTOR Pos, VECTOR grad)
{

  POVFPU_SetLocal(X, Pos[X] + MECHSIM_GRADIENT_ACCURACY);
  POVFPU_SetLocal(Y, Pos[Y]);
  POVFPU_SetLocal(Z, Pos[Z]);
  grad[X] = POVFPU_Run(funct);
  POVFPU_SetLocal(X, Pos[X] - MECHSIM_GRADIENT_ACCURACY);
  POVFPU_SetLocal(Y, Pos[Y]);
  POVFPU_SetLocal(Z, Pos[Z]);
  grad[X] -= POVFPU_Run(funct);
  grad[X] /= 2.0*MECHSIM_GRADIENT_ACCURACY;

  POVFPU_SetLocal(X, Pos[X]);
  POVFPU_SetLocal(Y, Pos[Y] + MECHSIM_GRADIENT_ACCURACY);
  POVFPU_SetLocal(Z, Pos[Z]);
  grad[Y] = POVFPU_Run(funct);
  POVFPU_SetLocal(X, Pos[X]);
  POVFPU_SetLocal(Y, Pos[Y] - MECHSIM_GRADIENT_ACCURACY);
  POVFPU_SetLocal(Z, Pos[Z]);
  grad[Y] -= POVFPU_Run(funct);
  grad[Y] /= 2.0*MECHSIM_GRADIENT_ACCURACY;

  POVFPU_SetLocal(X, Pos[X]);
  POVFPU_SetLocal(Y, Pos[Y]);
  POVFPU_SetLocal(Z, Pos[Z] + MECHSIM_GRADIENT_ACCURACY);
  grad[Z] = POVFPU_Run(funct);
  POVFPU_SetLocal(X, Pos[X]);
  POVFPU_SetLocal(Y, Pos[Y]);
  POVFPU_SetLocal(Z, Pos[Z] - MECHSIM_GRADIENT_ACCURACY);
  grad[Z] -= POVFPU_Run(funct);
  grad[Z] /= 2.0*MECHSIM_GRADIENT_ACCURACY;

}


/*****************************************************************************
*
* FUNCTION
*
*   func_triangle
*
* INPUT
*
*   Point - point to evaluate the triangle function at
*   P1, P2, P3 - vertices of the triangle
*
* OUTPUT
*
*   the distance of Point to the specified triangle
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates the distance of a point to a triangle.
*
*   Derieved from the internal function 'f_triangle' developed by ABX
*
* CHANGES
*
*   --- Sep 2002 : Creation
*
******************************************************************************/

int func_triangle(VECTOR Point, VECTOR CPoint, VECTOR Grad, VECTOR P1, VECTOR P2, VECTOR P3, DBL *Ratio)
{
        VECTOR v_temp,N,R1,R2,R3,E;
        DBL Nd,Ed,El, Dist;
        //VECTOR Distance;
        int Result;

        VSub(R1,P2,P1);
        VSub(R2,P3,P2);
        VSub(R3,P1,P3);
        VNormalizeCrossNotZeroLength(N,R2,R3);
        VNormalizeCrossNotZeroLength(E,N,R1);
        VDot(Ed,E,P1);
        VDot(El,E,Point);
        if ( Ed < El )
        {
          VNormalizeCrossNotZeroLength(E,N,R2);
          VDot(Ed,E,P2);
          VDot(El,E,Point);
          if ( Ed < El )
          {
            VNormalizeCrossNotZeroLength(E,N,R3);
            VDot(Ed,E,P3);
            VDot(El,E,Point);
            if ( Ed < El )
            {
              VDot(Nd,N,P1);
              VDot(Dist,N,Point);
              //Distance = fabs( Distance-Nd );
              VScale(Grad, N, Dist-Nd);
              VAddScaled(CPoint, Point, Nd-Dist, N);
              Result = 0;
            }
            else
            {
              //DistanceToEdge( Distance , R3 , P3 , Point );
              VToEdge( Grad , R3 , P3 , Point );
              VSub(v_temp, Point, P3);
              VNormalizeEq(R3);
              VDot(*Ratio,v_temp,R3);
              Result = 3;
            }
          }
          else
          {
            //DistanceToEdge( Distance , R2 , P2 , Point );
            VToEdge( Grad , R2 , P2 , Point );
            VSub(v_temp, Point, P2);
            VNormalizeEq(R2);
            VDot(*Ratio,v_temp,R2);
            Result = 2;
          }
        }
        else
        {
          //DistanceToEdge( Distance , R1 , P1 , Point );
          VToEdge( Grad , R1 , P1 , Point );
          VSub(v_temp, Point, P1);
          VNormalizeEq(R1);
          VDot(*Ratio,v_temp,R1);
          Result = 1;
        }

        return Result;
}

/*****************************************************************************
*
* FUNCTION
*
*   Triangle_Distribute
*
* INPUT
*
*   Point - impact point
*   P1, P2, P3 - vertices of the triangle
*
* OUTPUT
*
*   Weight1, Weight2, Weight3 - factors for weighting
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Weight the force on the triangle vertices according to static equilibrium
*
* CHANGES
*
*   --- Sep 2002 : Creation
*   --- Dec 2002 : Used a simpler method and probably fixed some problems
*
******************************************************************************/
void Triangle_Distribute(VECTOR Point, VECTOR P1, VECTOR P2, VECTOR P3, DBL *Weight1, DBL *Weight2, DBL *Weight3)
{
  DBL DSqSum1, DSqSum2, DSqSum3;
  VECTOR Dir1, Dir2, Dir3;
  DBL Val1, Val2, Val3;
  DBL Scale;

  /*
   * The method used here calculates the distance of the collision point
   * to the opposite side of the triangle.  With normalized scaling (i.e. 
   * distances between 0 and 1 the sum of all distances has to be 1.0.  
   * We just have to scale the distances with their sum and get correct 
   * weights for the force.
   */

  VSub(Dir1, P2, P1);
  DSqSum1 = VSumSqr(Dir1);

  VSub(Dir2, P3, P2);
  DSqSum2 = VSumSqr(Dir2);

  VSub(Dir3, P3, P1);
  DSqSum3 = VSumSqr(Dir3);

  /* calculate the distance of the collision point to each side */
  Val1=(Point[X] - P1[X])*Dir1[Y] - (Point[Y] - P1[Y])*Dir1[X];
  Val2=(Point[Y] - P1[Y])*Dir1[Z] - (Point[Z] - P1[Z])*Dir1[Y];
  Val3=(Point[Z] - P1[Z])*Dir1[X] - (Point[X] - P1[X])*Dir1[Z];

  (*Weight1) = sqrt((Val1*Val1 + Val2*Val2 + Val3*Val3)/DSqSum1);

  Val1=(Point[X] - P2[X])*Dir2[Y] - (Point[Y] - P2[Y])*Dir2[X];
  Val2=(Point[Y] - P2[Y])*Dir2[Z] - (Point[Z] - P2[Z])*Dir2[Y];
  Val3=(Point[Z] - P2[Z])*Dir2[X] - (Point[X] - P2[X])*Dir2[Z];

  (*Weight2) = sqrt((Val1*Val1 + Val2*Val2 + Val3*Val3)/DSqSum2);

  Val1=(Point[X] - P1[X])*Dir3[Y] - (Point[Y] - P1[Y])*Dir3[X];
  Val2=(Point[Y] - P1[Y])*Dir3[Z] - (Point[Z] - P1[Z])*Dir3[Y];
  Val3=(Point[Z] - P1[Z])*Dir3[X] - (Point[X] - P1[X])*Dir3[Z];

  (*Weight3) = sqrt((Val1*Val1 + Val2*Val2 + Val3*Val3)/DSqSum3);

  /* calculate the sum of weights */
  Scale = 1.0/((*Weight1) + (*Weight2) + (*Weight3));

  /* scale the weights */
  (*Weight1) *= Scale;
  (*Weight2) *= Scale;
  (*Weight3) *= Scale;

  //Status_Info("\n%g Weight=<%g, %g, %g>\n", (*Weight1) + (*Weight2) + (*Weight3), (*Weight1), (*Weight2), (*Weight3) );

}


/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Impact_Collide_Environment
*
* INPUT
*
*   Index - index of the mass
*   Pos_Array, Vel_Array - state of the masses
*   Time - current time index for the environment function
*   Env - Pointer to Environment
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates collision with the environment based on impact laws.
*
*   * If an environment function is given, it is evaluated and if
*     collision is found the gradient is calculated and the velocity
*     is mirrored at the surface and the position is updated.
*   * If only an object is given a ray is shot from the current position
*     in movement direction to determine the collision.
*
*   Friction and damping are applied as diminishing factors (see also
*   Christophe Bouffartique's cloth simulation)
*
*   ========== WARNING ====================================================
*
*   This does not necessarily work well with all integration methods,
*   The state of the system (i.e. positions and velocities) is modified
*   in this function instead of applying accelerations like we should do.
*
*   The object based method is also problematic with higher order
*   integration methods.  Apart from that it won't successfully find the
*   intersection in all situations like the function based method.
*
*   =======================================================================
*
* CHANGES
*
*   --- Sep 2002 : Creation
*       Oct 2002 : added object intersection based method
*
******************************************************************************/

void Mechsim_Impact_Collide_Environment(int Index, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_ENVIRONMENT *Env)
{
  DBL Norm_Len, Fn_Val;
  VECTOR Grad;

  if (Env->Function != NULL)  // ======== if function is given use it to calculate gradient ===========
  {
    FunctionCode *f = POVFPU_GetFunction(*Env->Function);

    // ---- collision test ----
    POVFPU_SetLocal(X, Pos_Array[Index][X]);
    POVFPU_SetLocal(Y, Pos_Array[Index][Y]);
    POVFPU_SetLocal(Z, Pos_Array[Index][Z]);
    if (f->parameter_cnt > 3) POVFPU_SetLocal(T, Time); // additional time parameter

    Fn_Val = POVFPU_Run(*Env->Function);

    if (Fn_Val < MechsimOptions.Data.Masses[Index].Radius)
    {
      Function_Gradient( *Env->Function, Pos_Array[Index], Grad);

      VNormalizeEq(Grad);

      /* now we do the bad thing and modify the velocity */
      VDot(Norm_Len, Vel_Array[Index], Grad);
      Norm_Len = -Norm_Len;
      VAddScaledEq(Vel_Array[Index], Norm_Len, Grad);  // make movement perpendicular to surface zero
      VScaleEq(Vel_Array[Index], 1.0-Env->Friction);   // apply friction
      VAddScaledEq(Vel_Array[Index], Norm_Len*Env->Damping, Grad);

      Increase_Counter(stats[MechSim_Environment_Collisions]);
    }
  }
  else if (Env->Object != NULL) // ======== if only object is given use intersection test ===========
  {
    INTERSECTION Intersect;
    RAY Ray;
    VECTOR Dir, Norm;
    DBL speed, dist;

    Initialize_Ray_Containers(&Ray);

    VNormalize(Dir, Vel_Array[Index]);
    VAssign(Ray.Direction, Dir);
    VAssign(Ray.Initial, Pos_Array[Index]);

    if(Intersection(&Intersect, Env->Object, &Ray))
    {
      VLength(speed, Vel_Array[Index]);
      VDist(dist, Intersect.IPoint, Pos_Array[Index]);
      // this does only make sense for single step (euler) integration method
      // but it could lead to agreeable result with other methods too
      if (dist < (MechsimOptions.Time_Step*speed + MechsimOptions.Data.Masses[Index].Radius))
      {
        Normal(Norm, Intersect.Object, &Intersect);
        VNormalizeEq(Norm);

        /* now we do the bad thing and modify the velocity */
        VDot(Norm_Len, Vel_Array[Index], Norm);
        Norm_Len = -Norm_Len;
        VAddScaledEq(Vel_Array[Index], Norm_Len, Norm);  // make movement perpendicular to surface zero
        VScaleEq(Vel_Array[Index], 1.0-Env->Friction);   // apply friction
        VAddScaledEq(Vel_Array[Index], Norm_Len*Env->Damping, Norm);

        Increase_Counter(stats[MechSim_Environment_Collisions]);
      }
    }

  }

}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Gradient_Collide_Environment
*
* INPUT
*
*   Index - index of the mass
*   Pos_Array - state of the masses
*   Vel_Array - *** Different usage here ***
*               stores the last position of the mass
*   Time - current time index for the environment function
*   Env - Pointer to Environment
*
* OUTPUT
*
*   calculated gradients of the potential fields are added to 'Gradient'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Variation of Mechsim_Force_Collide_Environment() for use with gradient
*   descent method.  No damping forces.
*
* CHANGES
*
*   --- Sep 2002 : Creation
*       Oct 2002 : added object intersection based method
*
******************************************************************************/

void Mechsim_Gradient_Collide_Environment(int Index, VECTOR Gradient, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_ENVIRONMENT *Env)
{
  DBL Potential, Fn_Val;
  VECTOR pot_force, Grad;

  if (Env->Function != NULL)
  {
    FunctionCode *f = POVFPU_GetFunction(*Env->Function);

    // ---- collision test ----
    POVFPU_SetLocal(X, Pos_Array[Index][X]);
    POVFPU_SetLocal(Y, Pos_Array[Index][Y]);
    POVFPU_SetLocal(Z, Pos_Array[Index][Z]);
    if (f->parameter_cnt > 3) POVFPU_SetLocal(T, Time); // additional time parameter

    Fn_Val = POVFPU_Run(*Env->Function);

    if (Fn_Val < MechsimOptions.Data.Masses[Index].Radius)
    {
      Function_Gradient( *Env->Function, Pos_Array[Index], Grad);

      // --- potential force ---
      Potential = min(0.0, Fn_Val - MechsimOptions.Data.Masses[Index].Radius);
      VScale(pot_force, Grad, -Potential*Env->Stiffness);

      //VNormalizeEq(Grad);
      //VScale(pot_force, Grad, Env->Stiffness);

      VAddEq(Gradient, pot_force);

      Increase_Counter(stats[MechSim_Environment_Collisions]);
    }
  }
  else if (Env->Object != NULL) // ======== if only object is given use intersection test ===========
  {
    INTERSECTION Intersect;
    RAY Ray;
    VECTOR Dir, Norm;
    DBL Len;

    Initialize_Ray_Containers(&Ray);

    // Vel_Array[Index] stores the position from the last step so:
    // direction = Pos_Array[Index] - Vel_Array[Index]
    // see also: Mechsim_Simulate()
    VSub(Dir, Pos_Array[Index], Vel_Array[Index]);

    if ((Dir[X] != 0.0) || (Dir[Y] != 0.0) || (Dir[Z] != 0.0))
    {
      VAssign(Ray.Direction, Dir);
      VAssign(Ray.Initial, Pos_Array[Index]);

      if(Intersection(&Intersect, Env->Object, &Ray))
      {
        Normal(Norm, Intersect.Object, &Intersect);      // get the normal
        VNormalizeEq(Norm);
        VSub(Dir, Intersect.IPoint, Pos_Array[Index]);   // approximate the distance (assuming flat surface)
        VDot(Len, Norm, Dir);

        // see if we need to invert the normal
        if (Len<0)
        {
          VScaleEq(Norm, -1);
          Len = -Len;
        }

        if (Len < MechsimOptions.Data.Masses[Index].Radius)
        {
          // --- potential force ---
          Potential = min(0.0, Len - MechsimOptions.Data.Masses[Index].Radius);
          VScale(pot_force, Norm, Potential*Env->Stiffness / MechsimOptions.Data.Masses[Index].Mass);

          VAddEq(Gradient, pot_force);

          Increase_Counter(stats[MechSim_Environment_Collisions]);

        }
      }
    }

  }

}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Force_Collide_Environment
*
* INPUT
*
*   Index - index of the mass
*   Pos_Array, Vel_Array - state of the masses
*   Time - current time index for the environment function
*   Env - Pointer to Environment
*
* OUTPUT
*
*   calculated accelerations are added to 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates the forces resulting from a collision with the environment.
*
*   * If an environment function is given, it is evaluated and if
*     collision is found the gradient is calculated and an elastic force
*     is applied in direction of the gradient of the environment field.
*   * If only an object is given a ray is shot from the current position
*     in movement direction to determine the collision.
*
*   Damping and Friction are applied, Friction already slightly above the
*   surface to avoid gliding.
*
*   ========== WARNING ====================================================
*
*   The object based method is problematic because it relies on the velocity
*   vector to determine the direction of the ray although the masses are
*   slowed down during collision and therefore the velocity will no more
*   be suited for a successful collision test.  It's usually better to use
*   the impact based method instead if no function is available.  Apart
*   from that it won't successfully find the intersection in all situations
*   like the function based method.
*
*   =======================================================================
*
* CHANGES
*
*   --- Aug 2002 : Creation
*       Oct 2002 : added object intersection based method
*
******************************************************************************/

void Mechsim_Force_Collide_Environment(int Index, VECTOR Accel,VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_ENVIRONMENT *Env)
{
  DBL Potential, Potential2, Fn_Val, Len;
  VECTOR pot_force, pot_force2, Grad;
  VECTOR FDir, FFrict, damp_force;

  if (Env->Function != NULL)  // ======== if function is given use it to calculate gradient ===========
  {
    FunctionCode *f = POVFPU_GetFunction(*Env->Function);

    // ---- collision test ----
    POVFPU_SetLocal(X, Pos_Array[Index][X]);
    POVFPU_SetLocal(Y, Pos_Array[Index][Y]);
    POVFPU_SetLocal(Z, Pos_Array[Index][Z]);
    if (f->parameter_cnt > 3) POVFPU_SetLocal(T, Time); // additional time parameter

    Fn_Val = POVFPU_Run(*Env->Function);

    if (Fn_Val < MechsimOptions.Data.Masses[Index].Radius*Env->Friction_Excess)
    {
      Function_Gradient( *Env->Function, Pos_Array[Index], Grad);

      // --- potential force ---
      Potential = min(0.0, Fn_Val - MechsimOptions.Data.Masses[Index].Radius);
      VScale(pot_force, Grad, -Potential*Env->Stiffness / MechsimOptions.Data.Masses[Index].Mass);

      // --- damping/friction force ---

      if (Env->Friction > 0.0)
      {
        Potential2 = min(0.0, Fn_Val - MechsimOptions.Data.Masses[Index].Radius*Env->Friction_Excess);
        // friction possibly already slightly above the surface
        VScale(pot_force2, Grad, -Potential2*Env->Stiffness);

        // --- friction force ---
        VLength(Len, Vel_Array[Index])
        if (Len>MECHSIM_EPSILON)
        {
          VCross(FFrict, Grad, Vel_Array[Index]);
          VCross(FDir, FFrict, Grad);
          VLength(Len, FDir);
          if (Len>MECHSIM_EPSILON) VInverseScaleEq(FDir, Len);

          VLength(Len, pot_force2);
          VScale(FFrict, FDir, Env->Friction*Len);
        }
        else Make_Vector(FFrict, 0.0, 0.0, 0.0);
      }
      else Make_Vector(FFrict, 0.0, 0.0, 0.0);

      damp_force[X] = (fabs(Grad[X])*Potential*Env->Damping*Vel_Array[Index][X] - FFrict[X])/MechsimOptions.Data.Masses[Index].Mass;
      damp_force[Y] = (fabs(Grad[Y])*Potential*Env->Damping*Vel_Array[Index][Y] - FFrict[Y])/MechsimOptions.Data.Masses[Index].Mass;
      damp_force[Z] = (fabs(Grad[Z])*Potential*Env->Damping*Vel_Array[Index][Z] - FFrict[Z])/MechsimOptions.Data.Masses[Index].Mass;

      VAddEq(pot_force, damp_force);
      VAddEq(Accel, pot_force);

      Increase_Counter(stats[MechSim_Environment_Collisions]);
    }
  }
  else if (Env->Object != NULL) // ======== if only object is given use intersection test ===========
  {
    INTERSECTION Intersect;
    RAY Ray;
    VECTOR Dir, Norm;

    Initialize_Ray_Containers(&Ray);

    VNormalize(Dir, Vel_Array[Index]);
    VAssign(Ray.Direction, Dir);
    VAssign(Ray.Initial, Pos_Array[Index]);

    if(Intersection(&Intersect, Env->Object, &Ray))
    {
      Normal(Norm, Intersect.Object, &Intersect);      // get the normal
      VNormalizeEq(Norm);
      VSub(Dir, Intersect.IPoint, Pos_Array[Index]);   // approximate the distance (assuming flat surface)
      VDot(Len, Norm, Dir);

      // see if we need to invert the normal
      if (Len<0)
      {
        VScaleEq(Norm, -1);
        Len = -Len;
      }

      if (Len < MechsimOptions.Data.Masses[Index].Radius)
      {
        // --- potential force ---
        Potential = min(0.0, Len - MechsimOptions.Data.Masses[Index].Radius);
        VScale(pot_force, Norm, Potential*Env->Stiffness / MechsimOptions.Data.Masses[Index].Mass);

        // --- damping/friction force ---

        if (Env->Friction > 0.0)
        {
          Potential2 = min(0.0, Len - MechsimOptions.Data.Masses[Index].Radius*Env->Friction_Excess);
          // friction possibly already slightly above the surface
          VScale(pot_force2, Norm, Potential2*Env->Stiffness);

          // --- friction force ---
          VLength(Len, Vel_Array[Index])
          if (Len>MECHSIM_EPSILON)
          {
            VCross(FFrict, Norm, Vel_Array[Index]);
            VCross(FDir, FFrict, Norm);
            VLength(Len, FDir);
            if (Len>MECHSIM_EPSILON) VInverseScaleEq(FDir, Len);

            VLength(Len, pot_force2);
            VScale(FFrict, FDir, Env->Friction*Len);
          }
          else Make_Vector(FFrict, 0.0, 0.0, 0.0);
        }
        else Make_Vector(FFrict, 0.0, 0.0, 0.0);

        damp_force[X] = (fabs(Norm[X])*Potential*Env->Damping*Vel_Array[Index][X] - FFrict[X])/MechsimOptions.Data.Masses[Index].Mass;
        damp_force[Y] = (fabs(Norm[Y])*Potential*Env->Damping*Vel_Array[Index][Y] - FFrict[Y])/MechsimOptions.Data.Masses[Index].Mass;
        damp_force[Z] = (fabs(Norm[Z])*Potential*Env->Damping*Vel_Array[Index][Z] - FFrict[Z])/MechsimOptions.Data.Masses[Index].Mass;

        VAddEq(pot_force, damp_force);
        VAddEq(Accel, pot_force);

        Increase_Counter(stats[MechSim_Environment_Collisions]);
      }
    }

  }

}


/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Gradient_Collide_Dynamic
*
* INPUT
*
*   Pos_Array - state of the masses
*
* OUTPUT
*
*   calculated gradients of the potential fields are added to 'Gradient'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Variation of Mechsim_Force_Collide_Dynamic() for use with gradient
*   descent method.  No damping forces.
*
* CHANGES
*
*   --- Sep 2002 : Creation
*
******************************************************************************/

void Mechsim_Gradient_Collide_Dynamic(VECTOR *Gradient, VECTOR *Pos_Array, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
 bounding_hashtable, int bounding_hashtable_size, int time_stamp)
{
  int i, j;
  DBL Dist, lAxis;
  VECTOR Axis;
  VECTOR pot_force1, pot_force2;
  DBL Potential;
  int Grp1, Grp2;
  int Idx1, Idx2;
  int Start_Group, End_Group;

  /* ==================================================================================  */
  /* mass-mass collisions (new group version)                                            */
  /* ==================================================================================  */
  if (MechsimOptions.Calc_Collision != MECHSIM_COLLISION_NONE)
  {
    if (MechsimOptions.Calc_Collision == MECHSIM_COLLISION_ALL) Start_Group=0;
    else Start_Group=1;

    for (Grp1=Start_Group; Grp1<MechsimOptions.Data.group_count; Grp1++)
    {
      if (MechsimOptions.Calc_Collision == MECHSIM_COLLISION_ALL) End_Group=Grp1;
      else End_Group=Grp1-1;

      for (Grp2=0; Grp2<=End_Group; Grp2++)
      {
        /*
           This double loop accesses each combination of
           two groups that need to be tested against each other once.
           If (MechsimOptions.Calc_Collision == MECHSIM_COLLISION_ALL)
           Grp1 and Grp2 can be identical (for testing collisions within each group)
        */
        for (i=0; i<MechsimOptions.Data.Groups[Grp1].mass_count; i++)
          for (j=0; j<MechsimOptions.Data.Groups[Grp2].mass_count; j++)
          {
            /*
              Indices of the masses in conventional numbering,
              'MechsimOptions.Data.Groups[Grp1].Masses[i]->Mass' and
              'MechsimOptions.Data.Masses[Idx1].Mass' are identical
              so this won't be necessary for the properties, but
              Vel_Array, Pos_Array and Accel need it.
            */
            Idx1 = MechsimOptions.Data.Groups[Grp1].Masses[i]->Index;
            Idx2 = MechsimOptions.Data.Groups[Grp2].Masses[j]->Index;

            if (Idx1!=Idx2) /* avoid mass colliding with itself */
            {
              VSub(Axis, Pos_Array[Idx1], Pos_Array[Idx2] );
              VLength(lAxis, Axis);
              Dist = MechsimOptions.Data.Masses[Idx1].Radius+MechsimOptions.Data.Masses[Idx2].Radius;

              if (lAxis < Dist)
              {
                // --- potential force ---
                Potential = Dist-lAxis;
                VScale(pot_force1, Axis,  Potential*MechsimOptions.Collision_Stiffness);
                VScale(pot_force2, Axis, -Potential*MechsimOptions.Collision_Stiffness);

                VAddEq(Gradient[Idx1], pot_force1);
                VAddEq(Gradient[Idx2], pot_force2);

                Increase_Counter(stats[MechSim_Mass_Collisions]);

              }
            }
          } /* mass for loop */

      } /* group for loop */
    }
  }

  /* ==================================================================================  */
  /* mass-face collisions                                                                */
  /* ==================================================================================  */
  if ((MechsimOptions.Calc_Collision_Faces != MECHSIM_COLLISION_NONE) &&
      (MechsimOptions.Data.faces_count>0) && (MechsimOptions.Bounding != MECHSIM_FACE_BOUNDING_HASH))
  {
    VECTOR Grad;
    VECTOR CPoint;
    DBL Fn_Val, Ratio;
    VECTOR pot_force;
    DBL Weight1, Weight2, Weight3;
    int Side;

    if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_BBOX) calc_faces_bounding_box(Pos_Array, Faces_BBox_Array);
    
    for (i=0; i<MechsimOptions.Data.mass_count; i++)
    /* loop through all masses */
    {
      for (j=0; j<MechsimOptions.Data.faces_count; j++)
      /* loop through all faces */

        if (
            (i != MechsimOptions.Data.Faces[j].Idx1) &&
            (i != MechsimOptions.Data.Faces[j].Idx2) &&
            (i != MechsimOptions.Data.Faces[j].Idx3) && /* avoid testing against a face the mass i belongs to */
            (
             (MechsimOptions.Calc_Collision_Faces == MECHSIM_COLLISION_ALL) ||
             (MechsimOptions.Data.Faces[j].Group != MechsimOptions.Data.Masses[i].Group)
            ) &&
            /* and make sure mass and face are in different groups */
	    (
	     (MechsimOptions.Bounding != MECHSIM_FACE_BOUNDING_BBOX) ||
	     (intersect_sphere_box(Faces_BBox_Array[j].bbox_min, Faces_BBox_Array[j].bbox_max, Pos_Array[i],
	      MechsimOptions.Data.Masses[i].Radius))
	    )
	    /* quick check if mass intersects face */
           )
        {
          // ---- collision test ----
          // also calculates contact point and gradient
          Side = func_triangle(Pos_Array[i], CPoint, Grad,
                               Pos_Array[MechsimOptions.Data.Faces[j].Idx1],
                               Pos_Array[MechsimOptions.Data.Faces[j].Idx2],
                               Pos_Array[MechsimOptions.Data.Faces[j].Idx3], &Ratio);

          VLength(Fn_Val, Grad);

          if (Fn_Val < MechsimOptions.Data.Masses[i].Radius)
          {
            VInverseScaleEq(Grad, Fn_Val);

            // --- potential force ---
            Potential = min(0.0, Fn_Val - MechsimOptions.Data.Masses[i].Radius);
            VScale(pot_force, Grad, -Potential*MechsimOptions.Collision_Stiffness);

            VAddEq(Gradient[i], pot_force);

            // --- now calculate distribution of reaction force between triangle vertices ---

            switch (Side)
            {
              // if outside, the triangle function has already returned the nearest side
              case 1: /* P1-P2 */
                Weight1 = 1.0-Ratio;
                Weight2 = Ratio;
                Weight3 = 0.0;
                break;
              case 2: /* P2-P3 */
                Weight2 = 1.0-Ratio;
                Weight3 = Ratio;
                Weight1 = 0.0;
                break;
              case 3: /* P3-P1 */
                Weight3 = 1.0-Ratio;
                Weight1 = Ratio;
                Weight2 = 0.0;
                break;
              default:
                // if inside triangle, calculate distribution between the three vertices
                Triangle_Distribute(CPoint,
                                    Pos_Array[MechsimOptions.Data.Faces[j].Idx1],
                                    Pos_Array[MechsimOptions.Data.Faces[j].Idx2],
                                    Pos_Array[MechsimOptions.Data.Faces[j].Idx3],
                                    &Weight1, &Weight2, &Weight3);
                break;
            }

            // apply forces to the vertices
            VAddScaledEq(Gradient[MechsimOptions.Data.Faces[j].Idx1], -Weight1, pot_force);
            VAddScaledEq(Gradient[MechsimOptions.Data.Faces[j].Idx2], -Weight2, pot_force);
            VAddScaledEq(Gradient[MechsimOptions.Data.Faces[j].Idx3], -Weight3, pot_force);

            Increase_Counter(stats[MechSim_Face_Collisions]);
          }
        }

    }
  }
  /* ==================================================================================  */
  /* mass-face collisions                                                                */
  /* ==================================================================================  */
  if ((MechsimOptions.Calc_Collision_Faces != MECHSIM_COLLISION_NONE) &&
      (MechsimOptions.Data.faces_count>0) && (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH))
  {
    VECTOR Grad;
    VECTOR CPoint;
    DBL Fn_Val, Ratio;
    VECTOR pot_force;
    DBL Weight1, Weight2, Weight3;
    int Side;

    int size, index, k;
    int* list;
    DBL Unit;
    
    size = 8;
    index = 0;
    list = (int *)POV_MALLOC(size*sizeof(int), "mechsim bounding hashtable intersect list");
    memset(list, 0, sizeof(int)*size);
    
    Unit = calc_faces_bounding_box(Pos_Array, Faces_BBox_Array);
    
    calc_faces_bounding_hashtable(bounding_hashtable, Unit, bounding_hashtable_size, time_stamp, Faces_BBox_Array);
    for (i = 0; i < MechsimOptions.Data.mass_count; i++)
    /* loop through all masses */
    {
      intersect_sphere_bounding_hashtable(bounding_hashtable, Pos_Array[i], MechsimOptions.Data.Masses[i].Radius, 
        Unit, &list, &size, &index, bounding_hashtable_size, time_stamp);
      for (k = 0; k < index; k++)
      {
        j = list[k];
        if (
            (i != MechsimOptions.Data.Faces[j].Idx1) &&
            (i != MechsimOptions.Data.Faces[j].Idx2) &&
            (i != MechsimOptions.Data.Faces[j].Idx3) && /* avoid testing against a face the mass i belongs to */
            (
             (MechsimOptions.Calc_Collision_Faces == MECHSIM_COLLISION_ALL) ||
             (MechsimOptions.Data.Faces[j].Group != MechsimOptions.Data.Masses[i].Group)
            ) &&
            /* and make sure mass and face are in different groups */
	    intersect_sphere_box(Faces_BBox_Array[j].bbox_min, Faces_BBox_Array[j].bbox_max,
	      Pos_Array[i], MechsimOptions.Data.Masses[i].Radius)
           )
        {
          // ---- collision test ----
          // also calculates contact point and gradient
          Side = func_triangle(Pos_Array[i], CPoint, Grad,
                               Pos_Array[MechsimOptions.Data.Faces[j].Idx1],
                               Pos_Array[MechsimOptions.Data.Faces[j].Idx2],
                               Pos_Array[MechsimOptions.Data.Faces[j].Idx3], &Ratio);

          VLength(Fn_Val, Grad);

          if (Fn_Val < MechsimOptions.Data.Masses[i].Radius)
          {
            VInverseScaleEq(Grad, Fn_Val);

            // --- potential force ---
            Potential = min(0.0, Fn_Val - MechsimOptions.Data.Masses[i].Radius);
            VScale(pot_force, Grad, -Potential*MechsimOptions.Collision_Stiffness);

            VAddEq(Gradient[i], pot_force);

            // --- now calculate distribution of reaction force between triangle vertices ---

            switch (Side)
            {
              // if outside, the triangle function has already returned the nearest side
              case 1: /* P1-P2 */
                Weight1 = 1.0-Ratio;
                Weight2 = Ratio;
                Weight3 = 0.0;
                break;
              case 2: /* P2-P3 */
                Weight2 = 1.0-Ratio;
                Weight3 = Ratio;
                Weight1 = 0.0;
                break;
              case 3: /* P3-P1 */
                Weight3 = 1.0-Ratio;
                Weight1 = Ratio;
                Weight2 = 0.0;
                break;
              default:
                // if inside triangle, calculate distribution between the three vertices
                Triangle_Distribute(CPoint,
                                    Pos_Array[MechsimOptions.Data.Faces[j].Idx1],
                                    Pos_Array[MechsimOptions.Data.Faces[j].Idx2],
                                    Pos_Array[MechsimOptions.Data.Faces[j].Idx3],
                                    &Weight1, &Weight2, &Weight3);
                break;
            }

            // apply forces to the vertices
            VAddScaledEq(Gradient[MechsimOptions.Data.Faces[j].Idx1], -Weight1, pot_force);
            VAddScaledEq(Gradient[MechsimOptions.Data.Faces[j].Idx2], -Weight2, pot_force);
            VAddScaledEq(Gradient[MechsimOptions.Data.Faces[j].Idx3], -Weight3, pot_force);

            Increase_Counter(stats[MechSim_Face_Collisions]);
          }
        }
      }
    }
    POV_FREE(list);
  }
}


/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Force_Collide_Dynamic
*
* INPUT
*
*   Pos_Array, Vel_Array - state of the masses
*
* OUTPUT
*
*   calculated accelerations are added to 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates the forces resulting from point mass collisions
*
*   For all mass pairs in collision elastic and damping force are calculated
*   and added to the 'Accel' vector of both masses.
*
* CHANGES
*
*   --- Aug 2002 : Creation
*   --- Sep 2002 : Modification for grouping masses for faster simulation
*
******************************************************************************/

void Mechsim_Force_Collide_Dynamic(VECTOR *Accel, VECTOR *Pos_Array, VECTOR *Vel_Array, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
  bounding_hashtable, int bounding_hashtable_size, int time_stamp)
{
  int i, j;
  DBL Dist, lAxis;
  VECTOR Axis;
  VECTOR pot_force1, pot_force2, damp_force;
  DBL Potential;
  int Grp1, Grp2;
  int Idx1, Idx2;
  int Start_Group, End_Group;

  /* ==================================================================================  */
  /* mass-mass collisions (new group version)                                            */
  /* ==================================================================================  */
  if (MechsimOptions.Calc_Collision != MECHSIM_COLLISION_NONE)
  {
    if (MechsimOptions.Calc_Collision == MECHSIM_COLLISION_ALL) Start_Group=0;
    else Start_Group=1;

    for (Grp1=Start_Group; Grp1<MechsimOptions.Data.group_count; Grp1++)
    {
      if (MechsimOptions.Calc_Collision == MECHSIM_COLLISION_ALL) End_Group=Grp1;
      else End_Group=Grp1-1;

      for (Grp2=0; Grp2<=End_Group; Grp2++)
      {
        /*
           This double loop accesses each combination of
           two groups that need to be tested against each other once.
           If (MechsimOptions.Calc_Collision == MECHSIM_COLLISION_ALL)
           Grp1 and Grp2 can be identical (for testing collisions within each group)
        */
        for (i=0; i<MechsimOptions.Data.Groups[Grp1].mass_count; i++)
          for (j=0; j<MechsimOptions.Data.Groups[Grp2].mass_count; j++)
          {
            /*
              Indices of the masses in conventional numbering,
              'MechsimOptions.Data.Groups[Grp1].Masses[i]->Mass' and
              'MechsimOptions.Data.Masses[Idx1].Mass' are identical
              so this won't be necessary for the properties, but
              Vel_Array, Pos_Array and Accel need it.
            */
            Idx1 = MechsimOptions.Data.Groups[Grp1].Masses[i]->Index;
            Idx2 = MechsimOptions.Data.Groups[Grp2].Masses[j]->Index;

            if (Idx1!=Idx2) /* avoid mass colliding with itself */
            {
              VSub(Axis, Pos_Array[Idx1], Pos_Array[Idx2] );
              VLength(lAxis, Axis);
              Dist = MechsimOptions.Data.Masses[Idx1].Radius+MechsimOptions.Data.Masses[Idx2].Radius;

              if (lAxis < Dist)
              {
                // --- potential force ---
                Potential = Dist-lAxis;
                VScale(pot_force1, Axis,  Potential*MechsimOptions.Collision_Stiffness / MechsimOptions.Data.Masses[Idx1].Mass);
                VScale(pot_force2, Axis, -Potential*MechsimOptions.Collision_Stiffness / MechsimOptions.Data.Masses[Idx2].Mass);


                // --- damping force ---
                //damp_force[X] = fabs(Axis[X])*Potential*MechsimOptions.Collision_Damping*(Vel_Array[i][X]-Vel_Array[j][X]);
                //damp_force[Y] = fabs(Axis[Y])*Potential*MechsimOptions.Collision_Damping*(Vel_Array[i][Y]-Vel_Array[j][Y]);
                //damp_force[Z] = fabs(Axis[Z])*Potential*MechsimOptions.Collision_Damping*(Vel_Array[i][Z]-Vel_Array[j][Z]);

                //VAddScaledEq(pot_force1, -1.0/MechsimOptions.Data.Masses[i].Mass, damp_force);
                //VAddScaledEq(pot_force2,  1.0/MechsimOptions.Data.Masses[j].Mass, damp_force);

                damp_force[X] = Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][X];
                damp_force[Y] = Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Y];
                damp_force[Z] = Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Z];

                VAddScaledEq(pot_force1, -1.0/MechsimOptions.Data.Masses[Idx1].Mass, damp_force);

                damp_force[X] = Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx2][X];
                damp_force[Y] = Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx2][Y];
                damp_force[Z] = Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx2][Z];

                VAddScaledEq(pot_force2, -1.0/MechsimOptions.Data.Masses[Idx2].Mass, damp_force);

                VAddEq(Accel[Idx1], pot_force1);
                VAddEq(Accel[Idx2], pot_force2);

                Increase_Counter(stats[MechSim_Mass_Collisions]);

                //Status_Info("\ntest %d, %d, %d, %d, %d, %d\n", Grp1, Grp2, Idx1, Idx2, i, j );
              }
            }
          } /* mass for loop */

      } /* group for loop */
    }
  }

  /* ==================================================================================  */
  /* mass-face collisions (new group version)                                            */
  /* ==================================================================================  */
  if ((MechsimOptions.Calc_Collision_Faces != MECHSIM_COLLISION_NONE) &&
      (MechsimOptions.Data.faces_count>0) && (MechsimOptions.Bounding != MECHSIM_FACE_BOUNDING_HASH))
  {
    VECTOR Grad;
    VECTOR CPoint;
    DBL Fn_Val, Ratio, Len;
    VECTOR pot_force, pot_force2, FDir, FFrict;
    DBL Weight1, Weight2, Weight3;
    int Side;
    DBL Potential2;

    if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_BBOX) calc_faces_bounding_box(Pos_Array, Faces_BBox_Array);
    
    for (Grp1=0; Grp1<MechsimOptions.Data.group_count; Grp1++)
      for (Grp2=0; Grp2<MechsimOptions.Data.group_count; Grp2++)
        if (
             (MechsimOptions.Calc_Collision_Faces == MECHSIM_COLLISION_ALL) ||
             (Grp1 != Grp2)
           )
        {
          /*
             This double loop accesses each combination of
             two groups that need to be tested against each other *TWICE*
             (in contrast to the mass mass collision.  One time for testing the
             faces of Grp1 with the masses of Grp2, the other vica verse.
             If (MechsimOptions.Calc_Collision == MECHSIM_COLLISION_ALL)
             Grp1 and Grp2 can be identical (for testing collisions within each group)
          */
          for (i=0; i<MechsimOptions.Data.Groups[Grp1].mass_count; i++)
            for (j=0; j<MechsimOptions.Data.Groups[Grp2].faces_count; j++)
            {
              /*
                Indices of the masses and faces in conventional numbering,
                'MechsimOptions.Data.Groups[Grp1].Masses[i]->Mass' and
                'MechsimOptions.Data.Masses[Idx1].Mass' are identical
                so this won't be necessary for the properties, but
                Vel_Array, Pos_Array and Accel need it.
              */
              Idx1 = MechsimOptions.Data.Groups[Grp1].Masses[i]->Index;
              Idx2 = MechsimOptions.Data.Groups[Grp2].Faces[j]->Index;

              if (
                  (Idx1 != MechsimOptions.Data.Faces[Idx2].Idx1) &&
                  (Idx1 != MechsimOptions.Data.Faces[Idx2].Idx2) &&
                  (Idx1 != MechsimOptions.Data.Faces[Idx2].Idx3) && /* avoid testing against a face the mass i belongs to */
	    	  (
	     	   (MechsimOptions.Bounding != MECHSIM_FACE_BOUNDING_BBOX) ||
	           (intersect_sphere_box(Faces_BBox_Array[Idx2].bbox_min, Faces_BBox_Array[Idx2].bbox_max, Pos_Array[Idx1],
	      	    MechsimOptions.Data.Masses[Idx1].Radius))
	    	  )
	          /* quick check if mass intersects face */
                 )
              {

                // ---- collision test ----
                // also calculates contact point and gradient
                Side = func_triangle(Pos_Array[Idx1], CPoint, Grad,
                                     Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1],
                                     Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2],
                                     Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3], &Ratio);

                VLength(Fn_Val, Grad);

                if (Fn_Val < MechsimOptions.Data.Masses[Idx1].Radius)
                {

                  //Status_Info("\ntest %d, %d, %d, %d\n", Grp1, Grp2, Idx1, Idx2 );

                  VInverseScaleEq(Grad, Fn_Val);

                  // --- potential force ---
                  Potential = min(0.0, Fn_Val - MechsimOptions.Data.Masses[Idx1].Radius);
                  VScale(pot_force, Grad, -Potential*MechsimOptions.Collision_Stiffness);

                  // --- damping force ---
                  //damp_force[X] = (fabs(Grad[X])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][X])/MechsimOptions.Data.Masses[Idx1].Mass;
                  //damp_force[Y] = (fabs(Grad[Y])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Y])/MechsimOptions.Data.Masses[Idx1].Mass;
                  //damp_force[Z] = (fabs(Grad[Z])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Z])/MechsimOptions.Data.Masses[Idx1].Mass;

                  // --- damping/friction force ---
                  if (MechsimOptions.Collision_Friction > 0.0)
                  {
                    Potential2 = min(0.0, Fn_Val - MechsimOptions.Data.Masses[Idx1].Radius*MechsimOptions.Collision_Friction_Excess);
                    // friction already slightly above the surface
                    VScale(pot_force2, Grad, -Potential2*MechsimOptions.Collision_Stiffness);

                    // --- friction force ---
                    VLength(Len, Vel_Array[Idx1])
                    if (Len>MECHSIM_EPSILON)
                    {
                      VCross(FFrict, Grad, Vel_Array[Idx1]);
                      VCross(FDir, FFrict, Grad);
                      VLength(Len, FDir);
                      if (Len>MECHSIM_EPSILON) VInverseScaleEq(FDir, Len);

                      VLength(Len, pot_force2);
                      VScale(FFrict, FDir, MechsimOptions.Collision_Friction*Len);
                    }
                    else Make_Vector(FFrict, 0.0, 0.0, 0.0);
                  }
                  else Make_Vector(FFrict, 0.0, 0.0, 0.0);

                  damp_force[X] = (fabs(Grad[X])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][X] - FFrict[X])/
                                   MechsimOptions.Data.Masses[Idx1].Mass;
                  damp_force[Y] = (fabs(Grad[Y])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Y] - FFrict[Y])/
                                   MechsimOptions.Data.Masses[Idx1].Mass;
                  damp_force[Z] = (fabs(Grad[Z])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Z] - FFrict[Z])/
                                   MechsimOptions.Data.Masses[Idx1].Mass;

                  VAddEq(Accel[Idx1], damp_force); // apply to mass
                  VAddScaledEq(Accel[Idx1], 1.0/MechsimOptions.Data.Masses[Idx1].Mass, pot_force);

                  // --- now calculate distribution of reaction force between triangle vertices ---

                  switch (Side)
                  {
                    // if outside, the triangle function has already returned the nearest side
                    case 1: /* P1-P2 */
                      Weight1 = 1.0-Ratio;
                      Weight2 = Ratio;
                      Weight3 = 0.0;
                      break;
                    case 2: /* P2-P3 */
                      Weight2 = 1.0-Ratio;
                      Weight3 = Ratio;
                      Weight1 = 0.0;
                      break;
                    case 3: /* P3-P1 */
                      Weight3 = 1.0-Ratio;
                      Weight1 = Ratio;
                      Weight2 = 0.0;
                      break;
                    default:
                      // if inside triangle, calculate distribution between the three vertices
                      Triangle_Distribute(CPoint,
                                          Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1],
                                          Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2],
                                          Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3],
                                          &Weight1, &Weight2, &Weight3);
                      break;
                  }

                  // apply forces to the vertices
                  VAddScaledEq(Accel[MechsimOptions.Data.Faces[Idx2].Idx1],
                               -Weight1/MechsimOptions.Data.Masses[MechsimOptions.Data.Faces[Idx2].Idx1].Mass,
                               pot_force);
                  VAddScaledEq(Accel[MechsimOptions.Data.Faces[Idx2].Idx2],
                               -Weight2/MechsimOptions.Data.Masses[MechsimOptions.Data.Faces[Idx2].Idx2].Mass,
                               pot_force);
                  VAddScaledEq(Accel[MechsimOptions.Data.Faces[Idx2].Idx3],
                               -Weight3/MechsimOptions.Data.Masses[MechsimOptions.Data.Faces[Idx2].Idx3].Mass,
                               pot_force);


                  Increase_Counter(stats[MechSim_Face_Collisions]);

                  //Status_Info("\n%g Grad=<%g, %g, %g>", Fn_Val, Grad[X], Grad[Y], Grad[Z] );
                  /*
                  Status_Info("\n%g CPoint=<%g, %g, %g>", Fn_Val, CPoint[X], CPoint[Y], CPoint[Z] );
                  Status_Info("\nPos%d=<%g, %g, %g>", MechsimOptions.Data.Faces[Idx2].Idx1,
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1][X], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1][Y], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1][Z] );
                  Status_Info("\nPos%d=<%g, %g, %g>", MechsimOptions.Data.Faces[Idx2].Idx2,
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2][X], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2][Y], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2][Z] );
                   Status_Info("\nPos%d=<%g, %g, %g>", MechsimOptions.Data.Faces[Idx2].Idx3,
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3][X], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3][Y], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3][Z] );
                  Status_Info("\n%d Weight=<%g, %g, %g>\n", Side, Weight1, Weight2, Weight3 );
                  */
                }
              }
            } /* mass for loop */

        } /* group for loop */

  }
  /* ==================================================================================  */
  /* mass-face collisions (new group version)                                            */
  /* ==================================================================================  */
  if ((MechsimOptions.Calc_Collision_Faces != MECHSIM_COLLISION_NONE) &&
      (MechsimOptions.Data.faces_count>0) && (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH))
  {
    VECTOR Grad;
    VECTOR CPoint;
    DBL Fn_Val, Ratio, Len;
    VECTOR pot_force, pot_force2, FDir, FFrict;
    DBL Weight1, Weight2, Weight3;
    int Side;
    DBL Potential2;

    int size, index, k;
    int* list;
    DBL Unit;
    
    size = 8;
    index = 0;
    list = (int *)POV_MALLOC(size*sizeof(int), "mechsim bounding hashtable intersect list");
    memset(list, 0, sizeof(int)*size);
    
    Unit = calc_faces_bounding_box(Pos_Array, Faces_BBox_Array);
    
    calc_faces_bounding_hashtable(bounding_hashtable, Unit, bounding_hashtable_size, time_stamp, Faces_BBox_Array);
    for (Idx1 = 0; Idx1 < MechsimOptions.Data.mass_count; Idx1++)
    /* loop through all masses */
    {
      intersect_sphere_bounding_hashtable(bounding_hashtable, Pos_Array[Idx1], MechsimOptions.Data.Masses[Idx1].Radius, 
        Unit, &list, &size, &index, bounding_hashtable_size, time_stamp);
      for (k = 0; k < index; k++)
      {
        Idx2 = list[k];
        if (
            (Idx1 != MechsimOptions.Data.Faces[Idx2].Idx1) &&
            (Idx1 != MechsimOptions.Data.Faces[Idx2].Idx2) &&
            (Idx1 != MechsimOptions.Data.Faces[Idx2].Idx3) && /* avoid testing against a face the mass i belongs to */
            (
             (MechsimOptions.Calc_Collision_Faces == MECHSIM_COLLISION_ALL) ||
             (MechsimOptions.Data.Faces[Idx2].Group != MechsimOptions.Data.Masses[Idx1].Group)
            ) &&
            /* and make sure mass and face are in different groups */
	    intersect_sphere_box(Faces_BBox_Array[Idx2].bbox_min, Faces_BBox_Array[Idx2].bbox_max,
	      Pos_Array[Idx1], MechsimOptions.Data.Masses[Idx1].Radius)
           )
        {

                // ---- collision test ----
                // also calculates contact point and gradient
                Side = func_triangle(Pos_Array[Idx1], CPoint, Grad,
                                     Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1],
                                     Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2],
                                     Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3], &Ratio);

                VLength(Fn_Val, Grad);

                if (Fn_Val < MechsimOptions.Data.Masses[Idx1].Radius)
                {

                  //Status_Info("\ntest %d, %d, %d, %d\n", Grp1, Grp2, Idx1, Idx2 );

                  VInverseScaleEq(Grad, Fn_Val);

                  // --- potential force ---
                  Potential = min(0.0, Fn_Val - MechsimOptions.Data.Masses[Idx1].Radius);
                  VScale(pot_force, Grad, -Potential*MechsimOptions.Collision_Stiffness);

                  // --- damping force ---
                  //damp_force[X] = (fabs(Grad[X])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][X])/MechsimOptions.Data.Masses[Idx1].Mass;
                  //damp_force[Y] = (fabs(Grad[Y])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Y])/MechsimOptions.Data.Masses[Idx1].Mass;
                  //damp_force[Z] = (fabs(Grad[Z])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Z])/MechsimOptions.Data.Masses[Idx1].Mass;

                  // --- damping/friction force ---
                  if (MechsimOptions.Collision_Friction > 0.0)
                  {
                    Potential2 = min(0.0, Fn_Val - MechsimOptions.Data.Masses[Idx1].Radius*MechsimOptions.Collision_Friction_Excess);
                    // friction already slightly above the surface
                    VScale(pot_force2, Grad, -Potential2*MechsimOptions.Collision_Stiffness);

                    // --- friction force ---
                    VLength(Len, Vel_Array[Idx1])
                    if (Len>MECHSIM_EPSILON)
                    {
                      VCross(FFrict, Grad, Vel_Array[Idx1]);
                      VCross(FDir, FFrict, Grad);
                      VLength(Len, FDir);
                      if (Len>MECHSIM_EPSILON) VInverseScaleEq(FDir, Len);

                      VLength(Len, pot_force2);
                      VScale(FFrict, FDir, MechsimOptions.Collision_Friction*Len);
                    }
                    else Make_Vector(FFrict, 0.0, 0.0, 0.0);
                  }
                  else Make_Vector(FFrict, 0.0, 0.0, 0.0);

                  damp_force[X] = (fabs(Grad[X])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][X] - FFrict[X])/
                                   MechsimOptions.Data.Masses[Idx1].Mass;
                  damp_force[Y] = (fabs(Grad[Y])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Y] - FFrict[Y])/
                                   MechsimOptions.Data.Masses[Idx1].Mass;
                  damp_force[Z] = (fabs(Grad[Z])*Potential*MechsimOptions.Collision_Damping*Vel_Array[Idx1][Z] - FFrict[Z])/
                                   MechsimOptions.Data.Masses[Idx1].Mass;

                  VAddEq(Accel[Idx1], damp_force); // apply to mass
                  VAddScaledEq(Accel[Idx1], 1.0/MechsimOptions.Data.Masses[Idx1].Mass, pot_force);

                  // --- now calculate distribution of reaction force between triangle vertices ---

                  switch (Side)
                  {
                    // if outside, the triangle function has already returned the nearest side
                    case 1: /* P1-P2 */
                      Weight1 = 1.0-Ratio;
                      Weight2 = Ratio;
                      Weight3 = 0.0;
                      break;
                    case 2: /* P2-P3 */
                      Weight2 = 1.0-Ratio;
                      Weight3 = Ratio;
                      Weight1 = 0.0;
                      break;
                    case 3: /* P3-P1 */
                      Weight3 = 1.0-Ratio;
                      Weight1 = Ratio;
                      Weight2 = 0.0;
                      break;
                    default:
                      // if inside triangle, calculate distribution between the three vertices
                      Triangle_Distribute(CPoint,
                                          Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1],
                                          Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2],
                                          Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3],
                                          &Weight1, &Weight2, &Weight3);
                      break;
                  }

                  // apply forces to the vertices
                  VAddScaledEq(Accel[MechsimOptions.Data.Faces[Idx2].Idx1],
                               -Weight1/MechsimOptions.Data.Masses[MechsimOptions.Data.Faces[Idx2].Idx1].Mass,
                               pot_force);
                  VAddScaledEq(Accel[MechsimOptions.Data.Faces[Idx2].Idx2],
                               -Weight2/MechsimOptions.Data.Masses[MechsimOptions.Data.Faces[Idx2].Idx2].Mass,
                               pot_force);
                  VAddScaledEq(Accel[MechsimOptions.Data.Faces[Idx2].Idx3],
                               -Weight3/MechsimOptions.Data.Masses[MechsimOptions.Data.Faces[Idx2].Idx3].Mass,
                               pot_force);


                  Increase_Counter(stats[MechSim_Face_Collisions]);

                  //Status_Info("\n%g Grad=<%g, %g, %g>", Fn_Val, Grad[X], Grad[Y], Grad[Z] );
                  /*
                  Status_Info("\n%g CPoint=<%g, %g, %g>", Fn_Val, CPoint[X], CPoint[Y], CPoint[Z] );
                  Status_Info("\nPos%d=<%g, %g, %g>", MechsimOptions.Data.Faces[Idx2].Idx1,
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1][X], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1][Y], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx1][Z] );
                  Status_Info("\nPos%d=<%g, %g, %g>", MechsimOptions.Data.Faces[Idx2].Idx2,
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2][X], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2][Y], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx2][Z] );
                   Status_Info("\nPos%d=<%g, %g, %g>", MechsimOptions.Data.Faces[Idx2].Idx3,
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3][X], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3][Y], 
                                                    Pos_Array[MechsimOptions.Data.Faces[Idx2].Idx3][Z] );
                  Status_Info("\n%d Weight=<%g, %g, %g>\n", Side, Weight1, Weight2, Weight3 );
                  */
                }
              }
            } /* mass for loop */

        } /* group for loop */

    POV_FREE(list);
  }

}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Gradient_Connect
*
* INPUT
*
*   Pos_Array - state of the masses
*
* OUTPUT
*
*   calculated gradients of the potential fields are added to 'Gradient'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Variation of Mechsim_Force_Connect() for use with gradient
*   descent method.  No damping forces.
*
* CHANGES
*
*   --- Sep 2002 : Creation
*
******************************************************************************/

void Mechsim_Gradient_Connect(VECTOR *Gradient, VECTOR *Pos_Array)
{
  int i;
  DBL lAxis;
  VECTOR Axis;
  VECTOR pot_force;

  for (i=0; i<MechsimOptions.Data.connect_count; i++)
  {
    VSub(Axis, Pos_Array[MechsimOptions.Data.Connects[i].Idx1],
               Pos_Array[MechsimOptions.Data.Connects[i].Idx2] );
    VLength(lAxis, Axis);

    if (lAxis>0)  // otherwise we can't conclude the direction
    {
      // --- potential force ---
      VScale(pot_force, Axis, -((lAxis-MechsimOptions.Data.Connects[i].Length)/lAxis)*MechsimOptions.Data.Connects[i].Stiffness);

      VAddEq(Gradient[MechsimOptions.Data.Connects[i].Idx1], pot_force);
      VSubEq(Gradient[MechsimOptions.Data.Connects[i].Idx2], pot_force);

    }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Force_Connect
*
* INPUT
*
*   Pos_Array, Vel_Array - state of the masses
*
* OUTPUT
*
*   calculated accelerations are added to 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates the forces resulting from connections
*
*   Looping through the connection array all connected mass pairs are
*   handled and accelerations added to the 'Accel' vector of both masses.
*
* CHANGES
*
*   --- Aug 2002 : Creation
*
******************************************************************************/

void Mechsim_Force_Connect(VECTOR *Accel, VECTOR *Pos_Array, VECTOR *Vel_Array)
{
  int i;
  DBL lAxis;
  VECTOR Axis;
  VECTOR pot_force, pot_force1, pot_force2, damp_force;
  DBL damp_fact;

  for (i=0; i<MechsimOptions.Data.connect_count; i++)
  {
    VSub(Axis, Pos_Array[MechsimOptions.Data.Connects[i].Idx1],
               Pos_Array[MechsimOptions.Data.Connects[i].Idx2] );
    VLength(lAxis, Axis);

    if (lAxis>0)  // otherwise we can't conclude the direction
    {
      // --- potential force ---
      VScale(pot_force, Axis, -((lAxis-MechsimOptions.Data.Connects[i].Length)/lAxis)*MechsimOptions.Data.Connects[i].Stiffness);

      // --- damping force ---

      if (MechsimOptions.Data.Connects[i].Damping > 0.0)
      {
        damp_fact =   // (x*vx + y*vy + z*vz)/ (x^2 + y^2 + z^2)
          (
           (Pos_Array[MechsimOptions.Data.Connects[i].Idx1][X]-Pos_Array[MechsimOptions.Data.Connects[i].Idx2][X])*
           (Vel_Array[MechsimOptions.Data.Connects[i].Idx1][X]-Vel_Array[MechsimOptions.Data.Connects[i].Idx2][X])
           +
           (Pos_Array[MechsimOptions.Data.Connects[i].Idx1][Y]-Pos_Array[MechsimOptions.Data.Connects[i].Idx2][Y])*
           (Vel_Array[MechsimOptions.Data.Connects[i].Idx1][Y]-Vel_Array[MechsimOptions.Data.Connects[i].Idx2][Y])
           +
           (Pos_Array[MechsimOptions.Data.Connects[i].Idx1][Z]-Pos_Array[MechsimOptions.Data.Connects[i].Idx2][Z])*
           (Vel_Array[MechsimOptions.Data.Connects[i].Idx1][Z]-Vel_Array[MechsimOptions.Data.Connects[i].Idx2][Z])
          )/(lAxis*lAxis);


        VScale(damp_force, Axis, -damp_fact*MechsimOptions.Data.Connects[i].Damping);

        VAddEq(pot_force, damp_force);
      }

      VScale(pot_force1, pot_force,  1.0/MechsimOptions.Data.Masses[MechsimOptions.Data.Connects[i].Idx1].Mass);
      VScale(pot_force2, pot_force, -1.0/MechsimOptions.Data.Masses[MechsimOptions.Data.Connects[i].Idx2].Mass);

      VAddEq(Accel[MechsimOptions.Data.Connects[i].Idx1], pot_force1);
      VAddEq(Accel[MechsimOptions.Data.Connects[i].Idx2], pot_force2);

    }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Gradient_Interactions
*
* INPUT
*
*   Pos_Array - positions of the masses
*
* OUTPUT
*
*   calculated accelerations are added to 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Gradient descent version for the interactions
*
* CHANGES
*
*   --- Nov 2002 : Creation
*
******************************************************************************/

void Mechsim_Gradient_Interactions(VECTOR *Accel, VECTOR *Pos_Array)
{
  int i, j, k;
  DBL Fn_Val, lAxis;
  VECTOR Axis, Pos;
  VECTOR pot_force;

  for (i=0; i<MechsimOptions.Data.mass_count; i++)
  {
    for (j=0; j<i; j++)
    /*
       This double loop accesses each combination of
       two different masses exactly once
    */
    {
      VSub(Axis, Pos_Array[i], Pos_Array[j] );
      VLength(lAxis, Axis);

      for (k=0; k<MechsimOptions.Interactions.count; k++)
      {
        /*
           Determine point in the middle of the mass positions
           and evaluate function there
        */
        VLinComb2(Pos, 0.5, Pos_Array[i], 0.5, Pos_Array[j])

        FunctionCode *f = POVFPU_GetFunction(*(MechsimOptions.Interactions.Function[k]));

        POVFPU_SetLocal(X, Pos[X]);
        POVFPU_SetLocal(Y, Pos[Y]);
        POVFPU_SetLocal(Z, Pos[Z]);
        if (f->parameter_cnt > 3) POVFPU_SetLocal(T, lAxis); // additional distance parameter

        Fn_Val = POVFPU_Run(*(MechsimOptions.Interactions.Function[k]));

        if (fabs(Fn_Val) > MECHSIM_EPSILON)
        {
          /* apply forces */
          VScale(pot_force, Axis, -Fn_Val);
          VAddEq(Accel[i], pot_force);
          VSubEq(Accel[j], pot_force);
        }
      }
    }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Force_Interactions
*
* INPUT
*
*   Pos_Array - positions of the masses
*
* OUTPUT
*
*   calculated accelerations are added to 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates the function defined interactions between the masses
*
*   Interaction functions can have 4 parameters: the three spartial
*   coordinates and the distance between the masses.
*
* CHANGES
*
*   --- Nov 2002 : Creation
*
******************************************************************************/

void Mechsim_Force_Interactions(VECTOR *Accel, VECTOR *Pos_Array)
{
  int i, j, k;
  DBL Fn_Val, lAxis;
  VECTOR Axis, Pos;
  VECTOR pot_force;

  for (i=0; i<MechsimOptions.Data.mass_count; i++)
  {
    for (j=0; j<i; j++)
    /*
       This double loop accesses each combination of
       two different masses exactly once
    */
    {
      VSub(Axis, Pos_Array[i], Pos_Array[j] );
      VLength(lAxis, Axis);

      for (k=0; k<MechsimOptions.Interactions.count; k++)
      {
        /*
           Determine point in the middle of the mass positions
           and evaluate function there
        */
        VLinComb2(Pos, 0.5, Pos_Array[i], 0.5, Pos_Array[j])

        FunctionCode *f = POVFPU_GetFunction(*(MechsimOptions.Interactions.Function[k]));

        POVFPU_SetLocal(X, Pos[X]);
        POVFPU_SetLocal(Y, Pos[Y]);
        POVFPU_SetLocal(Z, Pos[Z]);
        if (f->parameter_cnt > 3) POVFPU_SetLocal(T, lAxis); // additional distance parameter

        Fn_Val = POVFPU_Run(*(MechsimOptions.Interactions.Function[k]));

        if (fabs(Fn_Val) > MECHSIM_EPSILON)
        {
          /* apply forces */
          VScale(pot_force, Axis, -Fn_Val);
          VAddScaledEq(Accel[i],  1.0/MechsimOptions.Data.Masses[i].Mass, pot_force);
          VAddScaledEq(Accel[j], -1.0/MechsimOptions.Data.Masses[j].Mass, pot_force);
        }
      }
    }
  }

}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Gradient_Fields
*
* INPUT
*
*   Index - index of the mass
*   Pos_Array - positions of the masses
*
* OUTPUT
*
*   calculated acceleration is added to 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Gradient descent version for the field forces
*
* CHANGES
*
*   --- Aug 2002 : Creation
*   --- Nov 2002 : Added custom force fields
*
******************************************************************************/

void Mechsim_Gradient_Fields(int Index, VECTOR Accel, VECTOR *Pos_Array)
{
  int k, param;

  VAddEq( Accel, MechsimOptions.gravity);

  for (k=0; k<MechsimOptions.Fields.count; k++)
  {
    FunctionCode *f = POVFPU_GetFunction(*(MechsimOptions.Fields.Function[k]));

    POVFPU_SetLocal(X, Pos_Array[Index][X]);
    POVFPU_SetLocal(Y, Pos_Array[Index][Y]);
    POVFPU_SetLocal(Z, Pos_Array[Index][Z]);

    (void)POVFPU_Run(*(MechsimOptions.Fields.Function[k]));

    if (f->return_size >= 3)
      for(param = 0; param < 3; param++)
        Accel[param] += POVFPU_GetLocal(param);
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Force_Fields
*
* INPUT
*
*   Index - index of the mass
*   Pos_Array - positions of the masses
*
* OUTPUT
*
*   calculated acceleration is added to 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Calculates the field forces on a point mass
*
* CHANGES
*
*   --- Aug 2002 : Creation
*   --- Nov 2002 : Added custom force fields
*
******************************************************************************/

void Mechsim_Force_Fields(int Index, VECTOR Accel, VECTOR *Pos_Array)
{
  int k, param;

  VAddEq( Accel, MechsimOptions.gravity);

  for (k=0; k<MechsimOptions.Fields.count; k++)
  {
    FunctionCode *f = POVFPU_GetFunction(*(MechsimOptions.Fields.Function[k]));

    POVFPU_SetLocal(X, Pos_Array[Index][X]);
    POVFPU_SetLocal(Y, Pos_Array[Index][Y]);
    POVFPU_SetLocal(Z, Pos_Array[Index][Z]);

    (void)POVFPU_Run(*(MechsimOptions.Fields.Function[k]));

    if (f->return_size >= 3)
      for(param = 0; param < 3; param++)
        Accel[param] += POVFPU_GetLocal(param) / MechsimOptions.Data.Masses[Index].Mass;
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Gradients
*
* INPUT
*
*   Pos_Array - state of the masses
*   Vel_Array - *** Different usage here ***
*               stores the last position of the mass
*   Time - current time index for the environment function
*
* OUTPUT
*
*   calculated acceleration is returned via 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Wrapper for calculating the gradients of the potential fields
*
* CHANGES
*
*   --- Sep 2002 : Creation
*
******************************************************************************/

void Mechsim_Gradients(VECTOR *Gradient, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
 bounding_hashtable, int bounding_hashtable_size, int time_stamp)
{
  int i;
  MECHSIM_ENVIRONMENT *Env;

  /* first set all Gradients to 0.0 */
  for (i=0; i<MechsimOptions.Data.mass_count; i++)
    Make_Vector(Gradient[i], 0.0, 0.0, 0.0);

  /* calculate gradients that apply to several masses */
  if ((MechsimOptions.Calc_Collision != MECHSIM_COLLISION_NONE) ||
      (MechsimOptions.Calc_Collision_Faces  != MECHSIM_COLLISION_NONE))
    Mechsim_Gradient_Collide_Dynamic(Gradient, Pos_Array, Faces_BBox_Array, bounding_hashtable, bounding_hashtable_size, time_stamp);

  if (MechsimOptions.Interactions.count>0)
    Mechsim_Gradient_Interactions(Gradient, Pos_Array);

  if (MechsimOptions.Data.connect_count>0)
    Mechsim_Gradient_Connect(Gradient, Pos_Array);

  for (i=0; i<MechsimOptions.Data.mass_count; i++)
  {
    if ((!( MechsimOptions.Data.Masses[i].Flag & MECHSIM_FLAG_FIXED )) &&
        (MechsimOptions.Data.Masses[i].Attach < 0))
    {
      /* calculate gradients that only apply to this mass */

      if (MechsimOptions.Environment != NULL) // Environments
      {
        Env = MechsimOptions.Environment;

        while (Env != NULL)
        {
          if (Env->Function != NULL || Env->Object != NULL)
            Mechsim_Gradient_Collide_Environment(i, Gradient[i], Pos_Array, Vel_Array, Time, Env);

          Env = Env->Next;
        }
      }

      Mechsim_Gradient_Fields(i, Gradient[i], Pos_Array);

    }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Forces
*
* INPUT
*
*   Pos_Array, Vel_Array - state of the masses
*   Time - current time index (for the environment function)
*
* OUTPUT
*
*   calculated acceleration is returned via 'Accel'.
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Wrapper for calculating all forces on all masses
*
* CHANGES
*
*   --- Aug 2002 : Creation
*
******************************************************************************/

void Mechsim_Forces(VECTOR *Accel, VECTOR *Pos_Array, VECTOR *Vel_Array, DBL Time, MECHSIM_BBOX *Faces_BBox_Array, BOUNDING_HASHTABLE_ITEM*
 bounding_hashtable, int bounding_hashtable_size, int time_stamp)
{
  int i;
  MECHSIM_ENVIRONMENT *Env;

  /* first set all Accelerations to 0.0 */
  for (i=0; i<MechsimOptions.Data.mass_count; i++)
    Make_Vector(Accel[i], 0.0, 0.0, 0.0);

  /* calculate forces that apply to several masses */
  if ((MechsimOptions.Calc_Collision != MECHSIM_COLLISION_NONE) ||
      (MechsimOptions.Calc_Collision_Faces  != MECHSIM_COLLISION_NONE))
    Mechsim_Force_Collide_Dynamic(Accel, Pos_Array, Vel_Array, Faces_BBox_Array, bounding_hashtable, bounding_hashtable_size, time_stamp);

  if (MechsimOptions.Interactions.count>0)
    Mechsim_Force_Interactions(Accel, Pos_Array);

  if (MechsimOptions.Data.connect_count>0)
    Mechsim_Force_Connect(Accel, Pos_Array, Vel_Array);

  for (i=0; i<MechsimOptions.Data.mass_count; i++)
  {
    if ((!( MechsimOptions.Data.Masses[i].Flag & MECHSIM_FLAG_FIXED )) &&
        (MechsimOptions.Data.Masses[i].Attach < 0))
    {
      /* calculate forces that only apply to this mass */

      if (MechsimOptions.Environment != NULL) // Environments
      {
        Env = MechsimOptions.Environment;

        while (Env != NULL)
        {
          if (Env->Method==2)
          {
            if (Env->Function != NULL || Env->Object != NULL)
              Mechsim_Impact_Collide_Environment(i, Pos_Array, Vel_Array, Time, Env);
          }
          else
          {
            if (Env->Function != NULL || Env->Object != NULL)
              Mechsim_Force_Collide_Environment(i, Accel[i], Pos_Array, Vel_Array, Time, Env);
          }
          Env = Env->Next;
        }
      }

      Mechsim_Force_Fields(i, Accel[i], Pos_Array);

      #if(MECHSIM_DEBUG == 1)
        if ((Steps==0) && (i==0)) Status_Info("\nAcc=<%g, %g, %g>\n", Accel[0][X],Accel[0][Y],Accel[0][Z] );
      #endif
    }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Check_Col
*
* INPUT
*
*   Pos - new position
*   Old_Pos - old position to restore when mass is inside environment shapes
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Penetration prevention for object intersection based environments
*
* CHANGES
*
*   --- Oct 2002 : Creation
*
******************************************************************************/

void Mechsim_Check_Col(VECTOR Pos, VECTOR Old_Pos)
{
  MECHSIM_ENVIRONMENT *Env;

  if (MechsimOptions.Environment != NULL) // Environments
  {
    Env = MechsimOptions.Environment;

    while (Env != NULL)
    {
      if (Env->Function == NULL && Env->Object != NULL) // only for object based environments
        if (Inside_Object(Pos, Env->Object))
        {
          Assign_Vector(Pos, Old_Pos);
          Increase_Counter(stats[MechSim_Environment_Penetrations]);
        }

      Env = Env->Next;
    }
  }

}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Check_Instability
*
* INPUT
*
*   Pos - position of mass
*   Vel - velocity of mass
*   Accel - acceleration of mass
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Tries to do a rough estimation if the simulation grows instable and
*   drifts into unrealistic stages. Extreme instabilities might cause
*   program crashes if not detected.   
*
* CHANGES
*
*   --- Dec 2002 : Creation
*
******************************************************************************/

void Mechsim_Check_Instability(VECTOR Pos, VECTOR Vel, VECTOR Accel)
{
  DBL lAcc;

  VLength(lAcc, Accel);

  /*
    simply test if the acceleration exceeds a certain value. 
    the exact value is probably not critical since an acceleration 
    just a bit lower will be likely to cause an even higher value 
    some steps later. 
  */

  if (lAcc > 1.0e7) 
    Error("Instability detected during simulation.  Using smaller\ntime steps or changing the system parameters might help.\n");

}


/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Simulate
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Do the actual mechanics simulation
*
* CHANGES
*
*   --- Aug 2002 : Creation
*
******************************************************************************/

void Mechsim_Simulate(void)
{
  int i, Steps;
  DBL Time;
  DBL TotalTime;

  VECTOR *Pos_Array;
  VECTOR *Vel_Array;
  VECTOR *Pos2;
  VECTOR *Vel2;
  VECTOR *k1_p;
  VECTOR *k1_v;

  VECTOR k2_p;
  VECTOR k2_v;

  VECTOR Old_Pos;

  VECTOR *Accel;

  VECTOR *Attach_Move = NULL;

  int param;

  MECHSIM_BBOX *Faces_BBox_Array;  
  
  BOUNDING_HASHTABLE_ITEM *bounding_hashtable;
  
  int bounding_hashtable_size;
  int time_stamp;

  if (MechsimOptions.Step_Count>0)
  {

    Status_Info("\nCalculating Mechanics Simulation...");

    /* fill the group data structures */
    Mechsim_Prepare_Data();

    /* Allocate data blocks for integration */
    Pos_Array = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");
    Vel_Array = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");

    /* Read data from MechsimOptions.Data  */
    for (i=0; i<MechsimOptions.Data.mass_count; i++)
    {
      Assign_Vector(Pos_Array[i], MechsimOptions.Data.Masses[i].Position);
      Assign_Vector(Vel_Array[i], MechsimOptions.Data.Masses[i].Velocity);
    }

    /* Attachments movement data cache  */
    if (MechsimOptions.Attachments.count>0)
      Attach_Move = (VECTOR *)POV_MALLOC(MechsimOptions.Attachments.count*sizeof (VECTOR), "mechanics simulation data block");

    TotalTime = MechsimOptions.Step_Count*MechsimOptions.Time_Step;
    Time = MechsimOptions.Start_Time;

    if ((MechsimOptions.Bounding != MECHSIM_FACE_BOUNDING_NO) &&
        (MechsimOptions.Bounding != MECHSIM_FACE_BOUNDING_BBOX) &&
        (MechsimOptions.Bounding != MECHSIM_FACE_BOUNDING_HASH))
    {
      Error("mechanics bounding method %d not supported", MechsimOptions.Bounding);
    }
/*    else
    {
      Render_Info("\nBounding %d.\n", MechsimOptions.Bounding);
    }*/
    
    /* Just turn bounding on if it would be faster */
    if (((MechsimOptions.Data.max_masses < 100) || (MechsimOptions.Data.max_faces < 100)) 
      && (MechsimOptions.Data.max_masses*MechsimOptions.Data.max_faces < 10000))
    {
      if (((MechsimOptions.Data.max_masses < 10) || (MechsimOptions.Data.max_faces < 10)) 
        && (MechsimOptions.Data.max_masses*MechsimOptions.Data.max_faces < 100))
      {
        MechsimOptions.Bounding = MECHSIM_FACE_BOUNDING_NO;
      }
      else
      {
        if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH)
	{
	  MechsimOptions.Bounding = MECHSIM_FACE_BOUNDING_BBOX;
	}
      }
    }
    
    /* Initialization of bounding */
    if (MechsimOptions.Bounding >= MECHSIM_FACE_BOUNDING_BBOX)
    {
      if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH)
      {
        bounding_hashtable_size = MechsimOptions.Data.max_faces;
        bounding_hashtable = (BOUNDING_HASHTABLE_ITEM*)POV_MALLOC(bounding_hashtable_size*sizeof(BOUNDING_HASHTABLE_ITEM), 
          "mechsim faces bounding hashtable");
	  
        memset(bounding_hashtable, 0, bounding_hashtable_size*sizeof(BOUNDING_HASHTABLE_ITEM));
	
	time_stamp = 1;
	
      }
      Faces_BBox_Array = (MECHSIM_BBOX *)POV_MALLOC(MechsimOptions.Data.max_faces*sizeof (MECHSIM_BBOX), 
        "mechsim faces bounding boxes");
    }
    
    /* clear the line for statur display */
    Status_Info("\n");

    switch (MechsimOptions.Method)
    {
      case MECHSIM_METHOD_EULER:             // ---- EULER ----

        /* Allocate additional data blocks for integration */
        Accel = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");

        Steps=0;

        while (Steps < MechsimOptions.Step_Count)
        {

          /* calculate the movement of the attachments during this time step */
          if (MechsimOptions.Attachments.count>0)
          {
            for (i=0; i<MechsimOptions.Attachments.count; i++)
            {
              FunctionCode *f = POVFPU_GetFunction(*(MechsimOptions.Attachments.Function[i]));

              POVFPU_SetLocal(f->return_size, Time+MechsimOptions.Time_Step);
              (void)POVFPU_Run(*(MechsimOptions.Attachments.Function[i]));
              for(param = 0; param < min((int)f->return_size, 3); param++) Attach_Move[i][param] = POVFPU_GetLocal(param);
              
              POVFPU_SetLocal(f->return_size, Time);
              (void)POVFPU_Run(*(MechsimOptions.Attachments.Function[i]));
              for(param = 0; param < min((int)f->return_size, 3); param++) Attach_Move[i][param] -= POVFPU_GetLocal(param);

            }
          }

          /* calculate all forces */
          Mechsim_Forces(Accel, Pos_Array, Vel_Array, Time, Faces_BBox_Array, bounding_hashtable, bounding_hashtable_size, time_stamp);
	  if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH) time_stamp++;

          Do_Cooperate(1);

          for (i=0; i<MechsimOptions.Data.mass_count; i++)
          {
            if (!( MechsimOptions.Data.Masses[i].Flag & MECHSIM_FLAG_FIXED ))
            {
              if (MechsimOptions.Data.Masses[i].Attach >= 0)
              {
                /* move the attached masses */
                VAddEq(Pos_Array[i], Attach_Move[MechsimOptions.Data.Masses[i].Attach]);
                VAddScaledEq(Vel_Array[i], 1.0/MechsimOptions.Time_Step, Attach_Move[MechsimOptions.Data.Masses[i].Attach]);
              }
              else
              {
                /* move the free masses */
                Assign_Vector(Old_Pos, Pos_Array[i]);
                /* y_new = y_old + h*f(y_old) */
                VAddScaledEq(Pos_Array[i], MechsimOptions.Time_Step, Vel_Array[i]);
                VAddScaledEq(Vel_Array[i], MechsimOptions.Time_Step, Accel[i]);

                Do_Cooperate(1);

                Mechsim_Check_Instability(Pos_Array[i], Vel_Array[i], Accel[i]);
                Mechsim_Check_Col(Pos_Array[i], Old_Pos);
              }
            }
          }

          Status_Info("\rCalculating Mechanics Simulation... Time=%.4fs (%d percent)", Time, (int)(100.0*Steps/(MechsimOptions.Step_Count-1)) );

          Time += MechsimOptions.Time_Step;
          Steps++;

          Increase_Counter(stats[MechSim_Steps]);

        }

        POV_FREE(Accel);

        break;

      case MECHSIM_METHOD_HEUN:

        /* Allocate additional data blocks for integration */
        Pos2 = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");
        Vel2 = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");
        k1_p = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");
        k1_v = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");

        Accel = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");

        Steps=0;

        while (Steps < MechsimOptions.Step_Count)
        {
          /* =================================================================== */


          /* calculate the movement of the attachments during this time step */
          if (MechsimOptions.Attachments.count>0)
          {
            for (i=0; i<MechsimOptions.Attachments.count; i++)
            {
              FunctionCode *f = POVFPU_GetFunction(*(MechsimOptions.Attachments.Function[i]));

              POVFPU_SetLocal(f->return_size, Time+MechsimOptions.Time_Step);
              (void)POVFPU_Run(*(MechsimOptions.Attachments.Function[i]));
              for(param = 0; param < min((int)f->return_size, 3); param++) Attach_Move[i][param] = POVFPU_GetLocal(param);

              POVFPU_SetLocal(f->return_size, Time);
              (void)POVFPU_Run(*(MechsimOptions.Attachments.Function[i]));
              for(param = 0; param < min((int)f->return_size, 3); param++) Attach_Move[i][param] -= POVFPU_GetLocal(param);
            }
          }

          /* calculate all forces */
          Mechsim_Forces(Accel, Pos_Array, Vel_Array, Time, Faces_BBox_Array, bounding_hashtable, bounding_hashtable_size, time_stamp);
	  if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH) time_stamp++;
	  
          Do_Cooperate(1);

          for (i=0; i<MechsimOptions.Data.mass_count; i++)
          {
            if ((!( MechsimOptions.Data.Masses[i].Flag & MECHSIM_FLAG_FIXED )) &&
                (MechsimOptions.Data.Masses[i].Attach < 0))
            {
              /* k1 = f(y_old) */
              Assign_Vector(k1_p[i], Vel_Array[i]);
              Assign_Vector(k1_v[i], Accel[i]);

              /* y2 = y_old + h*f(y_old) */
              Pos2[i][X] = Pos_Array[i][X] + MechsimOptions.Time_Step*Vel_Array[i][X];
              Pos2[i][Y] = Pos_Array[i][Y] + MechsimOptions.Time_Step*Vel_Array[i][Y];
              Pos2[i][Z] = Pos_Array[i][Z] + MechsimOptions.Time_Step*Vel_Array[i][Z];

              Vel2[i][X] = Vel_Array[i][X] + MechsimOptions.Time_Step*Accel[i][X];
              Vel2[i][Y] = Vel_Array[i][Y] + MechsimOptions.Time_Step*Accel[i][Y];
              Vel2[i][Z] = Vel_Array[i][Z] + MechsimOptions.Time_Step*Accel[i][Z];

              Do_Cooperate(1);
            }
          }

          /* =================================================================== */

          /* calculate all forces #FIXME: time value# */
          Mechsim_Forces(Accel, Pos2, Vel2, Time, Faces_BBox_Array, bounding_hashtable, bounding_hashtable_size, time_stamp);
	  if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH) time_stamp++;

          Do_Cooperate(1);

          for (i=0; i<MechsimOptions.Data.mass_count; i++)
          {
            if (!( MechsimOptions.Data.Masses[i].Flag & MECHSIM_FLAG_FIXED ))
            {
              if (MechsimOptions.Data.Masses[i].Attach >= 0)
              {
                /* move the attached masses */
                VAddEq(Pos_Array[i], Attach_Move[MechsimOptions.Data.Masses[i].Attach]);
                VAddScaledEq(Vel_Array[i], 1.0/MechsimOptions.Time_Step, Attach_Move[MechsimOptions.Data.Masses[i].Attach]);
              }
              else
              {
                /* move the free masses */

                /* k2 = f(y2) */
                Assign_Vector(k2_p, Vel2[i]);
                Assign_Vector(k2_v, Accel[i]);

                /* y_new = y_old + (h/2)*(k1 + k2) */
                Pos_Array[i][X] = Pos_Array[i][X] + MechsimOptions.Time_Step*0.5*(k1_p[i][X]+k2_p[X]);
                Pos_Array[i][Y] = Pos_Array[i][Y] + MechsimOptions.Time_Step*0.5*(k1_p[i][Y]+k2_p[Y]);
                Pos_Array[i][Z] = Pos_Array[i][Z] + MechsimOptions.Time_Step*0.5*(k1_p[i][Z]+k2_p[Z]);

                Vel_Array[i][X] = Vel_Array[i][X] + MechsimOptions.Time_Step*0.5*(k1_v[i][X]+k2_v[X]);
                Vel_Array[i][Y] = Vel_Array[i][Y] + MechsimOptions.Time_Step*0.5*(k1_v[i][Y]+k2_v[Y]);
                Vel_Array[i][Z] = Vel_Array[i][Z] + MechsimOptions.Time_Step*0.5*(k1_v[i][Z]+k2_v[Z]);

                Do_Cooperate(1);

                Mechsim_Check_Instability(Pos_Array[i], Vel_Array[i], Accel[i]);
              }
            }
          }
          /* =================================================================== */

          Status_Info("\rCalculating Mechanics Simulation... Time=%.4fs (%d percent)", Time, (int)(100.0*Steps/(MechsimOptions.Step_Count-1)) );

          Time += MechsimOptions.Time_Step;
          Steps++;

          Increase_Counter(stats[MechSim_Steps]);

        }

        POV_FREE(Pos2);
        POV_FREE(Vel2);
        POV_FREE(k1_p);
        POV_FREE(k1_v);
        POV_FREE(Accel);

        break;

    case MECHSIM_METHOD_GRADIENT:

        /* Allocate additional data blocks for gradient descent method */
        Accel = (VECTOR *)POV_MALLOC(MechsimOptions.Data.max_masses*sizeof (VECTOR), "mechanics simulation data block");

        for (i=0; i<MechsimOptions.Data.mass_count; i++) Assign_Vector(Vel_Array[i], Pos_Array[i]);

        Steps=0;

        while (Steps < MechsimOptions.Step_Count)
        {
          /* calculate the movement of the attachments during this time step */
          if (MechsimOptions.Attachments.count>0)
          {
            for (i=0; i<MechsimOptions.Attachments.count; i++)
            {
              FunctionCode *f = POVFPU_GetFunction(*(MechsimOptions.Attachments.Function[i]));

              POVFPU_SetLocal(f->return_size, Time+MechsimOptions.Time_Step);
              (void)POVFPU_Run(*(MechsimOptions.Attachments.Function[i]));
              for(param = 0; param < min((int)f->return_size, 3); param++) Attach_Move[i][param] = POVFPU_GetLocal(param);

              POVFPU_SetLocal(f->return_size, Time);
              (void)POVFPU_Run(*(MechsimOptions.Attachments.Function[i]));
              for(param = 0; param < min((int)f->return_size, 3); param++) Attach_Move[i][param] -= POVFPU_GetLocal(param);
            }
          }

          /* calculate all gradients */
          Mechsim_Gradients(Accel, Pos_Array, Vel_Array, Time, Faces_BBox_Array, bounding_hashtable, bounding_hashtable_size, time_stamp);
	  if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH) time_stamp++;

          Do_Cooperate(1);

          for (i=0; i<MechsimOptions.Data.mass_count; i++)
          {
            if (!( MechsimOptions.Data.Masses[i].Flag & MECHSIM_FLAG_FIXED ))
            {
              if (MechsimOptions.Data.Masses[i].Attach >= 0)
              {
                /* move the attached masses */
                VAddEq(Pos_Array[i], Attach_Move[MechsimOptions.Data.Masses[i].Attach]);
              }
              else
              {
                /* move the free masses */

                // store last position to allow object intersection based environments
                // see function 'Mechsim_Gradient_Collide_Environment()' for how this is used
                Assign_Vector(Vel_Array[i], Pos_Array[i]);
                Assign_Vector(Old_Pos, Pos_Array[i]);

                /* move masses in direction of the gradient */
                Pos_Array[i][X] = Pos_Array[i][X] + MechsimOptions.Time_Step*Accel[i][X];
                Pos_Array[i][Y] = Pos_Array[i][Y] + MechsimOptions.Time_Step*Accel[i][Y];
                Pos_Array[i][Z] = Pos_Array[i][Z] + MechsimOptions.Time_Step*Accel[i][Z];

                Do_Cooperate(1);

                Mechsim_Check_Instability(Pos_Array[i], Accel[i], Accel[i]);
                Mechsim_Check_Col(Pos_Array[i], Old_Pos);
              }
            }
          }

          Status_Info("\rCalculating Mechanics Simulation... Time=%.4fs (%d percent)", Time, (int)(100.0*Steps/(MechsimOptions.Step_Count-1)) );

          Time += MechsimOptions.Time_Step;
          Steps++;

          Increase_Counter(stats[MechSim_Steps]);

        }

        POV_FREE(Accel);

        // make sure we save zero velocity to file
        for (i=0; i<MechsimOptions.Data.mass_count; i++)
        {
          Vel_Array[i][X] = 0.0;
          Vel_Array[i][Y] = 0.0;
          Vel_Array[i][Z] = 0.0;
        }

        break;

      default:
        Error("mechanics simulation method %d not supported", MechsimOptions.Method);
        break;
    }


    /* Write data back to MechsimOptions.Data  */
    for (i=0; i<MechsimOptions.Data.mass_count; i++)
    {
      Assign_Vector(MechsimOptions.Data.Masses[i].Position, Pos_Array[i]);
      Assign_Vector(MechsimOptions.Data.Masses[i].Velocity, Vel_Array[i]);
    }


    POV_FREE(Pos_Array);
    POV_FREE(Vel_Array);

    /* Deinitialization of bounding */
    if (MechsimOptions.Bounding >= MECHSIM_FACE_BOUNDING_BBOX)
    {
      if (MechsimOptions.Bounding == MECHSIM_FACE_BOUNDING_HASH)
      {
  	if (bounding_hashtable != NULL)
  	{
	  for (i = 0; i < bounding_hashtable_size; i++)
    	  {
      	    if (bounding_hashtable[i].index != NULL)
	    {
              POV_FREE(bounding_hashtable[i].index);
      	    }
    	  }
    	  POV_FREE(bounding_hashtable);
  	}
      }
      POV_FREE(Faces_BBox_Array);
    }

    Status_Info("\n");

  }

  if (MechsimOptions.Save_File) Mechsim_save_file();

}


/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Create_Group
*
* INPUT
*
*   Count - number of groups to create
*
* OUTPUT
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Create new group data structures
*
* CHANGES
*
*   --- Sep 2002 : Creation.
*
******************************************************************************/
void Mechsim_Create_Groups(int Count)
{
  int i;

  if (MechsimOptions.Data.max_groups==0) // first time
  {
    MechsimOptions.Data.max_groups = Count;
    MechsimOptions.Data.Groups = (MECHSIM_GROUP *)POV_MALLOC(MechsimOptions.Data.max_groups*sizeof (MECHSIM_GROUP), "mechanics simulation group data block");
  }
  else
  {
    if (MechsimOptions.Data.max_groups+Count >= INT_MAX)
    {
      Error("Too many groups in simulation data");
    }
    MechsimOptions.Data.max_groups += Count;
    MechsimOptions.Data.Groups = (MECHSIM_GROUP *)POV_REALLOC(MechsimOptions.Data.Groups, MechsimOptions.Data.max_groups*sizeof (MECHSIM_GROUP), "mechanics simulation group data block");
  }

  // for the start allocate space for 20 masses
  for (i=MechsimOptions.Data.max_groups-Count; i<MechsimOptions.Data.max_groups; i++)
  {
    MechsimOptions.Data.Groups[i].max_masses = 20;
    MechsimOptions.Data.Groups[i].max_connects = 20;
    MechsimOptions.Data.Groups[i].max_faces = 20;

    MechsimOptions.Data.Groups[i].Masses = (MECHSIM_MASS **)POV_MALLOC(MechsimOptions.Data.Groups[i].max_masses*sizeof (MECHSIM_MASS *), "mechanics simulation mass pointer data block");
    MechsimOptions.Data.Groups[i].Connects = (MECHSIM_CONNECT **)POV_MALLOC(MechsimOptions.Data.Groups[i].max_connects*sizeof (MECHSIM_CONNECT *), "mechanics simulation connection pointer data block");
    MechsimOptions.Data.Groups[i].Faces = (MECHSIM_FACE **)POV_MALLOC(MechsimOptions.Data.Groups[i].max_faces*sizeof (MECHSIM_FACE *), "mechanics simulation face pointer data block");

    MechsimOptions.Data.Groups[i].mass_count = 0;
    MechsimOptions.Data.Groups[i].connect_count = 0;
    MechsimOptions.Data.Groups[i].faces_count = 0;
  }

}


/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_Prepare_Data
*
* INPUT
*
* OUTPUT
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Prepares additional Simulation data structures (groups)
*
*   The groups are arrays of structures like the 'Masses', 'Connects' and
*   'Faces' fields.  The group structure contains arrays of pointers to
*   the actual data.  The purpose of all this is to speed up the collision
*   calculations.
*
*   (Data)
*     |_______(Masses) -> array of MECHSIM_MASS
*     |                                                         pointers to
*     |_______(Connects) -> array of MECHSIM_CONNECT          <------------
*     |                                                                    |
*     |_______(Faces) -> array of MECHSIM_FACE                             |
*     |                                                                    |
*     |_______(Groups) -> array of MECHSIM_GROUP                           |
*                |                                                         |
*                |_______(Masses) -> array of (MECHSIM_MASS *)       |     |
*                |                                                   |     |
*                |_______(Connects) -> array of (MECHSIM_CONNECT *)  |-----
*                |                                                   |
*                |_______(Faces) -> array of (MECHSIM_FACE *)        |
*
*
*
*
* CHANGES
*
*   --- Sep 2002 : Creation.
*
******************************************************************************/
void Mechsim_Prepare_Data(void)
{
  int i, Group;

  /*------ masses ------*/
  for (i=0; i<MechsimOptions.Data.mass_count; i++)
  {
    Group = MechsimOptions.Data.Masses[i].Group;

    // resize group array if necessary
    if ( Group >= MechsimOptions.Data.max_groups )
    {
      Mechsim_Create_Groups(Group-MechsimOptions.Data.max_groups+1);
    }

    if ( Group >= MechsimOptions.Data.group_count ) MechsimOptions.Data.group_count = Group+1;

    // resize mass pointer arrays of the group if necessary
    if ( MechsimOptions.Data.Groups[Group].mass_count >= MechsimOptions.Data.Groups[Group].max_masses )
    {
      if (MechsimOptions.Data.Groups[Group].max_masses >= INT_MAX/2)
      {
        Error("Too many masses in simulation data");
      }
      MechsimOptions.Data.Groups[Group].max_masses *= 2;
      MechsimOptions.Data.Groups[Group].Masses =
        (MECHSIM_MASS **)POV_REALLOC(MechsimOptions.Data.Groups[Group].Masses,
                                     MechsimOptions.Data.Groups[Group].max_masses*
                                     sizeof (MECHSIM_MASS *), "mechanics simulation mass pointer data block");
    }

    // set mass pointer
    MechsimOptions.Data.Groups[Group].Masses[MechsimOptions.Data.Groups[Group].mass_count] = &MechsimOptions.Data.Masses[i];

    // set counter
    MechsimOptions.Data.Groups[Group].mass_count++;

  }

  /*------ connects ------*/
  for (i=0; i<MechsimOptions.Data.connect_count; i++)
  {
    Group = MechsimOptions.Data.Connects[i].Group;

    // resize group array if necessary
    if ( Group >= MechsimOptions.Data.max_groups )
    {
      Mechsim_Create_Groups(Group-MechsimOptions.Data.max_groups+1);
    }

    if ( Group >= MechsimOptions.Data.group_count ) MechsimOptions.Data.group_count = Group+1;

    // resize connect pointer arrays of the group if necessary
    if ( MechsimOptions.Data.Groups[Group].connect_count >= MechsimOptions.Data.Groups[Group].max_connects )
    {
      if (MechsimOptions.Data.Groups[Group].max_connects >= INT_MAX/2)
      {
        Error("Too many connects in simulation data");
      }
      MechsimOptions.Data.Groups[Group].max_connects *= 2;
      MechsimOptions.Data.Groups[Group].Connects =
        (MECHSIM_CONNECT **)POV_REALLOC(MechsimOptions.Data.Groups[Group].Connects,
                                        MechsimOptions.Data.Groups[Group].max_connects*
                                        sizeof (MECHSIM_CONNECT *), "mechanics simulation connect pointer data block");
    }

    // set connect pointer
    MechsimOptions.Data.Groups[Group].Connects[MechsimOptions.Data.Groups[Group].connect_count] = &MechsimOptions.Data.Connects[i];

    // set counter
    MechsimOptions.Data.Groups[Group].connect_count++;

  }

  /*------ faces ------*/
  for (i=0; i<MechsimOptions.Data.faces_count; i++)
  {
    Group = MechsimOptions.Data.Faces[i].Group;

    // resize group array if necessary
    if ( Group >= MechsimOptions.Data.max_groups )
    {
      Mechsim_Create_Groups(Group-MechsimOptions.Data.max_groups+1);
    }

    if ( Group >= MechsimOptions.Data.group_count ) MechsimOptions.Data.group_count = Group+1;

    // resize connect pointer arrays of the group if necessary
    if ( MechsimOptions.Data.Groups[Group].faces_count >= MechsimOptions.Data.Groups[Group].max_faces )
    {
      if (MechsimOptions.Data.Groups[Group].max_faces >= INT_MAX/2)
      {
        Error("Too many faces in simulation data");
      }
      MechsimOptions.Data.Groups[Group].max_faces *= 2;
      MechsimOptions.Data.Groups[Group].Faces =
        (MECHSIM_FACE **)POV_REALLOC(MechsimOptions.Data.Groups[Group].Faces,
                                     MechsimOptions.Data.Groups[Group].max_faces*
                                     sizeof (MECHSIM_FACE *), "mechanics simulation face pointer data block");
    }

    // set connect pointer
    MechsimOptions.Data.Groups[Group].Faces[MechsimOptions.Data.Groups[Group].faces_count] = &MechsimOptions.Data.Faces[i];

    // set counter
    MechsimOptions.Data.Groups[Group].faces_count++;

  }


}


/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_read_file
*
* INPUT
*   file descriptor handle of file (already opened).
*   Group - the group the elements should be sorted into
*           if zero use groups as in file
*
* OUTPUT
*
* RETURNS - Success 1 / failure 0
*
* AUTHOR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Read in a mechanics simulation data file
*
* CHANGES
*
*   --- Aug 2002 : Creation based on radiosity data read function.
*
******************************************************************************/
bool Mechsim_read_file(POV_ISTREAM *fd, int Group)
{
  bool retval, got_eof=false;
  long line_num, goodreads;
  int goodparse, fformat;
  char file_id[5];
  char line[MAX_STR+2];

  int mass_count, connect_count, faces_count;
  char *Pos;
  VECTOR Posi, Vel;
  double Rad, Mass, Stiff, Damp, Length;
  int Idx1, Idx2, Idx3, Flag, Grp, Attach;
  int prev_mass_count;

  if ( fd != NULL )
  {
    if ( fd->getline (line, MAX_STR).eof () )
    {
      Warning(0, "Error reading mechanics simulation data (unexpected file end)");
      return false;
    }

    goodparse = 1;
    goodreads = 0;

    if (sscanf(line, "%4s, %d,\n", file_id, &fformat) == 2)
      if ((file_id[0] == 'M') &&
          (file_id[1] == 'S') &&
          (file_id[2] == 'I') &&
          (file_id[3] == 'M')
         )
      {
        switch ( fformat )
        {
        case 1: case 2: case 3:
        {

          if (fformat>2)  // file subtype 3 -> read start time
          {
            if (!(got_eof = fd->getline (line, MAX_STR).eof ()) && goodparse)
               Pos = Read_Float(line, &MechsimOptions.Start_Time);
            else Warning(0, "\nError reading mechanics simulation data (wrong file format)");
          }

          READ_INTEGER(mass_count)
          READ_INTEGER(connect_count)
          READ_INTEGER(faces_count)

          #if(MECHSIM_DEBUG == 1)
            Debug_Info("header %d %d %d\n", mass_count, connect_count, faces_count);
          #endif

          // save current mass count to get the connections and faces right.
          // their mass indices have to be increased by this number
          prev_mass_count = MechsimOptions.Data.mass_count;

          line_num = 0;

          while (!(got_eof = fd->getline (line, MAX_STR).eof ()) && goodparse)
          {

            Pos = strchr(line, '\n');           // remove line break
            if (Pos != NULL) Pos[0] = '\0';
            Pos = strchr(line, '\r');           // remove line break
            if (Pos != NULL) Pos[0] = '\0';

            line_num++;

            #if(MECHSIM_DEBUG == 1)
              //Debug_Info("line %d (%s)\n", line_num, line);
              if (line_num==1) Debug_Info("\n(%s)\n", line);
            #endif

            if (line_num <= mass_count)         // ================ masses ===================
            {
              Pos = Read_Vector(line, Posi);
              Pos = Read_Vector(Pos, Vel);
              Pos = Read_Float(Pos, &Mass);
              Pos = Read_Float(Pos, &Rad);
              Pos = Read_Int(Pos, &Flag);

              if (fformat>1) // file contains group specifications
              {
                Pos = Read_Int(Pos, &Grp);
                if (Grp<0)
                {
                  Warning(0, "mechanics simulation data (line %d): group index out of range", line_num);
                  Grp = 0;
                }
              }
              else Grp = 0;

              if (fformat>2)  // file subtype 3 -> read attachment index
              {
                Pos = Read_Int(Pos, &Attach);
              }
              else Attach = -1;

              if ( MechsimOptions.Data.mass_count >= MechsimOptions.Data.max_masses )
              {
                if (MechsimOptions.Data.max_masses >= INT_MAX/2)
                {
                  Error("Too many masses in simulation data");
                }
                MechsimOptions.Data.max_masses *= 2;
                MechsimOptions.Data.Masses = (MECHSIM_MASS *)POV_REALLOC(MechsimOptions.Data.Masses, MechsimOptions.Data.max_masses*sizeof (MECHSIM_MASS), "mechanics simulation mass data block");
              }

              Assign_Vector(MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Position, Posi);
              Assign_Vector(MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Velocity, Vel);
              MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Mass = Mass;
              MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Radius = Rad;
              MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Flag = Flag;
              MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Attach = Attach;

              if (Group==0) MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Group = Grp; // no group given -> use those from file
              else MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Group = Group;

              MechsimOptions.Data.Masses[MechsimOptions.Data.mass_count].Index = MechsimOptions.Data.mass_count;

              MechsimOptions.Data.mass_count++;

              #if(MECHSIM_DEBUG == 1)
                Debug_Info("mass load\n");
              #endif

              goodreads++;

            }
            else if (line_num <= mass_count+connect_count)    // ================ connections ================
            {
              Pos = Read_Int(line, &Idx1);
              Pos = Read_Int(Pos, &Idx2);
              Pos = Read_Float(Pos, &Length);
              Pos = Read_Float(Pos, &Stiff);
              Pos = Read_Float(Pos, &Damp);
              if (fformat==2) // file contains group specifications
              {
                Pos = Read_Int(Pos, &Grp);
                if (Grp<0)
                {
                  Warning(0, "mechanics simulation data (line %d): group index out of range", line_num);
                  Grp = 0;
                }
              }
              else Grp = 0;

              if ((Idx1>=MechsimOptions.Data.mass_count) || (Idx2>=MechsimOptions.Data.mass_count))
              {
                Warning(0, "mechanics simulation data (line %d): connection index out of range, ignoring", line_num);
                continue;
              }

              if ( MechsimOptions.Data.connect_count >= MechsimOptions.Data.max_connects )
              {
                if (MechsimOptions.Data.max_connects >= INT_MAX/2)
                {
                  Error("Too many connections in simulation data");
                }
                MechsimOptions.Data.max_connects *= 2;
                MechsimOptions.Data.Connects = (MECHSIM_CONNECT *)POV_REALLOC(MechsimOptions.Data.Connects, MechsimOptions.Data.max_connects*sizeof (MECHSIM_CONNECT), "mechanics simulation connection data block");
              }

              MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Idx1 = Idx1+prev_mass_count;
              MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Idx2 = Idx2+prev_mass_count;

              MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Length = Length;
              MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Stiffness = Stiff;
              MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Damping = Damp;
              if (Group==0) MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Group = Grp; // no group given -> use those from file
              else MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Group = Group;

              MechsimOptions.Data.Connects[MechsimOptions.Data.connect_count].Index = MechsimOptions.Data.connect_count;

              MechsimOptions.Data.connect_count++;

              #if(MECHSIM_DEBUG == 1)
                Debug_Info("connect load %d %d %g %g %g\n", Idx1, Idx2, Length, Stiff, Damp);
              #endif

              goodreads++;

            }
            else if (line_num <= mass_count+connect_count+faces_count)  // ================ faces ================
            {
              Pos = Read_Int(line, &Idx1);
              Pos = Read_Int(Pos, &Idx2);
              Pos = Read_Int(Pos, &Idx3);
              if (fformat==2) // file contains group specifications
              {
                Pos = Read_Int(Pos, &Grp);
                if (Grp<0)
                {
                  Warning(0, "mechanics simulation data (line %d): group index out of range", line_num);
                  Grp = 0;
                }
              }
              else Grp = 0;

              if ((Idx1>=MechsimOptions.Data.mass_count) || (Idx2>=MechsimOptions.Data.mass_count) || (Idx3>=MechsimOptions.Data.mass_count))
              {
                Warning(0, "mechanics simulation data (line %d): face index out of range, ignoring", line_num);
                continue;
              }

              if ( MechsimOptions.Data.faces_count >= MechsimOptions.Data.max_faces )
              {
                if (MechsimOptions.Data.max_faces >= INT_MAX/2)
                {
                  Error("Too many faces in simulation data");
                }
                MechsimOptions.Data.max_faces *= 2;
                MechsimOptions.Data.Faces = (MECHSIM_FACE *)POV_REALLOC(MechsimOptions.Data.Faces, MechsimOptions.Data.max_faces*sizeof (MECHSIM_FACE), "mechanics simulation face data block");
              }

              MechsimOptions.Data.Faces[MechsimOptions.Data.faces_count].Idx1 = Idx1+prev_mass_count;
              MechsimOptions.Data.Faces[MechsimOptions.Data.faces_count].Idx2 = Idx2+prev_mass_count;
              MechsimOptions.Data.Faces[MechsimOptions.Data.faces_count].Idx3 = Idx3+prev_mass_count;
              if (Group==0) MechsimOptions.Data.Faces[MechsimOptions.Data.faces_count].Group = Grp; // no group given -> use those from file
              else MechsimOptions.Data.Faces[MechsimOptions.Data.faces_count].Group = Group;

              MechsimOptions.Data.Faces[MechsimOptions.Data.faces_count].Index = MechsimOptions.Data.faces_count;

              MechsimOptions.Data.faces_count++;

              goodreads++;

            }

          }     /* end while-reading loop */


        break;
        }
        default:     /* other file types */
        {
          Error("mechanics simulation data format %d not supported", fformat );
          return false;
        }
        }
      }     /* end if "MSIM" */


    if ( !got_eof  ||  !goodparse ) {
      Status_Info("\nError reading mechanics simulation data");
      retval = false;
    }
    else
    {
      if ( goodreads > 0 )
      {
        Render_Info("\nLoaded %d masses, %d connections and %d faces from mechanics simulation data file %s.\n",
                    MechsimOptions.Data.mass_count,
                    MechsimOptions.Data.connect_count,
                    MechsimOptions.Data.faces_count,
                    MechsimOptions.Load_File_Name);
      }
      else
      {
        Status_Info("\nUnable to read any values from mechanics simulation data file.");
      }
      retval = true;
    }
  }
  else
  {
    retval = false;
  }

  return retval;
}

/*****************************************************************************
*
* FUNCTION
*
*   Mechsim_save_file
*
* INPUT
*   file descriptor handle of file (already opened).
*
* OUTPUT
*
* RETURNS 1 for success, 0 for failure.
*
* AUTHOUR
*
*   Christoph Hormann
*
* DESCRIPTION
*
*   Write mechanics simulation data to a file
*
* CHANGES
*
*   --- Aug 2002 : Creation
*
******************************************************************************/
bool Mechsim_save_file(void)
{
  POV_OSTREAM *fd;
  int i;
  bool retval = false;

  if ((fd = POV_NEW_OSTREAM(MechsimOptions.Save_File_Name, POV_File_Unknown, false)) == NULL)
  {
    POV_DELETE(fd, POV_OSTREAM);
    Error("Cannot save mechanics simulation data to file %s", MechsimOptions.Save_File_Name );
    return false;
  }

  if(fd != NULL)
  {
    fd->printf("MSIM, ");

    fd->printf("%d,\n", MechsimOptions.Save_File_Type);

    if (MechsimOptions.Save_File_Type>2)  // file subtype 3 -> save start time
      fd->printf("%g,\n", MechsimOptions.Start_Time + (MechsimOptions.Time_Step*MechsimOptions.Step_Count));

    fd->printf("%d,\n", MechsimOptions.Data.mass_count);
    fd->printf("%d,\n", MechsimOptions.Data.connect_count);
    fd->printf("%d,\n", MechsimOptions.Data.faces_count);

    for (i=0; i<MechsimOptions.Data.mass_count; i++)
    {
      #if(MECHSIM_DEBUG == 1)
        if (i==0)
        {
          Debug_Info("\n<%g, %g, %g>, ",
                     MechsimOptions.Data.Masses[i].Position[X],
                     MechsimOptions.Data.Masses[i].Position[Y],
                     MechsimOptions.Data.Masses[i].Position[Z]);
          Debug_Info("<%g, %g, %g>\n",
                     MechsimOptions.Data.Masses[i].Velocity[X],
                     MechsimOptions.Data.Masses[i].Velocity[Y],
                     MechsimOptions.Data.Masses[i].Velocity[Z]);
        }
      #endif

      fd->printf("<%.12f, %.12f, %.12f>, ",
                 MechsimOptions.Data.Masses[i].Position[X],
                 MechsimOptions.Data.Masses[i].Position[Y],
                 MechsimOptions.Data.Masses[i].Position[Z]);
      fd->printf("<%.12f, %.12f, %.12f>, ",
                 MechsimOptions.Data.Masses[i].Velocity[X],
                 MechsimOptions.Data.Masses[i].Velocity[Y],
                 MechsimOptions.Data.Masses[i].Velocity[Z]);
      fd->printf("%g, %g, %d, %d,",
                 MechsimOptions.Data.Masses[i].Mass,
                 MechsimOptions.Data.Masses[i].Radius,
                 MechsimOptions.Data.Masses[i].Flag,
                 MechsimOptions.Data.Masses[i].Group);

      if (MechsimOptions.Save_File_Type>2)  // file subtype 3 -> save attachment index
        fd->printf(" %d,\n",MechsimOptions.Data.Masses[i].Attach);
      else fd->printf("\n");
    }

    for (i=0; i<MechsimOptions.Data.connect_count; i++)
    {
      fd->printf("%d, %d, %g, %g, %g, %d,\n",
                 MechsimOptions.Data.Connects[i].Idx1,
                 MechsimOptions.Data.Connects[i].Idx2,
                 MechsimOptions.Data.Connects[i].Length,
                 MechsimOptions.Data.Connects[i].Stiffness,
                 MechsimOptions.Data.Connects[i].Damping,
                 MechsimOptions.Data.Connects[i].Group);
    }

    for (i=0; i<MechsimOptions.Data.faces_count; i++)
    {
      fd->printf("%d, %d, %d, %d,\n",
                 MechsimOptions.Data.Faces[i].Idx1,
                 MechsimOptions.Data.Faces[i].Idx2,
                 MechsimOptions.Data.Faces[i].Idx3,
                 MechsimOptions.Data.Faces[i].Group);
    }

    fd->printf("\n");

    POV_DELETE(fd, POV_OSTREAM);

    Render_Info("\nSaved simulation data to file %s\n", MechsimOptions.Save_File_Name);

  }
  else retval = true;

  POV_DELETE(fd, POV_OSTREAM);

  POV_FREE(MechsimOptions.Save_File_Name);
  MechsimOptions.Save_File_Name = NULL;

  return retval;
}



// ========================================================================

char *Read_Vector(char *buffer, VECTOR Vect)
{
  char *Pos;
  char *End;
  char *ptr;
  char str[MAX_STR];
  char str_nbr[MAX_STR];
  VECTOR result;

  strcpy(str, buffer);

  Pos=strchr(str, '<');                 // search vector start
  if (Pos != NULL)
  {
    Pos++;
    Pos = strtok(Pos, ", ");
    if (Pos != NULL)
    {
      strcpy(str_nbr, Pos);
      result[X] = strtod(str_nbr, &End);
      Pos = strtok(NULL, ", ");
      if (Pos != NULL)
      {
        strcpy(str_nbr, Pos);
        result[Y] = strtod(str_nbr, &End);
        Pos = strtok(NULL, ", ");
        if (Pos != NULL)
        {
          strcpy(str_nbr, Pos);
          result[Z] = strtod(str_nbr, &End);
        }
      }
    }
    else
    {
      Status_Info("\nWarning: strange format of mechanics simulation data (v1)");
      Vect[X] = 0.0;
      Vect[Y] = 0.0;
      Vect[Z] = 0.0;
      return buffer;
    }

    Assign_Vector(Vect, result);

  }
  else
  {
    Status_Info("\nWarning: strange format of mechanics simulation data (v2)");
    Vect[X] = 0.0;
    Vect[Y] = 0.0;
    Vect[Z] = 0.0;
    return buffer;
  }

  Pos = strchr(buffer, '>');               // search vector end
  Pos++;
  ptr = strchr(Pos, ',');

  if (ptr == NULL)                         // no comma
    return Pos;

  ptr++;

  return ptr;

}

// ========================================================================

char *Read_Float(char *buffer, double *Value)
{
  char *Pos;
  char *End;
  char str[MAX_STR];
  char str_nbr[MAX_STR];
  double result;

  strcpy(str, buffer);

  //Debug_Info("fl(%s)\n", str);

  Pos=strchr(str, ',');
  if (Pos == NULL) Pos=strchr(str, ' ');
  if (Pos != NULL) Pos[0] = '\0';            // cut comma

  strcpy(str_nbr, str);
  result = strtod(str_nbr, &End);

  *Value = result;

  Pos=strchr(buffer, ',');
  if (Pos == NULL) Pos=strchr(buffer, ' ');
  if (Pos == NULL)
  {
    Status_Info("\nWarning: strange format of mechanics simulation data (f)");
    return buffer;
  }

  Pos++;

  return Pos;

}

// ========================================================================

char *Read_Int(char *buffer, int *Value)
{
  char *Pos;
  char *End;
  char str[MAX_STR];
  char str_nbr[MAX_STR];
  int result;

  strcpy(str, buffer);

  //Debug_Info("int(%s)\n", str);

  Pos=strchr(str, ',');
  if (Pos == NULL) Pos=strchr(str, ' ');
  if (Pos != NULL) Pos[0] = '\0';            // cut comma

  strcpy(str_nbr, str);
  result = strtol(str_nbr, &End, 0);

  *Value = result;

  Pos=strchr(buffer, ',');
  if (Pos == NULL) Pos=strchr(buffer, ' ');
  if (Pos == NULL)
  {
    Status_Info("\nWarning: strange format of mechanics simulation data (i)");
    return buffer;
  }

  Pos++;

  return Pos;

}

/*****************************************************************************
*
* FUNCTION
*
*   calc_faces_bounding_box
*
* INPUT
*   Pos_Array array of masses positions.
*   Faces_BBox_Array array of faces bounding boxes.
*
* OUTPUT
*
* RETURNS Unit size.
*
* AUTHOUR
*
*   Daniel Jungmann
*
* DESCRIPTION
*
*   Calculates the bounding boxes for all faces and the unit size of the grid for faces.
*
* CHANGES
*
*   --- Mar 2004 : Creation
*
******************************************************************************/
static DBL calc_faces_bounding_box(VECTOR* Pos_Array, MECHSIM_BBOX* Faces_BBox_Array)
{
  DBL Unit;
  VECTOR t_max, t_min, tmp;
  int Idx1, Idx2, Idx3, i;
  
  Make_Vector(tmp, 0.0, 0.0, 0.0);
  for (i = 0; i < MechsimOptions.Data.faces_count; i++)
  {
    Idx1 = MechsimOptions.Data.Faces[i].Idx1;
    Idx2 = MechsimOptions.Data.Faces[i].Idx2;
    Idx3 = MechsimOptions.Data.Faces[i].Idx3;
    t_max[X] = max(Pos_Array[Idx1][X], max(Pos_Array[Idx2][X], Pos_Array[Idx3][X]));
    t_max[Y] = max(Pos_Array[Idx1][Y], max(Pos_Array[Idx2][Y], Pos_Array[Idx3][Y]));
    t_max[Z] = max(Pos_Array[Idx1][Z], max(Pos_Array[Idx2][Z], Pos_Array[Idx3][Z]));
    t_min[X] = min(Pos_Array[Idx1][X], min(Pos_Array[Idx2][X], Pos_Array[Idx3][X]));
    t_min[Y] = min(Pos_Array[Idx1][Y], min(Pos_Array[Idx2][Y], Pos_Array[Idx3][Y]));
    t_min[Z] = min(Pos_Array[Idx1][Z], min(Pos_Array[Idx2][Z], Pos_Array[Idx3][Z]));

    VAssign(Faces_BBox_Array[i].bbox_max, t_max);
    VAssign(Faces_BBox_Array[i].bbox_min, t_min);
    
    VSubEq(t_max, t_min);
    VAddEq(tmp, t_max);
  }
  Unit = tmp[X] + tmp[Y] + tmp[Z];
  Unit /= 3*MechsimOptions.Data.faces_count;
  return Unit;
}

/*****************************************************************************
*
* FUNCTION
*
*   intersect_sphere_box
*
* INPUT
*   Box_Min mins of box.
*   Box_Max maxs of box.
*   Center center of sphere.
*   Radius radius of sphere.
*
* OUTPUT
*
* RETURNS true if sphere intersects box else false.
*
* AUTHOUR
*
*   Daniel Jungmann
*
* DESCRIPTION
*
*   Test if a sphere intersects a box.
*
* CHANGES
*
*   --- Mar 2004 : Creation
*
******************************************************************************/
static bool intersect_sphere_box(VECTOR Box_Min, VECTOR Box_Max, VECTOR Center, DBL Radius)
{
  DBL s, d;

  d = 0;
  
  if(Center[X] < Box_Min[X])
  {
    s = Center[X] - Box_Min[X];
    d += s*s;
  }
  else if(Center[X] > Box_Max[X])
  {
    s = Center[X] - Box_Max[X];
    d += s*s;
  }
  if(Center[Y] < Box_Min[Y])
  {
    s = Center[Y] - Box_Min[Y];
    d += s*s;
  }
  else if(Center[Y] > Box_Max[Y])
  {
    s = Center[Y] - Box_Max[Y];
    d += s*s;
  }
  if(Center[Z] < Box_Min[Z])
  {
    s = Center[Z] - Box_Min[Z];
    d += s*s;
  }
  else if(Center[Z] > Box_Max[Z])
  {
    s = Center[Z] - Box_Max[Z];
    d += s*s;
  }
  
  return d <= Radius*Radius;
}

/*****************************************************************************
*
* FUNCTION
*
*   calc_faces_bounding_hashtable
*
* INPUT
*   bounding_hashtable old faces bounding hashtable.
*   Unit unit of the grid.
*   bounding_hashtable_size size of the bounding hashtable.
*   time_stamp time stamp for faster updating of the hashtable.
*
* OUTPUT
*   bounding_hashtable updated faces bounding hashtable.
*
* RETURNS 
*
* AUTHOUR
*
*   Daniel Jungmann
*
* DESCRIPTION
*
*   Calculates the bounding hashtable for all faces.
*
* CHANGES
*
*   --- Mar 2004 : Creation
*
******************************************************************************/
static void calc_faces_bounding_hashtable(BOUNDING_HASHTABLE_ITEM* bounding_hashtable, DBL Unit, int bounding_hashtable_size, int time_stamp, 
  MECHSIM_BBOX* Faces_BBox_Array)
{
  long int jx_min, jy_min, jz_min;
  long int jx_max, jy_max, jz_max;
  long int jx, jy, jz;
  int i;
  unsigned hash;  
  
  for (i = 0; i < MechsimOptions.Data.faces_count; i++)
  {
    jx_min = (long int)floor(Faces_BBox_Array[i].bbox_min[X]/Unit);
    jy_min = (long int)floor(Faces_BBox_Array[i].bbox_min[Y]/Unit);
    jz_min = (long int)floor(Faces_BBox_Array[i].bbox_min[Z]/Unit);

    jx_max = (long int)floor(Faces_BBox_Array[i].bbox_max[X]/Unit);
    jy_max = (long int)floor(Faces_BBox_Array[i].bbox_max[Y]/Unit);
    jz_max = (long int)floor(Faces_BBox_Array[i].bbox_max[Z]/Unit);

    for (jx = jx_min; jx <= jx_max; jx++)
      for (jy = jy_min; jy <= jy_max; jy++)
        for (jz = jz_min; jz <= jz_max; jz++)
        {
	  hash = bounding_hash_func(jx, jy, jz, bounding_hashtable_size);
	  insert_index_bounding_hastable(bounding_hashtable, hash, i, time_stamp);
        }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   insert_index_bounding_hastable
*
* INPUT
*   bounding_hashtable bounding hashtable.
*   hash hash of cell.
*   index index of triangle.
*   time_stamp time stamp for faster updating of the hashtable.
*
* OUTPUT
*
* RETURNS 
*
* AUTHOUR
*
*   Daniel Jungmann
*
* DESCRIPTION
*
*   Insert an index into the bounding hashtable at position hash. 
*
* CHANGES
*
*   --- Mar 2004 : Creation
*
******************************************************************************/
static void insert_index_bounding_hastable(BOUNDING_HASHTABLE_ITEM* bounding_hashtable, unsigned int hash, int index, int time_stamp)
{
  int size, used;
  if ((bounding_hashtable[hash].size == 0) || (bounding_hashtable[hash].index == NULL))
  {
    size = 4;
    used = 0;
    bounding_hashtable[hash].index = (int *)POV_MALLOC(size*sizeof(int), "mechsim bounding hashtable index's");
    bounding_hashtable[hash].size = size;
    bounding_hashtable[hash].used = used;
  }
  else
  {
    size = bounding_hashtable[hash].size;
    if (bounding_hashtable[hash].time_stamp < time_stamp) used = 0;
    else used = bounding_hashtable[hash].used;
  }
  if (size < (used+1))
  {
    size *= 2;
    bounding_hashtable[hash].index = (int *)POV_REALLOC(bounding_hashtable[hash].index, size*sizeof(int), 
      "mechsim bounding hashtable index's");
    bounding_hashtable[hash].size = size;
  }  
  bounding_hashtable[hash].index[used] = index;
  bounding_hashtable[hash].time_stamp = time_stamp;
  bounding_hashtable[hash].used = used+1;
}

/*****************************************************************************
*
* FUNCTION
*
*   intersect_sphere_bounding_hashtable
*
* INPUT
*   Center center of sphere.
*   Radius radius of sphere.
*   Unit unit of the bounding hashtable grid.
*   list list of intersections(sortet).
*   size size of list.
*   time_stamp time stamp for faster updating of the hashtable.
*
* OUTPUT
*
* RETURNS 
*
* AUTHOUR
*
*   Daniel Jungmann
*
* DESCRIPTION
*
*   Test if a sphere intersects a bounding hashtable.
*
* CHANGES
*
*   --- Mar 2004 : Creation
*
******************************************************************************/
static void intersect_sphere_bounding_hashtable(BOUNDING_HASHTABLE_ITEM* bounding_hashtable, VECTOR Center, DBL Radius, 
  DBL Unit, int** list, int* size, int* index, int bounding_hashtable_size, int time_stamp)
{
  VECTOR Box_Min, Box_Max;
  long int jx_min, jy_min, jz_min;
  long int jx_max, jy_max, jz_max;
  long int jx, jy, jz;
  unsigned int hash;
  int i, j;
  int idx, *t_list, t_size;
  
  jx_min = (long int)floor((Center[X]-Radius)/Unit);
  jy_min = (long int)floor((Center[Y]-Radius)/Unit);
  jz_min = (long int)floor((Center[Z]-Radius)/Unit);

  jx_max = (long int)floor((Center[X]+Radius)/Unit);
  jy_max = (long int)floor((Center[Y]+Radius)/Unit);
  jz_max = (long int)floor((Center[Z]+Radius)/Unit);
  
  idx = 0;
  t_list = *list;
  t_size = *size;
  Box_Min[X] = jx*Unit;
  Box_Max[X] = Box_Min[X]+Unit;
  for (jx = jx_min; jx <= jx_max; jx++)
  {
    Box_Min[Y] = jy*Unit;
    Box_Max[Y] = Box_Min[Y]+Unit;
    for (jy = jy_min; jy <= jy_max; jy++)
    {
      Box_Min[Z] = jz*Unit;
      Box_Max[Z] = Box_Min[Z]+Unit;
      for (jz = jz_min; jz <= jz_max; jz++)
      {
	if (intersect_sphere_box(Box_Min, Box_Max, Center, Radius))
	{
	  hash = bounding_hash_func(jx, jy, jz, bounding_hashtable_size);
	  if (bounding_hashtable[hash].time_stamp == time_stamp)
	  {
	    if (bounding_hashtable[hash].used > 0)
	    {
	      if (t_size < (idx+bounding_hashtable[hash].used))
	      {
	        t_size += t_size+bounding_hashtable[hash].used;
	        t_list = (int *)POV_REALLOC(t_list, t_size*sizeof(int), "mechsim bounding hashtable intersect list");
	      }	  
	      memcpy(&t_list[idx], bounding_hashtable[hash].index, bounding_hashtable[hash].used*sizeof(int));
	      idx += bounding_hashtable[hash].used;
	    }
	  }
        Box_Min[Z] += Unit;
        Box_Max[Z] += Unit;
        }
      Box_Min[Y] += Unit;
      Box_Max[Y] += Unit;
      }
    Box_Min[X] += Unit;
    Box_Max[X] += Unit;
    }
  }

  /* We need every index just once. So first we sort them. */
  QSORT((void *)t_list, idx, sizeof(int), compare_int);
  
  i = 0;
  j = 0;
  
  while (j < idx)
  {
    t_list[i] = t_list[j];
    j++;
    while ((j < idx) && (t_list[i] == t_list[j])) j++;
    i++;
  }
  
  idx = i;
  *index = idx;
  *size = t_size;
  *list = t_list;
}

/*****************************************************************************
*
* FUNCTION
*
*   compare_int
*
* INPUT
*
*   in_a, in_b - elements to compare
*
* OUTPUT
*
* RETURNS
*
*   int - result of comparison
*
* AUTHOR
*
*   Daniel Jungmann
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   --- Mar 2004 : Creation
*
******************************************************************************/

static int CDECL compare_int(const void *in_a, const void *in_b)
{
  int a, b;
  
  a = *(int *)in_a;
  b = *(int *)in_b;
  
  if (a < b) return (-1);
  else
  {
    if (a > b) return (1);
    else return (0);
  }
}

// ========================================================================

static unsigned int bounding_hash_func(long int x, long int y, long int z, long int n)
{
  long int r;
  unsigned int ret;
  x *= 73856093L;
  y *= 19349663L;
  z *= 83492791L;
  r = x ^ y ^ z;
  r %= n;
  ret = abs(r);
  return ret;
}

#endif

