/*
Copyright 1999-2001 Thorsten Froehlich. All rights reserved. 

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met: 

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer. 
    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution. 

THIS SOFTWARE IS PROVIDED BY THORSTEN FROEHLICH "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THORSTEN FROEHLICH BE LIABLE FOR ANY DIRECT, INDIRECT, 
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE. 
*/

#include <types.h>

#include "qpovparse.h"

// valid first characters in declare and macro names
bool gIsFirstWordChar[256] =
{
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,

/*    ! " # $ % & '   */
	0,0,0,1,0,0,0,0,
/*  ( ) * + , - .     */
	0,0,0,0,0,0,0,0,
/*  0 1 2 3 4 5 6 7   */
	0,0,0,0,0,0,0,0,
/*  8 9 : ; < = > ?   */
	0,0,0,0,0,0,0,0,
/*  @ A B C D E F G   */
	0,1,1,1,1,1,1,1,
/*  H I J K L M N O   */
	1,1,1,1,1,1,1,1,
/*  P Q R S T U V W   */
	1,1,1,1,1,1,1,1,
/*  X Y Z [   ] ^ _   */
	1,1,1,0,0,0,0,1,
/*    a b c d e f g   */
	0,1,1,1,1,1,1,1,
/*  h i j k l m n o   */
	1,1,1,1,1,1,1,1,
/*  p q r s t u v w   */
	1,1,1,1,1,1,1,1,
/*  x y z { | } ~     */
	1,1,1,0,0,0,0,0,

	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0
};

// valid characters in declare and macro names
bool gIsWordChar[256] =
{
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,

/*    ! " # $ % & '   */
	0,0,0,0,0,0,0,0,
/*  ( ) * + , - .     */
	0,0,0,0,0,0,0,0,
/*  0 1 2 3 4 5 6 7   */
	1,1,1,1,1,1,1,1,
/*  8 9 : ; < = > ?   */
	1,1,0,0,0,0,0,0,
/*  @ A B C D E F G   */
	0,1,1,1,1,1,1,1,
/*  H I J K L M N O   */
	1,1,1,1,1,1,1,1,
/*  P Q R S T U V W   */
	1,1,1,1,1,1,1,1,
/*  X Y Z [   ] ^ _   */
	1,1,1,0,0,0,0,1,
/*    a b c d e f g   */
	0,1,1,1,1,1,1,1,
/*  h i j k l m n o   */
	1,1,1,1,1,1,1,1,
/*  p q r s t u v w   */
	1,1,1,1,1,1,1,1,
/*  x y z { | } ~     */
	1,1,1,0,0,0,0,0,

	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0
};

// Constructor
BuildSymbolTable::BuildSymbolTable()
{
	// do nothing in here right now
}

// Destructor
BuildSymbolTable::~BuildSymbolTable()
{
	// do nothing in here right now
}

// Parse the text stream for POV language directives
bool BuildSymbolTable::Parse(ArrayStream& as, int maxlevel)
{
	int i,first;

	// watch maximum recursion level
	if(maxlevel <= 0)
		return false;

	// parse the whole stream
	for(i = 0; i < as.size(); i++)
	{
		// directives start with a "#"
		if(as[i] == '#')
		{
			// "declare" directives
			if(as[i+1] == 'd')
			{
				if((as[i+2] == 'e') && (as[i+3] == 'c') &&
				   (as[i+4] == 'l') && (as[i+5] == 'a') &&
				   (as[i+6] == 'r') && (as[i+7] == 'e'))
				{
					// skip until name
					for(i += 8; (i < as.size()) && (IsFirstWordChar(as[i]) == false); i++)
						;

					// find the end of the name
					for(first = i; (i < as.size()) && (IsWordChar(as[i]) == true); i++)
						;

					// if a valid name was found, add it
					if((i < as.size()) && (first < as.size()))
						AddDeclare(as.access(first, i - first), i - first);
				}			
			}
			// "macro" directives
			else if(as[i+1] == 'm')
			{
				if((as[i+2] == 'a') && (as[i+3] == 'c') &&
				   (as[i+4] == 'r') && (as[i+5] == 'o'))
				{
					// skip until name
					for(i += 8; (i < as.size()) && (IsFirstWordChar(as[i]) == false); i++)
						;

					// find the end of the name
					for(first = i; (i < as.size()) && (IsWordChar(as[i]) == true); i++)
						;

					// if a valid name was found, add it
					if((i < as.size()) && (first < as.size()))
						AddMacro(as.access(first, i - first), i - first);	
				}
			}
			// "include" directives
			else if(as[i+1] == 'i')
			{
				if((as[i+2] == 'n') && (as[i+3] == 'c') &&
				   (as[i+4] == 'l') && (as[i+5] == 'u') &&
				   (as[i+6] == 'd') && (as[i+7] == 'e'))
				{
					// skip until quotation mark
					for(i += 8; (i < as.size()) && (as[i] != '"') && (as[i] != 10) && (as[i] != 13); i++)
						;
					i++;
					// find the end of the name (which the next quotation mark)
					for(first = i; (i < as.size()) && (as[i] != '"') && (as[i] != 10) && (as[i] != 13); i++)
						;

					// if a valid name was found, use it
					if(((i - first) > 0) && ((i - first) < 256))
					{
						try
						{
							// locate the include file and get a stream for it
							ArrayStream *substream = OpenIncludeFile(as.access(first, i - first), i - first);

							// if a stream was returned, parse that file (recursive!)
							if(substream != NULL)
							{
								(void)Parse(*substream, maxlevel - 1); // remember to decrease maxlevel
								delete substream;
							}
						}
						catch(...)
						{
						}
					}
				}
			}
		}
		// skip comments, currentlz doesn"t work for recursive comments
		else if((as[i] == '/') && (i < (as.size() - 1)))
		{
			// single line comment
			if(as[i + 1] == '/')
			{
				for(; (i < as.size()) && (as[i] != 10) && (as[i] != 13); i++) // CR or LF is good enough here
					;
			}
			// multi-line comment - recursive!
			else if(as[i + 1] == '*')
			{
				i += 2;
				(void)ParseComment(as, i, maxlevel + 20); // remember to set useful maxlevel
			}
		}
	}

	return true;
}

// Checks if a given character is valid as the first character of a word
bool BuildSymbolTable::IsFirstWordChar(unsigned char c)
{
	return gIsFirstWordChar[c];
}

// Checks if a given character is valid as a (not first!) character of a word
bool BuildSymbolTable::IsWordChar(unsigned char c)
{
	return gIsWordChar[c];
}

// Recursive multi-line comment parsing
bool BuildSymbolTable::ParseComment(ArrayStream& as, int& i, int maxlevel)
{
	if(maxlevel <= 0)
		return false;

	// watch maximum recursion level
	for(; i < as.size() - 1; i++)
	{
		// close comment
		if(as[i] == '*')
		{
			if(as[i + 1] == '/')
				break;
		}
		// open another comment
		else if(as[i] == '/')
		{
			if(as[i + 1] == '*')
			{
				i += 2;
				if(ParseComment(as, i, maxlevel - 1) == false) // remember to decrease maxlevel
					return false;
			}
		}
	}

	return true;
}
