/*
Copyright 1999-2001 Thorsten Froehlich. All rights reserved. 

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met: 

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer. 
    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution. 

THIS SOFTWARE IS PROVIDED BY THORSTEN FROEHLICH "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THORSTEN FROEHLICH BE LIABLE FOR ANY DIRECT, INDIRECT, 
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE. 
*/

#include <types.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>

using namespace std;

#include "qpovparse.h"

class StdioBuildSymbolTable : public BuildSymbolTable
{
	public:
		StdioBuildSymbolTable();
		~StdioBuildSymbolTable();

		bool Parse(ArrayStream& as);
	protected:
		virtual void AddDeclare(const char *text, long textlen);
		virtual void AddMacro(const char *text, long textlen);
		virtual ArrayStream *OpenIncludeFile(const char *name, long namelen);
};


class StdioFileStream : public ArrayStream
{
	public:
		StdioFileStream(const char *filename)
		{
			long cnt = 0;

			totalBytes = 0;
			fileoffset = 0;

			fileReference = fopen(filename, "r");
			if(fileReference == NULL)
				throw -1;
			fseek(fileReference, 0, SEEK_END);
			totalBytes = ftell(fileReference);
			fseek(fileReference, 0, SEEK_SET);

			cnt = 256;
			cnt = fread(buffer, 1, cnt, fileReference);
		};
		~StdioFileStream()
		{
			fclose(fileReference);
		};

		char operator[](int i)
		{
			if(i >= totalBytes)
				return 0;

			if(i < fileoffset)
			{
				read(i);
				return buffer[0];
			}
			else if(i >= (fileoffset + 256))
			{
				read(i);
				return buffer[0];
			}

			return buffer[i - fileoffset];
		};
		virtual char *access(int i, int len)
		{
			if((i >= fileoffset) && (i < (fileoffset + 256 - len)))
				return &(buffer[i - fileoffset]);

			read(i);
			return &(buffer[0]);
		};
		inline int size() { return totalBytes; };
	private:
		long totalBytes;
		int fileoffset;
		FILE *fileReference;
		char buffer[256];

		void read(int i)
		{
			long cnt = 0;

			if(i > totalBytes)
				throw -1;

			fileoffset = i;
			fseek(fileReference, fileoffset, SEEK_SET);
			if((totalBytes - fileoffset) >= 256)
				cnt = 256;
			else
				cnt = (totalBytes - fileoffset);
			cnt = fread(buffer, 1, cnt, fileReference);

			if(cnt <= 0)
				throw -1;
		};
};


int main()
{	
	char str[256];

	puts("               Quick Parser");
	puts("==========================================");
	puts("");

	printf("Enter filename: ");
	gets(str);

	puts("");
	puts("Parsing...");

	StdioBuildSymbolTable t;

	t.Parse(StdioFileStream(str));

	puts("Done!");

	return 0;
}


StdioBuildSymbolTable::StdioBuildSymbolTable()
{
}

StdioBuildSymbolTable::~StdioBuildSymbolTable()
{
}

bool StdioBuildSymbolTable::Parse(ArrayStream& as)
{
	try
	{
		return BuildSymbolTable::Parse(as, 10);
	}
	catch(...)
	{
	}

	return false;
}

void StdioBuildSymbolTable::AddDeclare(const char *text, long textlen)
{
	int i;

	printf("#declare ");
	for(i = 0; i < textlen; i++)
		putc(text[i], stdout);
	puts("");
}

void StdioBuildSymbolTable::AddMacro(const char *text, long textlen)
{
	int i;

	printf("#macro ");
	for(i = 0; i < textlen; i++)
		putc(text[i], stdout);
	puts("");
}

ArrayStream *StdioBuildSymbolTable::OpenIncludeFile(const char *name, long namelen)
{
	char filename[256];

	memcpy((void *)filename, (void *)name, namelen);
	filename[namelen] = 0;

	printf("#include \"%s\"\n", filename);

	return new StdioFileStream(filename);
}
