/****************************************************************************
*                   colour.c
*
*  This module implements routines to manipulate colours.
*
*  from Persistence of Vision(tm) Ray Tracer
*  Copyright 1996,1999 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other 
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file.
*  If POVLEGAL.DOC is not available or for more info please contact the POV-Ray
*  Team Coordinator by email to team-coord@povray.org or visit us on the web at
*  http://www.povray.org. The latest version of POV-Ray may be found at this site.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
*****************************************************************************/

#include "frame.h"
#include "vector.h"
#include "povproto.h"
#include "colour.h"
#include "pigment.h"
#include "normal.h"
#include "texture.h"


/*****************************************************************************
* Local preprocessor defines
******************************************************************************/



/*****************************************************************************
* Local typedefs
******************************************************************************/



/*****************************************************************************
* Local variables
******************************************************************************/


/*****************************************************************************
* Static functions
******************************************************************************/



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

COLOUR *Create_Colour ()
{
  COLOUR *New;

  New = (COLOUR *)POV_MALLOC(sizeof (COLOUR), "color");

  Make_ColourA (*New, 0.0, 0.0, 0.0, 0.0, 0.0);

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

COLOUR *Copy_Colour (COLOUR Old)
{
  COLOUR *New;

  if (Old != NULL)
  {
    New = Create_Colour ();

    Assign_Colour(*New,Old);
  }
  else
  {
    New = NULL;
  }

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Aug 1995 : Use POV_CALLOC to initialize entries. [DB]
*
******************************************************************************/

BLEND_MAP_ENTRY *Create_BMap_Entries (int Map_Size)
{
  BLEND_MAP_ENTRY *New;

  New = (BLEND_MAP_ENTRY *)POV_CALLOC((size_t)Map_Size, sizeof (BLEND_MAP_ENTRY), "blend map entry");

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*
* CHANGES
*
******************************************************************************/

BLEND_MAP_ENTRY *Copy_BMap_Entries (BLEND_MAP_ENTRY *Old, int Map_Size, int  Type)
{
  int i;
  BLEND_MAP_ENTRY *New;

  if (Old != NULL)
  {
    New = Create_BMap_Entries (Map_Size);

    for (i = 0; i < Map_Size; i++)
    {
      switch (Type)
      {
        case PIGMENT_TYPE:

          New[i].Vals.Pigment = Copy_Pigment(Old[i].Vals.Pigment);

          break;

        case NORMAL_TYPE:

          New[i].Vals.Tnormal = Copy_Tnormal(Old[i].Vals.Tnormal);

          break;

        case TEXTURE_TYPE:

          New[i].Vals.Texture = Copy_Textures(Old[i].Vals.Texture);

          break;

        case COLOUR_TYPE:
        case SLOPE_TYPE:

          New[i] = Old[i];

          break;
      }
    }
  }
  else
  {
    New = NULL;
  }

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_Blend_Map
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

BLEND_MAP *Create_Blend_Map ()
{
  BLEND_MAP *New;

  New = (BLEND_MAP *)POV_MALLOC(sizeof (BLEND_MAP), "blend map");

  New->Users = 1;

  New->Number_Of_Entries = 0;

  New->Type = COLOUR_TYPE;

  New->Blend_Map_Entries = NULL;

  New->Transparency_Flag = FALSE;

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Copy_Blend_Map
*
* INPUT
*   
* OUTPUT
*
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

BLEND_MAP *Copy_Blend_Map (BLEND_MAP *Old)
{
  BLEND_MAP *New;

  New = Old;

  /* 
   * Do not increase the users field if it is negative.
   *
   * A negative users field incicates a reference to a static
   * or global memory area in the data segment, not on the heap!
   * Thus it must not be deleted later.
   */

  if ((New != NULL) && (New->Users >= 0))
  {
    New->Users++;
  }

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Colour_Distance
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

DBL Colour_Distance (COLOUR colour1, COLOUR  colour2)
{
  return (fabs(colour1[RED]   - colour2[RED]) +
          fabs(colour1[GREEN] - colour2[GREEN]) +
          fabs(colour1[BLUE]  - colour2[BLUE]));
}



/*****************************************************************************
*
* FUNCTION
*
*   Add_Colour
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Add_Colour (COLOUR result, COLOUR  colour1, COLOUR  colour2)
{
  result[RED]    = colour1[RED]    + colour2[RED];
  result[GREEN]  = colour1[GREEN]  + colour2[GREEN];
  result[BLUE]   = colour1[BLUE]   + colour2[BLUE];
  result[FILTER] = colour1[FILTER] + colour2[FILTER];
  result[TRANSM] = colour1[TRANSM] + colour2[TRANSM];
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_Colour
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Scale_Colour (COLOUR result, COLOUR  colour, DBL factor)
{
  result[RED]    = colour[RED]    * factor;
  result[GREEN]  = colour[GREEN]  * factor;
  result[BLUE]   = colour[BLUE]   * factor;
  result[FILTER] = colour[FILTER] * factor;
  result[TRANSM] = colour[TRANSM] * factor;
}



/*****************************************************************************
*
* FUNCTION
*
*   Clip_Colour
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Clip_Colour (COLOUR result, COLOUR  colour)
{
  if (colour[RED] > 1.0)
  {
    result[RED] = 1.0;
  }
  else
  {
    if (colour[RED] < 0.0)
    {
      result[RED] = 0.0;
    }
    else
    {
      result[RED] = colour[RED];
    }
  }

  if (colour[GREEN] > 1.0)
  {
    result[GREEN] = 1.0;
  }
  else
  {
    if (colour[GREEN] < 0.0)
    {
      result[GREEN] = 0.0;
    }
    else
    {
      result[GREEN] = colour[GREEN];
    }
  }

  if (colour[BLUE] > 1.0)
  {
    result[BLUE] = 1.0;
  }
  else
  {
    if (colour[BLUE] < 0.0)
    {
      result[BLUE] = 0.0;
    }
    else
    {
      result[BLUE] = colour[BLUE];
    }
  }

  if (colour[FILTER] > 1.0)
  {
    result[FILTER] = 1.0;
  }
  else
  {
    if (colour[FILTER] < 0.0)
    {
      result[FILTER] = 0.0;
    }
    else
    {
      result[FILTER] = colour[FILTER];
    }
  }

  if (colour[TRANSM] > 1.0)
  {
    result[TRANSM] = 1.0;
  }
  else
  {
    if (colour[TRANSM] < 0.0)
    {
      result[TRANSM] = 0.0;
    }
    else
    {
      result[TRANSM] = colour[TRANSM];
    }
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   Destroy_Blend_Map
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Destroy_Blend_Map (BLEND_MAP *BMap)
{
  int i;
  
  if (BMap != NULL)
  {
    if (--(BMap->Users) == 0)
    {
      for (i = 0; i < BMap->Number_Of_Entries; i++)
      {
        switch (BMap->Type)
        {
           case PIGMENT_TYPE:
           case DENSITY_TYPE:
             Destroy_Pigment(BMap->Blend_Map_Entries[i].Vals.Pigment);
             break;

           case NORMAL_TYPE:
             Destroy_Tnormal(BMap->Blend_Map_Entries[i].Vals.Tnormal);
             break;

           case TEXTURE_TYPE:
             Destroy_Textures(BMap->Blend_Map_Entries[i].Vals.Texture);
        }
      }

      POV_FREE (BMap->Blend_Map_Entries);

      POV_FREE (BMap);
    }
  }
}

/*****************************************************************************
*
* FUNCTION
*
*   RGBtoHue
*
* INPUT
*        c - Colour to convert to hue
*   
* OUTPUT
*
* RETURNS
*        hue value of given colour (range 0-1)
*   
* AUTHOR
*        NK
*   
* DESCRIPTION
*        this code from Mike's MSGTracer
*
* CHANGES
*
*   VK 01.07.2000 : Changed return value to standard hue (was specter shifted
*                   value). For specter value function RGBtoSpecter was created.
*
******************************************************************************/
DBL RGBtoHue( COLOUR c )
{
  DBL r, g, b;
  DBL mx, mn, delta;
  DBL h, s, v;
  DBL w;

  r = c[0];
  g = c[1];
  b = c[2];

  mx = max3(r,g,b);
  mn = min3(r,g,b);

  h = 0.0;

  delta = mx-mn;
  if( delta > 0.0 && mx > 0.0 )
   {
    if( r == mx )
      h = (g-b)/delta;
    else if( g == mx )
      h = 2.0 + (b-r)/delta;
    else if( b == mx )
      h = 4.0 + (r-g)/delta;
   }

  h /= 6.0;
  if( h < 0.0 ) h += 1.0;
  return h;
}


/*****************************************************************************
*
* FUNCTION
*
*   RGBtoSpecter
*
* INPUT
*        c - Colour to convert to specter value.
*   
* OUTPUT
*
* RETURNS
*        specter value for given colour.
*   
* AUTHOR
*        VK
*   
* DESCRIPTION
*
* CHANGES
*
*   VK 01.07.2000 : Created. Specter shift was taken from RGBtoHue.
*
******************************************************************************/
DBL RGBtoSpecter(COLOUR c)
{

  double w = RGBtoHue(c);

  w = w + 60.0/360.0;                         /* Split ultraviolet/red at -60 */
  if( w > 1.0 ) w -= 1.0;           

  return w;
}


/*****************************************************************************
*
* FUNCTION
*
*   CValue
*
* INPUT
*        n1 - aux value.
*        n2 - aux value.
*        hue- hue value (between 0 and 1).
*   
* OUTPUT
*
* RETURNS
*        colour value.
*   
* AUTHOR
*        VK
*   
* DESCRIPTION
*        Used in HSL=>RGB conversion.
*        Code is taken from "Computer Graphics: Principles and Practice",
*        2nd ed. in C, pg. 596.
*        Modified to accept hue value between (0,1)
*
* CHANGES
*
*   VK 01.07.2000 : Created.
*
******************************************************************************/
static COLC CValue(COLC n1, COLC n2, COLC hue)
{    
    if (hue > 1.0)
        hue -= 1.0;
    else if (hue < 0.0)
        hue += 1.0;
    
    if (hue < 1.0 / 6.0)
        return n1 + (n2-n1)*hue * 6.0;
    else if ( hue < 0.5)
        return n2;
    else if (hue < 2.0/3.0)
        return n1 + (n2-n1)*(2.0/3.0 - hue)*6.0;
    else 
        return n1;
}



/*****************************************************************************
*
* FUNCTION
*
*        Transform_Colour
*
* INPUT
*        Colour     - colour to convert.
*        to_space   - id for space, to which colour should be converted.
*        from_space - id for original colour space.
*   
* OUTPUT
*        Colour - converted colour.
*
* RETURNS
*        -
*   
* AUTHOR
*        VK
*   
* DESCRIPTION
*        Used conversion routines are taken from:
*          "Computer Graphics: Principles and Practice", 2nd ed. in C
*        and
*          "Color space FAQ", from "The Official POV-Ray CDROM", April 1995
*
* CHANGES
*
*   VK 01.07.2000 : Created.
*
******************************************************************************/
void Transform_Colour(RGB Colour, int to_space, int from_space){

    COLOUR tmp;
    tmp[0]=Colour[0];
    tmp[1]=Colour[1];
    tmp[2]=Colour[2];
    
    if (from_space != CS_RGB && to_space != CS_RGB){
        /* convert by RGB */
        Transform_Colour(Colour, from_space, CS_RGB);
        Transform_Colour(Colour, CS_RGB, to_space);        
        return;
    }
    
    if (from_space == CS_RGB){
        switch (to_space){
            case CS_HSV:
                {
                    COLC maxv = max3(tmp[0], tmp[1], tmp[2]);
                    COLC minv = min3(tmp[0], tmp[1], tmp[2]);
                    Colour[2] = maxv;
                    if (maxv != 0.0){
                        Colour[1] = (maxv-minv)/maxv;
                    }
                    else{
                        Colour[1] = 0.0;
                    }
                    if (Colour[1] == 0.0){
                        Colour[0] = 0.0;
                    }
                    else{
                        Colour[0] = RGBtoHue(tmp);
                    }
                }
                break;
            case CS_HSL:
                {
                    COLC maxv = max3(tmp[0], tmp[1], tmp[2]);
                    COLC minv = min3(tmp[0], tmp[1], tmp[2]);
                    Colour[2] = (maxv + minv)/2.0;
                    if (maxv == minv){
                        Colour[0] = 0.0;
                        Colour[1] = 0.0;
                    }
                    else{
                        COLC delta = maxv -minv;
                        if (Colour[2] <= 0.5){
                            Colour[1] = delta/(maxv+minv);
                        }
                        else{
                            Colour[1] = delta/(2.0 - (maxv+minv));
                        }
                        Colour[0] = RGBtoHue(tmp);
                    }
                }
                break;
            case CS_XYZ:
                Colour[0] = 0.607*tmp[0] + 0.174*tmp[1] + 0.200*tmp[2];
                Colour[1] = 0.299*tmp[0] + 0.587*tmp[1] + 0.114*tmp[2];
                Colour[2] =                0.066*tmp[1] + 1.111*tmp[2];
                break;
            case CS_XY_Y:
                {
                    COLC sum;
                    /* to XYZ */
                    Colour[0] = 0.607*tmp[0] + 0.174*tmp[1] + 0.200*tmp[2];
                    Colour[1] = 0.299*tmp[0] + 0.587*tmp[1] + 0.114*tmp[2];
                    Colour[2] =                0.066*tmp[1] + 1.111*tmp[2];
                    sum = Colour[0] + Colour[1] + Colour[2];
                    /* store Y value */
                    Colour[2]=Colour[1];
                    /* calculate x,y */
                    Colour[0] = Colour[0]/sum;
                    Colour[1] = Colour[1]/sum;
                }
                break;
            case CS_YIQ:
                Colour[0] = 0.299*tmp[0] + 0.587*tmp[1] + 0.114*tmp[2];
                Colour[1] = 0.596*tmp[0] - 0.275*tmp[1] - 0.321*tmp[2];
                Colour[2] = 0.212*tmp[0] - 0.523*tmp[1] + 0.311*tmp[2];         
                break;
        }
    }
    else{        
        /* convert to RGB */
        switch (from_space){
            case CS_HSV:
                if (tmp[1] == 0.0){
                    Colour[0] = Colour[1] = Colour[2] = tmp[2];
                }
                else{
                    COLC f, p, q, t;
                    int i;
                    if (tmp[0] == 1.0){
                        tmp[0] = 0.0;
                    }
                    tmp[0] *= 6; 
                    i = floor(tmp[0]);
                    f = tmp[0] - i;
                    p = tmp[2] * (1.0 - tmp[1]);
                    q = tmp[2] * (1.0 - (tmp[1] * f));
                    t = tmp[2] * (1.0 - (tmp[1] * (1.0 - f)));
                    switch(i){
                        case 0: 
                            Colour[0] = tmp[2]; Colour[1] = t; Colour[2] = p; 
                            break;
                        case 1: 
                            Colour[0] = q; Colour[1] = tmp[2]; Colour[2] = p; 
                            break;
                        case 2: 
                            Colour[0] = p; Colour[1] = tmp[2]; Colour[2] = t; 
                            break;
                        case 3: 
                            Colour[0] = p; Colour[1] = q; Colour[2] = tmp[2]; 
                            break;
                        case 4: 
                            Colour[0] = t; Colour[1] = p; Colour[2] = tmp[2]; 
                            break;
                        case 5: 
                            Colour[0] = tmp[2]; Colour[1] = p; Colour[2] = q; 
                            break;
                    }
                }
                break;
            case CS_HSL:
                {
                    COLC m1, m2;
                    m2 = (tmp[2] <= 0.5) ? 
                            (tmp[2] + (tmp[2] * tmp[1])) :
                            (tmp[2] + tmp[1] - (tmp[2] * tmp[1]));
                    m1 = 2.0 * tmp[2] - m2;
                    if (tmp[1] == 0.0){
                        Colour[0] = Colour[1] = Colour[2] = tmp[2];
                    }
                    else{
                        Colour[0] = CValue(m1, m2, tmp[0] + 1.0/3.0);
                        Colour[1] = CValue(m1, m2, tmp[0] );
                        Colour[2] = CValue(m1, m2, tmp[0] - 1.0/3.0);
                    }
                }
                break;
            case CS_XY_Y:
                {
                    /* convert from xyY to XYZ */
                    COLC z = 1-(tmp[0] + tmp[1]);
                    COLC valY = tmp[2];
                    tmp[0] = tmp[0]*tmp[2]/tmp[1];
                    tmp[2] = z*tmp[2]/tmp[1];
                    tmp[1] = valY;
                }
                /* NB! Deliberate fallthrough to convert from XYZ to RGB! */
            case CS_XYZ:
                Colour[0] =  1.9100*tmp[0] - 0.5338*tmp[1] - 0.2891*tmp[2];
                Colour[1] = -0.9844*tmp[0] + 1.9990*tmp[1] - 0.0279*tmp[2];
                Colour[2] =  0.0585*tmp[0] - 0.1187*tmp[1] + 0.9017*tmp[2];
                break;
            case CS_YIQ:
                Colour[0] = tmp[0] + 0.9557*tmp[1] + 0.6199*tmp[2];
                Colour[1] = tmp[0] - 0.2716*tmp[1] - 0.6469*tmp[2];
                Colour[2] = tmp[0] - 1.1082*tmp[1] + 1.7051*tmp[2];         
                break;
        }
    }
}



