/*==============================================================================
Project:	POV-Ray

Version:	3

File:	PovMacUtils.c

Description:
	Misc. global routines for POV-Ray MacOS
------------------------------------------------------------------------------
Authors:
	Eduard [esp] Schwan
------------------------------------------------------------------------------
	from Persistence of Vision(tm) Ray Tracer
	Copyright 1996,1999 Persistence of Vision Team
------------------------------------------------------------------------------
	NOTICE: This source code file is provided so that users may experiment
	with enhancements to POV-Ray and to port the software to platforms other 
	than those supported by the POV-Ray Team.  There are strict rules under
	which you are permitted to use this file.  The rules are in the file
	named POVLEGAL.DOC which should be distributed with this file.
	If POVLEGAL.DOC is not available or for more info please contact the POV-Ray
	Team Coordinator by email to team-coord@povray.org or visit us on the web at
	http://www.povray.org. The latest version of POV-Ray may be found at this site.

	This program is based on the popular DKB raytracer version 2.12.
	DKBTrace was originally written by David K. Buck.
	DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
------------------------------------------------------------------------------
More Info:
	This Macintosh version of POV-Ray was updated and compiled by Eduard Schwan
	and now builds under Metrowerks CodeWarrior CW Pro 3+ for 68k and PPC.
	It used to, and may still build under Apple MPW 3.3.x, and Symantec C 8.
	The Macintosh code was based (loosely) on the original DKB 0.5 port by
	Thomas Okken and David Lichtman, with some help from Glenn Sugden.
	Jim Nitchals was instrumental in getting 1.0 out, Jim and Eduard jointly
	pulled off 2.x with help from Dave Harr for initial Balloon help and the
	custom palette code, and 3.0 was done by Eduard Schwan.  3.1 was done jointly
	by Eduard Schwan and Thorsten Froehlich.

	For bug reports regarding the Macintosh version, you may contact the
	POV-Ray MacOS team at the following e-mail address.  Please send in bugs
	by filling out the bug report form, in the Documentation folder with POV-Ray:
	POV-Ray MacOS Team
		mailto:	mac-team@povray.org
		WWW:	http://mac.povray.org/
		News:	news://news.povray.org/
------------------------------------------------------------------------------
Change History:
	980530	[esp]	split off from PovMac.c
==============================================================================*/

/*==== POV-Ray std headers ====*/
#include "config.h"
#include "Frame.h"

#include "povray.h"		// for alt_main
#include "OptOut.h"		// DISTRIBUTION_MESSAGE

/*==== Standard C headers ====*/
#include <stdlib.h>
#include <setjmp.h>


/*==== Macintosh-specific headers ====*/

#include <Types.h>
#include <Resources.h>
#include <Timer.h> // For Microseconds call! [TRF]
#include <Math64.h> // For Microseconds call! [TRF]

#include "PovMacUtils.h"
#include "PovMac.h"		// gQuit
#include "AppPrefs.h"
#include "FilePrefs.h"	// gPrefs2Use
#include "ImageWindow.h"
#include "AboutBox.h"

SndChannelPtr			gSndChannelPtr = NULL;	// Beep this!


static Boolean			gQuit = false;

// ---------------------------------------------------------------------
// return the state of the Quit flag
#pragma segment Main

Boolean AppIsQuitting(void)
	{
	return (gQuit && ImageWindow_CanQuit());
	} // AppIsQuitting()


// ---------------------------------------------------------------------
// turn ON the Quit flag
#pragma segment Main

void SetAppQuit(Boolean quitting)
	{
	gQuit = quitting;
	} // SetAppQuit()


// ------------------------------------------------------------------
// look up font name and size from STR# indexes passed
void GetCfgFontInfo(short fontNameIdx, short fontSizeIdx, short * fontFamIDPtr, short * fontSizePtr)
{
	long	longFontSize;
	Str255	fontStr;
	// Get the font name to use
	fontStr[0]=0;
	GetIndString(fontStr, kSTRFONTS_ID, fontNameIdx);
	if (!fontStr[0])
		SysBeep(2);	// sort of an error
	// get font family ID from its name
	GetFNum(fontStr, fontFamIDPtr);
	// and get its font size
	fontStr[0]=0;
	GetIndString(fontStr, kSTRFONTS_ID, fontSizeIdx);
	StringToNum(fontStr, &longFontSize);
	//return it
	*fontSizePtr	= longFontSize;
}

// ------------------------------------------------------------------
// look up font name and size from STR# indexes passed, & set current window
void SetCfgFontInfo(short fontNameIdx, short fontSizeIdx, Boolean doForDialog)
{
	short	fontFamID, fontSize;
	// Get font info
	GetCfgFontInfo(fontNameIdx, fontSizeIdx, &fontFamID, &fontSize);
	// -- Now set it
//	TextFont(fontFamID);
	// set dialog to use this font
//	if (doForDialog)
//		SetDialogFont(fontFamID);
	// set its size in the window/dialog
//	if (fontSize > 0)
//		TextSize(fontSize);
}

// ---------------------------------------------------------------------
// Set default (current) dir to that of the POV-Ray app, so any C-library
// file open calls will look for files in this directory.
OSErr SetCurrentDirToAppDir(void)
{
	OSErr	anError;
	FSSpec	myAppSpec;
	ProcessSerialNumber	myPSN;
	ProcessInfoRec		myProcessRec;

	// Find info about our process
	myPSN.highLongOfPSN = 0;
	myPSN.lowLongOfPSN  = kCurrentProcess; // ourself
	myProcessRec.processAppSpec = &myAppSpec;
	myProcessRec.processInfoLength = sizeof(myProcessRec);
	myProcessRec.processName = NULL;
	anError = GetProcessInformation(&myPSN, &myProcessRec);
	// set current directory to the application directory
	if (!anError)
		anError = HSetVol(NULL, myAppSpec.vRefNum, myAppSpec.parID);
	return anError;
} // SetCurrentDirToAppDir

// ---------------------------------------------------------------------
// Play a nifty "finished" sound
void PlayNotifySound(void)
{
// if you don't have universal headers...
// #define SndListHandle Handle
	SndListHandle	theSndHandle;
	OSErr			anError;
	short			soundResID;
	ResType			soundResType;
	Str255			soundName;

	// Get the user's chosen sound ID from prefs
	soundResID = APGetDataAccessor(kAPData_AllDoneSoundResID, NULL);
	// load that sound
	theSndHandle = (SndListHandle)GetResource(soundListRsrc, soundResID);
	// play it
	if (theSndHandle)
	{
		if ((**gPrefs2Use_h).progress == eProgDebug)
			{
			GetResInfo((Handle)theSndHandle, &soundResID, &soundResType, soundName);
			P2CStr(soundName);
			printf("\n### Sound playing is #%d/'%s' (%ldK bytes)\n\n",
					soundResID, soundName, GetResourceSizeOnDisk((Handle)theSndHandle)/1024L);
			}
		anError = noErr;
		// Create a sound channel first time through
		if (gSndChannelPtr == NULL)
			anError = SndNewChannel(&gSndChannelPtr, sampledSynth, 0L, NULL);
		// Normally we can play the sound asynchronously, but if we're automatically
		// quitting, we need to play synchronously, so we don't quit the app
		// while still playing the sound (which simply truncates the sound.)
		if (!anError)
			anError = SndPlay(gSndChannelPtr, theSndHandle, !AppIsQuitting());
	}
} // PlayNotifySound


// ---------------------------------------------------------------------
// Convert current date/time into an ASCII C string for display
char * GetDateTimeString(char *pbuff)
{
	unsigned long	dtsecs;
	char			tbuff[64];
	
	// Does anybody really know what time it is?
	GetDateTime(&dtsecs);

	// convert date to string
	DateString(dtsecs, shortDate, (StringPtr)tbuff, NULL);
	P2CStr((StringPtr)tbuff);
	strcpy(pbuff, tbuff);

	// put date/time separator in buffer
	strcat(pbuff, "  ");

	// convert time to string
	TimeString(dtsecs, true, (StringPtr)tbuff, NULL);
	P2CStr((StringPtr)tbuff);
	strcat(pbuff, tbuff);

	// "give user his filled buffer back"
	return (pbuff);
} // GetDateTimeString



// ---------------------------------------------------------------------
// TickCount replacement for future Mac OS X
UInt32 GetTicks(void)
{
	UInt64 ms,msConst; 

	Microseconds((UnsignedWide *)(&ms));
	msConst = U64SetU(60L);
	ms = U64Multiply(ms, msConst);
	msConst = U64SetU(1000000L);
	ms = U64Div(ms, msConst);

	return U32SetU(ms);
}

// ---------------------------------------------------------------------
// Precision timer support
UInt64 gMacMicroTimerCount;
unsigned long gMacMicroTimerStop;

// ---------------------------------------------------------------------
// Precision timer start
void MacMicroTimerStart(void)
{
	Microseconds((UnsignedWide *)(&gMacMicroTimerCount));
}


// ---------------------------------------------------------------------
// Precision timer end
void MacMicroTimerStop(void)
{
	UInt64 curCnt;

	Microseconds((UnsignedWide *)(&curCnt));

	U64Subtract(curCnt, gMacMicroTimerCount);

	gMacMicroTimerStop = U32SetU(curCnt);
}


// ---------------------------------------------------------------------
// Precision timer count
unsigned long MacMicroTimerCount(void)
{
	return gMacMicroTimerStop;
}



// ---------------------------------------------------------------------
// Display additional credits on POV-Ray startup
void PrintMacCredits(void)
{
	char	appVers[32];

	GetAppVersionPString(1, (StringPtr)appVers);
	P2CStr((StringPtr)appVers);
	fprintf (stderr,"\n");
	fprintf (stderr,"POV-Ray version %s [%s]\n\n", appVers,COMPILER_VER);

	fprintf (stderr,"\n");
	fprintf (stderr,"**********************************************************\n");
	fprintf (stderr,"  This is an unofficial version of POV-Ray! The POV-Team\n");
	fprintf (stderr,"  is not responsible for supporting this version!\n");
	fprintf (stderr,"**********************************************************\n");
	fprintf (stderr,"\n");

	fprintf (stderr,"Original Macintosh implementation of POV-Ray is brought to you by:\n");
	fprintf (stderr,"  Eduard Schwan   Thorsten Froehlich\n");

	fprintf (stderr,"Past Macintosh development help:\n");
	fprintf (stderr,"  David Harr      Jim Nitchals\n");

#if defined(powerc) || defined (__powerc)
	fprintf (stderr,"Past Power Macintosh development guidance:\n");
	fprintf (stderr,"  David Harr      Paul Snively     Monroe Williams\n");
#endif

	fprintf (stderr,"----------------------------------------------------------------------\n\n");
} // PrintMacCredits



