/****************************************************************************
*                f_param.c
*
*  This module implements the parametric shapetype.
*  This module was written by D.Skarda&T.Bily and modified by R.Suzuki 
*  for POV3.0.
*
*****************************************************************************/
#include "frame.h"
#include "povray.h"
#include "objects.h"
#include "boxes.h"
#include "povproto.h"
#include "vector.h"
#include "matrices.h"
#include "bbox.h"
#include "mem.h"
#include "isosrf.h"
#include "f_expr.h"


#define Param_Tolerance 1.0e-8
#define Max_intNumber 10000000
#define close(x, y) (fabs(x-y) < EPSILON ? 1 : 0)

/*****************************************************************************
* Static functions
******************************************************************************/
static int  All_Parametric_Intersections (OBJECT *Object, RAY *Ray, ISTACK *Depth_Stack);
static int  Inside_Parametric (VECTOR point, OBJECT *Object);
static void Parametric_Normal (VECTOR Result, OBJECT *Object, INTERSECTION *Inter);
static void Translate_Parametric (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Rotate_Parametric (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Scale_Parametric (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Transform_Parametric (OBJECT *Object, TRANSFORM *Trans);
static void Invert_Parametric (OBJECT *Object);

METHODS Parametric_Methods =
  { 
  All_Parametric_Intersections,
  Inside_Parametric, Parametric_Normal,	Default_UVCoord,
  Copy_Parametric,
  Translate_Parametric, Rotate_Parametric,
  Scale_Parametric, Transform_Parametric, Invert_Parametric,
  Destroy_Parametric
};


void compute_param_normal( PARAMETRIC *Object,DBL u, DBL v, VECTOR Result);
int Intersect_Par_Box (RAY *Ray, PARAMETRIC *box, DBL *Depth1, DBL *Depth2, int *Side1, int *Side2);

static DBL Intervals_Low[2][32], Intervals_Hi[2][32];
static int SectorNum[32];

#define INDEX_U 0
#define INDEX_V 1

#define CONTINUE  { i--; continue;}

int All_Parametric_Intersections (OBJECT *Object, RAY    *Ray, ISTACK *Depth_Stack)
  {
   PARAMETRIC *Par= (PARAMETRIC *) Object;
   PRECOMP_PAR_DATA *PData= ((PARAMETRIC *) Object)->PData;
   int split, i=0, Side1, Side2;
   VECTOR P, D, N, IPoint;

   DBL  Depth1, Depth2, temp, Len, UResult, VResult, TResult= HUGE_VAL;
   DBL  XRayMin, XRayMax, YRayMin, YRayMax, ZRayMin, ZRayMax, TPotRes,TLen;
   int  parX, parY;
   int  MaxPrecompX,MaxPrecompY,MaxPrecompZ;


   Increase_Counter(stats[Ray_Par_Box_Tests]);
   Increase_Counter(stats[Ray_Par_Tests]);
   if (! Intersect_Par_Box (Ray, Object, &Depth1, &Depth2, &Side1, &Side2))
         return (FALSE);
   Increase_Counter(stats[Ray_Par_Box_Tests_Succeeded]);
   Increase_Counter(stats[Ray_Parametric_Tests]);

   if (Par->Trans != NULL) 
     {
      MInvTransPoint(P, Ray->Initial, Par->Trans);
      MInvTransDirection(D, Ray->Direction, Par->Trans);
     }
   else 
     {
      P[X] = Ray->Initial[X];
      P[Y] = Ray->Initial[Y];
      P[Z] = Ray->Initial[Z];
      D[X] = Ray->Direction[X];
      D[Y] = Ray->Direction[Y];
      D[Z] = Ray->Direction[Z];
     }
 
   if (Depth1==Depth2) Depth1=0;

   if((Par->bounds[0][X]<=P[X])&&(P[X]<=Par->bounds[1][X])&&  
       (Par->bounds[0][Y]<=P[Y])&&(P[Y]<=Par->bounds[1][Y])&&  
       (Par->bounds[0][Z]<=P[Z])&&(P[Z]<=Par->bounds[1][Z])) 
         if ((Depth1+= 4*Par->accuracy) > Depth2) return (FALSE);
   

   Intervals_Low[INDEX_U][0]= Par->umin;
   Intervals_Hi [INDEX_U][0]= Par->umax;


   Intervals_Low[INDEX_V][0]= Par->vmin;
   Intervals_Hi [INDEX_V][0]= Par->vmax;
/* Fri 09-27-1996 0. */
   SectorNum[0]= 1;

   MaxPrecompX= MaxPrecompY= MaxPrecompZ= 0;
   if (PData != NULL)
    {
     if (((PData -> flags)& OK_X)!=0) MaxPrecompX= 1 << (PData->depth);
     if (((PData -> flags)& OK_Y)!=0) MaxPrecompY= 1 << (PData->depth);
     if (((PData -> flags)& OK_Z)!=0) MaxPrecompZ= 1 << (PData->depth);

     cs_hi_top= cs_int_hi;
     cs_low_top= cs_int_low;
    }
/* 0 */
   while ( i>=0 )
    {
     func_U_low= Intervals_Low[INDEX_U][i];
     func_U_hi = Intervals_Hi [INDEX_U][i];
     Len= func_U_hi - func_U_low;
     split=INDEX_U; 

     func_V_low= Intervals_Low[INDEX_V][i];
     func_V_hi = Intervals_Hi [INDEX_V][i];
     temp= func_V_hi - func_V_low;
     if (temp > Len) {Len= temp; split=INDEX_V;}
     parX= parY= 0;
     TLen=0;

  /* X */
     if (SectorNum[i] < MaxPrecompX) 
       { 
        *cs_low_top= PData->Low[0][SectorNum[i]];
        *cs_hi_top=  PData->Hi [0][SectorNum[i]];
       }
     else evaluate_interval( Par -> Func[0], 0 /*!!!!*/ );
                  /* fabs(D[X] *(T2-T1)) is not OK with new method */

     if (close(D[0],0))
      {
       parX=1;
       if ((*cs_hi_top < P[0])||(*cs_low_top > P[0])) CONTINUE 
      } 
     else 
      {
       XRayMin= (*cs_hi_top - P[0])/D[0];
       XRayMax= (*cs_low_top - P[0])/D[0];
       if (XRayMin > XRayMax)
         { temp= XRayMin; XRayMin= XRayMax; XRayMax= temp; }

       if ((XRayMin>Depth2)||(XRayMax<Depth1)) CONTINUE  
       if ((TPotRes= XRayMin) > TResult) CONTINUE

       TLen= XRayMax-XRayMin;
      }

  /* Y */
     if (SectorNum[i] < MaxPrecompY) 
       { 
        *cs_low_top= PData->Low[1][SectorNum[i]];
        *cs_hi_top=  PData->Hi [1][SectorNum[i]];
       }
     else evaluate_interval( Par -> Func[1], 0 /*!!!*/);

     if (close(D[1],0))
      {
       parY=1;
       if ((*cs_hi_top < P[1])||(*cs_low_top > P[1])) CONTINUE 
      } 
     else 
      {
       YRayMin= (*cs_hi_top - P[1])/D[1];
       YRayMax= (*cs_low_top - P[1])/D[1];
       if (YRayMin > YRayMax)
         { temp= YRayMin; YRayMin= YRayMax; YRayMax= temp; }

       if ((YRayMin>Depth2)||(YRayMax<Depth1)) CONTINUE
       if ((TPotRes= YRayMin) > TResult) CONTINUE
       if (parX==0)
         if ((YRayMin > XRayMax)||(YRayMax < XRayMin)) CONTINUE 

       if ((temp= YRayMax-YRayMin) > TLen) TLen= temp;
      }

  /* Z */
     if ((SectorNum[i] < MaxPrecompZ)&&(0<SectorNum[i])) 
       { 
        *cs_low_top= PData->Low[2][SectorNum[i]];
        *cs_hi_top=  PData->Hi [2][SectorNum[i]];
       }
     else evaluate_interval( Par -> Func[2], 0/*!!!!!!*/ );

     if (close(D[2],0))
      {
       if ((*cs_hi_top < P[2])||(*cs_low_top > P[2])) CONTINUE 
      } 
     else 
      {
       ZRayMin= (*cs_hi_top - P[2])/D[2];
       ZRayMax= (*cs_low_top - P[2])/D[2];
       if (ZRayMin > ZRayMax)
         { temp= ZRayMin; ZRayMin= ZRayMax; ZRayMax= temp; }

       if ((ZRayMin>Depth2)||(ZRayMax<Depth1)) CONTINUE
       if ((TPotRes= ZRayMin) > TResult) CONTINUE
       if (parX==0)
         if ((ZRayMin > XRayMax)||(ZRayMax < XRayMin)) CONTINUE 

       if (parY==0)
         if ((ZRayMin > YRayMax)||(ZRayMax < YRayMin)) CONTINUE 

       if ((temp= ZRayMax-ZRayMin) > TLen) TLen= temp;
      }
     if (Len > TLen) Len= TLen;



     if ( Len < Par->accuracy ) 
      {
       if ((TResult > TPotRes)&&(TPotRes>Depth1))
        {
         TResult= TPotRes;
         Par->last_u= UResult= func_U_low;
         Par->last_v= VResult= func_V_low;
        }
       i--;
      }
     else   
      {
    /* 1) copy */
       if ((SectorNum[i]*= 2)>=Max_intNumber) SectorNum[i]=Max_intNumber;
       SectorNum[i+1]= SectorNum[i];
       SectorNum[i]++;
       i++;   
       Intervals_Low[INDEX_U][i]= func_U_low;
       Intervals_Hi [INDEX_U][i]= func_U_hi;

       Intervals_Low[INDEX_V][i]= func_V_low;
       Intervals_Hi [INDEX_V][i]= func_V_hi; 

    /* 2) split */
       temp= (Intervals_Hi[split][i] + Intervals_Low[split][i]) / 2.0 ;
       Intervals_Hi[ split][i]= temp;
       Intervals_Low[ split][i-1]= temp;
      }
    }

   if (TResult < Depth2)
    {
     Increase_Counter(stats[Ray_Parametric_Tests_Succeeded]);
     VScale (IPoint, Ray->Direction, TResult );
     VAddEq (IPoint, Ray->Initial);
 
     if (Point_In_Clip (IPoint, Par->Clip))
       {
      /*
        compute_param_normal( Par, UResult, VResult , &N); 
        push_normal_entry( TResult ,IPoint, N, (OBJECT *) Object, Depth_Stack);
      */
        push_entry( TResult ,IPoint, (OBJECT *) Object, Depth_Stack);
        return (TRUE);
       }
    }

   return (FALSE);
  }
/* 0 */

int Inside_Parametric (VECTOR IPoint, OBJECT *Object)
  {
   return (FALSE);
  }

/* Fri 09-27-1996 0. */
void Parametric_Normal (VECTOR Result, OBJECT *Object, INTERSECTION *Inter )
  {
   VECTOR     RU, RV;
   PARAMETRIC *Par= (PARAMETRIC *) Object;
   VECTOR *IPoint= &(Inter->IPoint);

   func_U= Par->last_u; 
   func_V= Par->last_v;

   evaluate_function( Par -> Func[0] );
   RU[0]= RV[0]= - *calc_stack;         

   evaluate_function( Par -> Func[1] );
   RU[1]= RV[1]= - *calc_stack;

   evaluate_function( Par -> Func[2] );
   RU[2]= RV[2]= - *calc_stack;


   func_U+= Par-> accuracy;
   evaluate_function( Par -> Func[0] );
   RU[0]+= *calc_stack;
   evaluate_function( Par -> Func[1]  );
   RU[1]+= *calc_stack;             
   evaluate_function( Par -> Func[2]  );
   RU[2]+= *calc_stack;

   func_U= Par->last_u;  
   func_V+= Par-> accuracy;
   evaluate_function( Par -> Func[0] );
   RV[0]+= *calc_stack;
   evaluate_function( Par -> Func[1] );
   RV[1]+= *calc_stack;
   evaluate_function( Par -> Func[2] );
   RV[2]+= *calc_stack;

   VCross( Result, RU, RV );
   if (Par->Trans != NULL) MTransNormal( Result, Result, Par->Trans);
   VNormalize( Result, Result );

  }
/* 0 */


void Compute_Parametric_BBox(PARAMETRIC *Param)
{
  Assign_BBox_Vect(Param->BBox.Lower_Left, Param->bounds[0]);

  VSub(Param->BBox.Lengths, Param->bounds[1], Param->bounds[0]);

  if (Param->Trans != NULL)
  {
    Recompute_BBox(&Param->BBox, Param->Trans);
  }
}



void Translate_Parametric (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans)
 {
  Transform_Parametric(Object, Trans);
 }

void Rotate_Parametric (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans)
 {
  Transform_Parametric(Object, Trans);
 }

void Scale_Parametric  (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans)
 {
  Transform_Parametric(Object, Trans);
 }

void Transform_Parametric(OBJECT *Object,TRANSFORM *Trans)
 {
  PARAMETRIC *Param= (PARAMETRIC *) Object;
  if (Param->Trans == NULL)
    Param->Trans = Create_Transform();

  Compose_Transforms(Param->Trans, Trans);

  Compute_Parametric_BBox(Param);

 }

void Invert_Parametric (OBJECT *Object)
  {
  }

void *Copy_Parametric (OBJECT *Object)
 {
  PARAMETRIC *New, *Old;

  Old= (PARAMETRIC *) Object;
  New= Create_Parametric();
  COPY_OBJECT_FIELDS( New, Object );
     
  New->Func[0]=Copy_Function( Old->Func[0] );      
  New->Func[1]=Copy_Function( Old->Func[1] );    
  New->Func[2]=Copy_Function( Old->Func[2] );    
     
  New->Trans= Copy_Transform( Old->Trans );
  New->PData= Copy_PrecompParVal( Old->PData );
     
  Assign_Vector(New->bounds[0],Old->bounds[0]);
  Assign_Vector(New->bounds[1],Old->bounds[1]);
  New->accuracy= Old->accuracy;

  New->last_u = Old->last_u;
  New->last_v = Old->last_v;

  New->umin = Old->umin;
  New->vmin = Old->vmin;
  New->umax = Old->umax;
  New->vmax = Old->vmax;

  New->Inverted = Old->Inverted;
  
  return ((OBJECT *) New);
 }


void Destroy_Parametric (OBJECT *Object)
 {
  Destroy_Transform(((PARAMETRIC *)Object)->Trans);
  Destroy_Function(((PARAMETRIC *)Object)->Func[0]);
  Destroy_Function(((PARAMETRIC *)Object)->Func[1]);
  Destroy_Function(((PARAMETRIC *)Object)->Func[2]);
  Destroy_PrecompParVal(((PARAMETRIC *)Object)->PData);
  POV_FREE( Object );
 }


PARAMETRIC *Create_Parametric ()
 {
  PARAMETRIC *New;

  New = (PARAMETRIC *) POV_MALLOC(sizeof (PARAMETRIC), "parametric");

  INIT_OBJECT_FIELDS(New, PARAMETRIC_OBJECT, &Parametric_Methods)
  Make_Vector ((New->bounds[1]), 1.0, 1.0, 1.0);
  Make_Vector ((New->bounds[0]), -1.0, -1.0, -1.0);

  New->accuracy= 0.001;

  New->Trans = NULL;
  New->Inverted = FALSE;
  New->PData= NULL;

  return (New);
 }

static PRECOMP_PAR_DATA *PrecParData;
static PARAMETRIC *PrecompParFunc;
static int         PrecompLastDepth;

static void Precomp_Par_Int( int depth, DBL umin, DBL vmin, DBL umax, DBL vmax)
 {
  int j;

  if (depth>=PrecompLastDepth)
   {  /* compute */
    func_U_low= umin;
    func_U_hi = umax;
    func_V_low= vmin;
    func_V_hi = vmax;

    for(j=0; j<3; j++)
     if (PrecParData->flags & (1<<j))
     {
      evaluate_interval( PrecompParFunc->Func[j], 0/*!!!*/ );
      PrecParData->Low[j][depth]= *cs_low_top;
      PrecParData->Hi [j][depth]= *cs_hi_top;
     }
   }
  else  /* split */
   {
    if ( umax-umin < vmax-vmin )
     {
      Precomp_Par_Int( 2*depth  ,umin,vmin,umax,(vmin+vmax)/2.0);
      Precomp_Par_Int( 2*depth+1,umin,(vmin+vmax)/2.0,umax,vmax);
     }
    else
     {
      Precomp_Par_Int( 2*depth  ,umin,vmin,(umin+umax)/2.0,vmax);
      Precomp_Par_Int( 2*depth+1,(umin+umax)/2.0,vmin,umax,vmax );
     }

    for (j=0; j<3; j++)
     if (PrecParData->flags & (1<<j))
     {
      if (PrecParData->Hi[j][2*depth] > PrecParData->Hi[j][2*depth+1])
           PrecParData->Hi[j][depth]= PrecParData->Hi[j][2*depth];
      else PrecParData->Hi[j][depth]= PrecParData->Hi[j][2*depth+1];

      if (PrecParData->Low[j][2*depth] < PrecParData->Low[j][2*depth+1])
           PrecParData->Low[j][depth]= PrecParData->Low[j][2*depth];
      else PrecParData->Low[j][depth]= PrecParData->Low[j][2*depth+1];
     }
   }
 }



PRECOMP_PAR_DATA *Precompute_Parametric_Values (PARAMETRIC *Par, char flags, int depth)
  {
   PRECOMP_PAR_DATA *PData;
   DBL *Xlow, *XHi, *Ylow, *YHi, *ZLow, *ZHi, *Last;
   char *es="precompute";
   int nmb;

   if ((depth<1)||(depth>20)) Error("Precompute: invalid depth");
   nmb= 1 << depth;

   PData= (PRECOMP_PAR_DATA*) POV_MALLOC( sizeof( PRECOMP_PAR_DATA ), es);
   if (PData==NULL) MAError("precompute", sizeof( PRECOMP_PAR_DATA ));
   PData-> flags= flags;
   PData-> depth= depth;
   PData-> use= 1;

   if (flags & OK_X)
    {
           PData-> Low[0]= (DBL *) POV_MALLOC( sizeof(DBL) * nmb, es );
     Last= PData-> Hi[0] = (DBL *) POV_MALLOC( sizeof(DBL) * nmb, es );
    }
   if (flags & OK_Y)
    {
           PData-> Low[1]= (DBL *) POV_MALLOC( sizeof(DBL) * nmb, es );
     Last= PData-> Hi[1] = (DBL *) POV_MALLOC( sizeof(DBL) * nmb, es );
    }
   if (flags & OK_Z)
    {
           PData-> Low[2]= (DBL *) POV_MALLOC( sizeof(DBL) * nmb, es );
     Last= PData-> Hi[2] = (DBL *) POV_MALLOC( sizeof(DBL) * nmb, es );
    }
   if (Last == NULL) MAError("precompute", sizeof(DBL) * nmb);

   PrecompLastDepth= 1 << (depth-1);
   PrecParData= PData;
   PrecompParFunc= Par;

   Precomp_Par_Int( 1, Par->umin, Par->vmin, Par->umax, Par->vmax);

   return (PData);
  }

PRECOMP_PAR_DATA *Copy_PrecompParVal ( PRECOMP_PAR_DATA *PPV )
 {
  if ( PPV )
	 (PPV->use)++;
  return (PPV);
 }

void Destroy_PrecompParVal ( PRECOMP_PAR_DATA *PPV )
 {
  if (PPV==NULL)return;
  if (((PPV->use)--)==0)
   {
    if (PPV->flags & OK_X) { POV_FREE(PPV->Low[0]); free(PPV->Hi[0]); }
    if (PPV->flags & OK_Y) { POV_FREE(PPV->Low[1]); free(PPV->Hi[1]); }
    if (PPV->flags & OK_Z) { POV_FREE(PPV->Low[2]); free(PPV->Hi[2]); }
    free( PPV );
   }
 }



/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

/* Minimal intersection depth. */

#define DEPTH_TOLERANCE 1.0e-6

/* Two values are equal if their difference is small than CLOSE_TOLERANCE. */

#define CLOSE_TOLERANCE 1.0e-6

/* Side hit. */

#define SIDE_X_0 1
#define SIDE_X_1 2
#define SIDE_Y_0 3
#define SIDE_Y_1 4
#define SIDE_Z_0 5
#define SIDE_Z_1 6


int Intersect_Par_Box(RAY *Ray, PARAMETRIC *box, DBL *Depth1, DBL *Depth2, int *Side1, int *Side2)
{
  int smin = 0, smax = 0;    /* Side hit for min/max intersection. */
  DBL t, tmin, tmax;
  VECTOR P, D;

  /* Transform the point into the boxes space */

  if (box->Trans != NULL)
  {
    MInvTransPoint(P, Ray->Initial, box->Trans);
    MInvTransDirection(D, Ray->Direction, box->Trans);
  }
  else
  {
    Assign_Vector(P, Ray->Initial);
    Assign_Vector(D, Ray->Direction);
  }

  tmin = 0.0;
  tmax = BOUND_HUGE;

  /*
   * Sides first.
   */

  if (D[X] < -EPSILON)
  {
    t = (box->bounds[0][X] - P[X]) / D[X];

    if (t < tmin) return(FALSE);

    if (t <= tmax)
    {
      smax = SIDE_X_0;
      tmax = t;
    }

    t = (box->bounds[1][X] - P[X]) / D[X];

    if (t >= tmin)
    {
      if (t > tmax) return(FALSE);

      smin = SIDE_X_1;
      tmin = t;
    }
  }
  else
  {
    if (D[X] > EPSILON)
    {
      t = (box->bounds[1][X] - P[X]) / D[X];

      if (t < tmin) return(FALSE);

      if (t <= tmax)
      {
        smax = SIDE_X_1;
        tmax = t;
      }

      t = (box->bounds[0][X] - P[X]) / D[X];

      if (t >= tmin)
      {
        if (t > tmax) return(FALSE);

        smin = SIDE_X_0;
        tmin = t;
      }
    }
    else
    {
      if ((P[X] < box->bounds[0][X]) || (P[X] > box->bounds[1][X]))
      {
        return(FALSE);
      }
    }
  }

  /*
   * Check Top/Bottom.
   */

  if (D[Y] < -EPSILON)
  {
    t = (box->bounds[0][Y] - P[Y]) / D[Y];

    if (t < tmin) return(FALSE);

    if (t <= tmax - CLOSE_TOLERANCE)
    {
      smax = SIDE_Y_0;
      tmax = t;
    }
    else
    {
      /*
       * If intersection points are close to each other find out
       * which side to use, i.e. is most probably hit. [DB 9/94]
       */

      if (t <= tmax + CLOSE_TOLERANCE)
      {
        if (-D[Y] > fabs(D[X])) smax = SIDE_Y_0;
      }
    }

    t = (box->bounds[1][Y] - P[Y]) / D[Y];

    if (t >= tmin + CLOSE_TOLERANCE)
    {
      if (t > tmax) return(FALSE);

      smin = SIDE_Y_1;
      tmin = t;
    }
    else
    {
      /*
       * If intersection points are close to each other find out
       * which side to use, i.e. is most probably hit. [DB 9/94]
       */

      if (t >= tmin - CLOSE_TOLERANCE)
      {
        if (-D[Y] > fabs(D[X])) smin = SIDE_Y_1;
      }
    }
  }
  else
  {
    if (D[Y] > EPSILON)
    {
      t = (box->bounds[1][Y] - P[Y]) / D[Y];

      if (t < tmin) return(FALSE);

      if (t <= tmax - CLOSE_TOLERANCE)
      {
        smax = SIDE_Y_1;
        tmax = t;
      }
      else
      {
        /*
         * If intersection points are close to each other find out
         * which side to use, i.e. is most probably hit. [DB 9/94]
         */

        if (t <= tmax + CLOSE_TOLERANCE)
        {
          if (D[Y] > fabs(D[X])) smax = SIDE_Y_1;
        }
      }

      t = (box->bounds[0][Y] - P[Y]) / D[Y];

      if (t >= tmin + CLOSE_TOLERANCE)
      {
        if (t > tmax) return(FALSE);

        smin = SIDE_Y_0;
        tmin = t;
      }
      else
      {
        /*
         * If intersection points are close to each other find out
         * which side to use, i.e. is most probably hit. [DB 9/94]
         */

        if (t >= tmin - CLOSE_TOLERANCE)
        {
          if (D[Y] > fabs(D[X])) smin = SIDE_Y_0;
        }
      }
    }
    else
    {
      if ((P[Y] < box->bounds[0][Y]) || (P[Y] > box->bounds[1][Y]))
      {
        return(FALSE);
      }
    }
  }

  /* Now front/back */

  if (D[Z] < -EPSILON)
  {
    t = (box->bounds[0][Z] - P[Z]) / D[Z];

    if (t < tmin) return(FALSE);

    if (t <= tmax - CLOSE_TOLERANCE)
    {
      smax = SIDE_Z_0;
      tmax = t;
    }
    else
    {
      /*
       * If intersection points are close to each other find out
       * which side to use, i.e. is most probably hit. [DB 9/94]
       */

      if (t <= tmax + CLOSE_TOLERANCE)
      {
        switch (smax)
        {
          case SIDE_X_0 :
          case SIDE_X_1 : if (-D[Z] > fabs(D[X])) smax = SIDE_Z_0; break;

          case SIDE_Y_0 :
          case SIDE_Y_1 : if (-D[Z] > fabs(D[Y])) smax = SIDE_Z_0; break;
        }
      }
    }

    t = (box->bounds[1][Z] - P[Z]) / D[Z];

    if (t >= tmin + CLOSE_TOLERANCE)
    {
      if (t > tmax) return(FALSE);

      smin = SIDE_Z_1;
      tmin = t;
    }
    else
    {
      /*
       * If intersection points are close to each other find out
       * which side to use, i.e. is most probably hit. [DB 9/94]
       */

      if (t >= tmin - CLOSE_TOLERANCE)
      {
        switch (smin)
        {
          case SIDE_X_0 :
          case SIDE_X_1 : if (-D[Z] > fabs(D[X])) smin = SIDE_Z_1; break;

          case SIDE_Y_0 :
          case SIDE_Y_1 : if (-D[Z] > fabs(D[Y])) smin = SIDE_Z_1; break;
        }
      }
    }
  }
  else
  {
    if (D[Z] > EPSILON)
    {
      t = (box->bounds[1][Z] - P[Z]) / D[Z];

      if (t < tmin) return(FALSE);

      if (t <= tmax - CLOSE_TOLERANCE)
      {
        smax = SIDE_Z_1;
        tmax = t;
      }
      else
      {
        /*
         * If intersection points are close to each other find out
         * which side to use, i.e. is most probably hit. [DB 9/94]
         */

        if (t <= tmax + CLOSE_TOLERANCE)
        {
          switch (smax)
          {
            case SIDE_X_0 :
            case SIDE_X_1 : if (D[Z] > fabs(D[X])) smax = SIDE_Z_1; break;

            case SIDE_Y_0 :
            case SIDE_Y_1 : if (D[Z] > fabs(D[Y])) smax = SIDE_Z_1; break;
          }
        }
      }

      t = (box->bounds[0][Z] - P[Z]) / D[Z];

      if (t >= tmin + CLOSE_TOLERANCE)
      {
        if (t > tmax) return(FALSE);

        smin = SIDE_Z_0;
        tmin = t;
      }
      else
      {
        /*
         * If intersection points are close to each other find out
         * which side to use, i.e. is most probably hit. [DB 9/94]
         */

        if (t >= tmin - CLOSE_TOLERANCE)
        {
          switch (smin)
          {
            case SIDE_X_0 :
            case SIDE_X_1 : if (D[Z] > fabs(D[X])) smin = SIDE_Z_0; break;

            case SIDE_Y_0 :
            case SIDE_Y_1 : if (D[Z] > fabs(D[Y])) smin = SIDE_Z_0; break;
          }
        }
      }
    }
    else
    {
      if ((P[Z] < box->bounds[0][Z]) || (P[Z] > box->bounds[1][Z]))
      {
        return(FALSE);
      }
    }
  }

  if (tmax < DEPTH_TOLERANCE)
  {
    return (FALSE);
  }

  *Depth1 = tmin;
  *Depth2 = tmax;

  *Side1 = smin;
  *Side2 = smax;

  return(TRUE);
}


