/* quadtree.inc
 * this code was inspired by a conversation with Bill "Bald Eagle" Walker,
 * see 'quadtree.html' for documentation.  version 202005.2.
 */

#if (!defined(qtree_include_temp))

#declare qtree_include_temp = version;

#version 3.8;

#if (defined(View_POV_Include_Stack))
#debug "including quadtree.inc\n"
#end

#if (!defined(global.qtree_debug))
#declare qtree_debug = no;
#end

/* ------------------------------------------------------------------------- */
/* outside = '0' else '{index} + 1' */
#macro qtree__in_quadrant(a_, b_, p_)
  #local rtn_ = 0;
  #if (a_.x <= p_.x  &  p_.x <= b_.x)
    #if (a_.y <= p_.y  &  p_.y <= b_.y)
      #local c_ = qtree__mk_centre(a_, b_);
      #if (p_.x < c_.x)
        #if (p_.y < c_.y)
          #local rtn_ = 1;
        #else
          #local rtn_ = 3;
        #end
      #else
        #if (p_.y < c_.y)
          #local rtn_ = 2;
        #else
          #local rtn_ = 4;
        #end
      #end
    #end
  #end
  rtn_
#end

#macro qtree__inserts_point(t_, i_, p_)
  #local rtn_ = 0;
  #local q_ = qtree__in_quadrant(t_.nodes[i_][0], t_.nodes[i_][1], p_);
  #if (q_)
    #if (!dimension_size(t_.nodes[i_][2],1) &
            dimension_size(t_.nodes[i_][3],1) < t_.maxPts)
      #local t_.nodes[i_][3][dimension_size(t_.nodes[i_][3],1)] = p_;
      #local t_.nPts = t_.nPts + 1;
      #local rtn_ = 1;
    #else
      #if (!dimension_size(t_.nodes[i_][2],1))
        qtree__splits_node(t_, i_)
      #end
      #local rtn_ = qtree__inserts_point(t_, t_.nodes[i_][2][q_-1], p_);
    #end
  #end
  rtn_
#end

#macro qtree__intersect(a1_, a2_, b1_, b2_)
  (!(a2_.x < b1_.x  |  a1_.x > b2_.x  |  a1_.y > b2_.y  |  a2_.y < b1_.y))
#end

#macro qtree__is_rectangle(a_, b_)
  (a_.x < b_.x  &  a_.y < b_.y)
#end

#macro qtree__mk_centre(a_, b_)
<(a_.x + b_.x) / 2, (a_.y + b_.y) / 2>
#end

#macro qtree__mk_node(a_, b_)
array mixed {a_, b_, array {}, array {}}
#end

#macro qtree__prints_node(t_, i_)
  #if (!t_.root)
    #error "tree not rooted."
  #end
  qtree__emitVal("node id", i_, 0)
  qtree__emit2V("area from", t_.nodes[i_][0], 4)
  qtree__emit2V("to", t_.nodes[i_][1], 4)
  #if (dimension_size(t_.nodes[i_][2],1))
    #local s_ = concat("NW ",str(t_.nodes[i_][2][0],0,0)," ");
    #local s_ = concat(s_,"SW ",str(t_.nodes[i_][2][1],0,0)," ");
    #local s_ = concat(s_,"SE ",str(t_.nodes[i_][2][2],0,0)," ");
    #local s_ = concat(s_,"NE ",str(t_.nodes[i_][2][3],0,0));
  #else
    #local s_ = "none";
  #end
  qtree__emitStr("children", s_)
  #local n_ = dimension_size(t_.nodes[i_][3],1);
  #if (n_)
    qtree__emitStr("bucket", qtree__3VFmt(t_.nodes[i_][3][0],4))
    #for (j_, 1, n_ - 1)
      #debug concat("\t\t\t", qtree__3VFmt(t_.nodes[i_][3][j_],4), "\n")
    #end
  #else
    qtree__emitStr("bucket", "empty")
  #end
#end

#macro qtree__searches_node(t_, i_, a_, b_, res_)
  #local rtn_ = 0;
  #if (qtree__intersect(a_, b_, t_.nodes[i_][0], t_.nodes[i_][1]))
    #if (dimension_size(t_.nodes[i_][2],1))
      #for (j_, 0, 3)
        #local rtn_ = rtn_ +
                qtree__searches_node(t_, t_.nodes[i_][2][j_], a_, b_, res_);
      #end
    #else
      #for (j_, 0, dimension_size(t_.nodes[i_][3],1) - 1)
        #if (qtree__in_quadrant(a_, b_, t_.nodes[i_][3][j_]))
          #local res_[dimension_size(res_,1)] = t_.nodes[i_][3][j_];
          #local rtn_ = rtn_ + 1;
        #end
      #end
    #end
  #end
  rtn_
#end

#macro qtree__splits_node(t_, i_)
  #if (qtree_debug)
    #debug "[II] splitting node.  before:\n"
    qtree__prints_node(t_, i_)
  #end
  #local a_ = t_.nodes[i_][0];
  #local b_ = t_.nodes[i_][1];
  #local c_ = qtree__mk_centre(a_, b_);
  #local n_ = dimension_size(t_.nodes,1);
  #for (j_, 0, 3)
    #local t_.nodes[i_][2][dimension_size(t_.nodes[i_][2],1)] = n_ + j_;
  #end
  #local t_.nodes[t_.nodes[i_][2][0]] = qtree__mk_node(a_, c_);
  #local t_.nodes[t_.nodes[i_][2][1]] =
          qtree__mk_node(<c_.x, a_.y>, <b_.x, c_.y>);
  #local t_.nodes[t_.nodes[i_][2][2]] =
          qtree__mk_node(<a_.x, c_.y>, <c_.x, b_.y>);
  #local t_.nodes[t_.nodes[i_][2][3]] = qtree__mk_node(c_, b_);
  #for (j_, 0, dimension_size(t_.nodes[i_][3],1) - 1)
    #local q_ = qtree__in_quadrant(a_, b_, t_.nodes[i_][3][j_]);
    #local n_ = t_.nodes[i_][2][q_-1];
    #local t_.nodes[n_][3][dimension_size(t_.nodes[n_][3],1)] =
            t_.nodes[i_][3][j_];
  #end
  #local t_.nodes[i_][3] = array {};
  #if (qtree_debug)
    #debug "[II] splitting node.  after:\n"
    qtree__prints_node(t_, i_)
    #for (j_, 0, 3)
      qtree__prints_node(t_, t_.nodes[i_][2][j_])
    #end
  #end
#end
/* pretty printing */
#macro qtree__hyphens(n_)
  #local s_ = "";
  #for (i_,1,n_)
    #local s_ = concat(s_,"-");
  #end
  s_
#end
#macro qtree__strFmt(s_)
  #local w_ = 22;
  #local n_ = strlen(s_);
  #if (w_ > n_)
    #local s_ = concat(substr("                      ",1,(w_-n_)),s_);
  #end
  s_
#end
#macro qtree__2VFmt(v_,p_)
  concat("<",vstr(2,v_,", ",0,p_),">")
#end
#macro qtree__3VFmt(v_,p_)
  concat("<",vstr(3,v_,", ",0,p_),">")
#end
#macro qtree__emit2V(s_,v_,p_)
  #debug concat(qtree__strFmt(s_),": <",vstr(2,v_,", ",0,p_),">\n")
#end
#macro qtree__emitStr(s_,v_)
  #debug concat(qtree__strFmt(s_),": ",v_,"\n")
#end
#macro qtree__emitVal(s_,v_,p_)
  #debug concat(qtree__strFmt(s_),": ",str(v_,0,p_),"\n")
#end

/* ------------------------------------------------------------------------- */

#macro qtree_dump_tree(t_)
  #if (!t_.root)
    #error "oops, tree not \"root\"ed."
  #end
  #debug concat(qtree__hyphens(5), "[quadtree dump]", qtree__hyphens(52), "\n")
  qtree__emitVal("root node id", t_.root, 0)
  qtree__emit2V("area from", t_.nodes[t_.root][0], 4)
  qtree__emit2V("to", t_.nodes[t_.root][1], 4)
  #local n_ = dimension_size(t_.nodes,1);
  qtree__emitVal("# nodes", n_ - 1, 0)
  qtree__emitVal("bucket size", t_.maxPts, 0)
  qtree__emitVal("# points", t_.nPts, 0)
  #debug "\n"
  #for (i_, 1, n_ - 1)
    qtree__prints_node(t_, i_)
    #debug "\n"
  #end
  #debug concat(qtree__hyphens(72),"\n")
#end

#macro qtree_get_bucket_size(t_)
  #if (!t_.root)
    #error "oops, tree not \"root\"ed."
  #end
  (t_.maxPts)
#end

#macro qtree_get_number_nodes(t_)
  #if (!t_.root)
    #error "oops, tree not \"root\"ed."
  #end
  (dimension_size(t_.nodes,1) - 1)
#end

#macro qtree_get_number_points(t_)
  #if (!t_.root)
    #error "oops, tree not \"root\"ed."
  #end
  (t_.nPts)
#end

#macro qtree_insert_point(t_, p_)
  #if (!t_.root)
    #error "oops, tree not \"root\"ed."
  #end
  #local rtn_ = 0;
  #if (qtree__in_quadrant(t_.nodes[t_.root][0], t_.nodes[t_.root][1], p_))
    #local rtn_ = qtree__inserts_point(t_, t_.root, p_);
  #end
  #if (qtree_debug)
    #if (rtn_)
      #debug concat("[II] inserted point ",qtree__3VFmt(p_,4),".\n")
    #else
      #debug concat("[II] inserting point ",qtree__3VFmt(p_,4)," failed.\n")
    #end
  #end
  rtn_
#end

#macro qtree_make_root(t_, a_, b_)
  #if (!qtree__is_rectangle(a_, b_))
    #error concat("oops, points ", qtree__2VFmt(a_,4), " and ",
            qtree__2VFmt(b_,4), " do not make rectangle.")
  #end
  #if (!t_.root)
    #if (dimension_size(t_.nodes,1))
      #error "oops, tree has nodes, and invalid root id '0'."
    #end
    #local t_.nodes[1] = qtree__mk_node(a_, b_);
    #local t_.root = 1;
  #else
    // re-root.
    #warning "\"re-rooting\" a tree is To Be Implemented."
    #break
  #end
  #if (qtree_debug)
    #debug concat("[II] new tree root.  id ", str(t_.root,0,0), ", area ",
            qtree__2VFmt(a_,4), ",", qtree__2VFmt(b_,4), ",\n\t",
            str(dimension_size(t_.nodes,1)-1,0,0), " nodes, ",
            str(t_.nPts,0,0), " points.\n")
  #end
#end

#macro qtree_new_tree(optional bs_)
  #if (!defined(local.bs_))
    #local bs_ = 10;
  #end
  #if (qtree_debug)
    #debug concat("[II] new tree.  bucket size ", str(bs_,0,0), ".\n")
  #end
dictionary {
  .nodes:  array {},
  .root:   0,
  .maxPts: bs_,
  .nPts:   0
}
#end

#macro qtree_remove_point(t_, p_)
  #if (!t_.root)
    #error "oops, tree not \"root\"ed."
  #end
  #warning "removing points is To Be Implemented."
#end

#macro qtree_search_tree(t_, a_, b_, res_)
  #if (!t_.root)
    #error "oops, tree not \"root\"ed."
  #elseif (!qtree__is_rectangle(a_, b_))
    #error concat("oops, points ", qtree__2VFmt(a_,4), " and ",
            qtree__2VFmt(b_,4), " do not make rectangle.")
  #elseif (dimension_size(res_,1))
    #error "oops, search result array is not empty."
  #end
  #if (qtree__intersect(a_, b_, t_.nodes[t_.root][0], t_.nodes[t_.root][1]))
    #local rtn_ = qtree__searches_node(t_, t_.root, a_, b_, res_);
  #else
    #local rtn_ = -1;
  #end
  rtn_
#end

#version qtree_include_temp;

#end

/* --------------------------------------------------------------- *
  * the content above is covered by the GPLv3+, see file COPYING. * 
  * copyright (c) 2020 jr <creature.eternal@gmail.com>.           * 
  * all rights reserved.                                          * 
 * --------------------------------------------------------------- */
