/*Polar zonohedra models, created within POV 3.0,
by Russell Towle, June 1997. email: rtowle@inreach.com

Polar zonohedra are convex rhombic polyhedra determined by n vectors of equal magnitude. These vectors are identical to those connecting the apex to the base of a right regular n-gonal pyramid. Like such pyramids, polar zonohedra may be short and squashed, or tall and skinny. A polar zonohedron is symmetrical by a rotation of 360/n degrees around its symmetry axis, here coincident with the z axis.  This scene calls two "include" files and creates an ordinary polar zonohedron, with rhombic faces, but with spheres at each vertex and cylinders at each edge.

There are eight user-defined parameters in the procedure below:

1.  N, the number of vectors; N>=3. Default: N=10.
2.  PITCH, the angle between the vectors and the xy plane; 0<=PITCH<=90 degrees. Default: PITCH=35.2643.
3.  A, the edge length. Default: A=tan(pi/N).
4.  B, the spheres' radius. Default: A/5.
5.  C, the cylinders' radius. Default: A/10
6.  T1, texture for the spheres. Default: texture{Gold_Metal}.
7.  T2, texture for the cylinders. Default: texture{T_Silver_1A}.
8.  T3, texture for the rhombic faces. Default: texture{Copper_Metal}.

A polar zonohedron is the most spherical, and has the greatest volume for any given edge length, when PITCH = 35.2643+ degrees.  At that PITCH, polar zonohedra are isometric, orthogonal, solid shadows of N-cubes. The default scaling forces approximate unit equatorial radius; however, because the polar diameter varies according to PITCH, these forms may still stretch out of camera view when PITCH is high. Polar zonohedra become squashed and oblate as PITCH becomes lower (less than 35.2643 degrees) and become increasingly stretched out, spindle-like, and prolate as PITCH becomes higher (greater than 35.2643 degrees).

They are centered upon the z axis, with the sphere at the "south" pole just kissing the z=0 plane, whatever the PITCH. The camera and light statements will be found below the CYLLOHEDRON routine, since they use variables defined in that routine.*/

#version 3.0
global_settings { assumed_gamma 2.2 }
#include "colors.inc"
#include "shapes.inc"
#include "glass.inc"
#include "metals.inc"
#include "textures.inc"
#default {texture{pigment{color White}finish{phong 0.01 ambient 0.2 diffuse 0.6}}}

//The two most important user-defined parameters
#declare N = 5            //choose an integer >= 3
#declare PITCH = 35.2643  // 0<=pitch<=90 degrees

#declare A = tan(pi/N)    //scale factor; force equatorial radii to approx. 1
#declare B = A/5    //ball size; A/5 default
#declare C = A/10   //cylinder size; A/10 default

#declare T1 = texture{Gold_Metal}   //Gold_Metal good default
#declare T2 = texture{T_Silver_1A}  //T_Silver_1A good default
#declare T3 = texture{Copper_Metal} //Copper_Metal good default

#declare RISE = A * tan(radians(PITCH)) 
#declare NORTH = <0,0,(B/2)+(N*RISE)>  //put the north pole here
#declare SOUTH = <0,0,B/2> //put the south pole here

#include "PZ.inc"
#include "CYLLO.inc"

camera {
	location <.1, -5, 10>
	direction <0, 0, 3.75>
	up <0, 0, 1>
	right <4/3, 0, 0> //render to 4:3 width:height
	sky <0., 0., 1.>
	look_at <0, 0, (B/2)+((N/2)*RISE)> //look at center of CYLLOHEDRON
    rotate<0,12,0>    //a bit of tilt for art
}

light_source {<8, -12, 22> color rgb <1, 1, 1>}
light_source {<0, 0, 15> color rgb <1, 1, 1>}

object{CYLLOHEDRON}  //comment this line out to display only faces
object{POLAR_ZONOHEDRON}  //comment this line out to display only frame

plane{z, 0
 texture {
      pigment { colour rgb<.75,.25,1> }
      normal {
         ripples 1
         frequency 20
         scale 5
      }
      finish {
         ambient 0.2
         diffuse 0.6
      }
   }
}

