/* test_no_radiosity.pov
 * Persistence of Vision Raytracer scene description file
 *
 * Test whether an object with no_radiosity blocks the radiosity contribution
 * of surfaces behind it.
 *
 * Copyright © 2022 Richard Callwood III.  Some rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 2022-Jul-17  Abridged and adapted from model_occluded_light.pov.
 * 2022-Jul-18  The masking object is given a light gray pigment.
 */
// +KFF5 +W320 +H280 +A +R5
// +KFF5 +W336 +H294 +A +R5 +FJ Compression=90
#version 3.7;
/*
Frame   Light Bulb     Mask
-----   ----------     ----
  1     omitted        none
  2     no_radiosity   none
  3     no_radiosity   smaller
  4     no_radiosity   same size
  5     no_radiosity   larger
*/

#declare NCASES = 5;
#if (frame_number < 1 | frame_number > NCASES)
  #error concat ("Please render with +KFF", str (NCASES, 0, 0), ".")
#end

#declare RAD_COUNT = 120;
#declare RAD_WEIGHT = 500;

#include "screen.inc"

global_settings
{ assumed_gamma 1
  radiosity
  { count RAD_COUNT * RAD_WEIGHT, RAD_COUNT * RAD_WEIGHT * 101
    error_bound 0.5
    pretrace_start 16 / image_width
    pretrace_end 2 / image_width
    recursion_limit 4
  }
}
#default
{ finish { diffuse 1 }
  radiosity { importance 1 / RAD_WEIGHT }
}
box
{ -<200, 100, 200>, <200, 200, 200> hollow
  pigment { rgb 0.5 }
}

Set_Camera (<0, 55, -20>, <0, 0, -17>, 60)

//=================================== TEST =====================================

//----- Light numbers -----
#declare RBULB = 3;
#declare RDIST = 100 / sqrt (4 * pi);
#declare FADE = 0.3; // arbitrarily tiny
#declare LEVEL = 25;
#declare Brightness = LEVEL * (1 + pow (RDIST / FADE, 2)) / 2;
#declare Surface = Brightness * 2 / (1 + pow (RBULB / FADE, 2));

#declare BULB_WHITE = 0.75;

//----- Hood numbers -----

#declare INCH = 2.54;
#declare WHITE = 0.9;
#declare THIN = 0.08;
#declare RHOOD = 2.5 * INCH - THIN;
#declare DHOOD = INCH * 7;
#declare RBUTT = 1.25 * INCH;

//----- Area light numbers -----

#declare RES = 17;
#declare Res_adj = (RES - 1) / RES;
#declare dBulbA = 2 * RBULB * Res_adj;
#declare dHoodA = 2 * RHOOD * Res_adj;

//---------- Hood ----------------------

union
{ difference
  { union
    { sphere { 0, 1 scale <RHOOD, RHOOD, RHOOD * 2> + THIN }
      cylinder { RHOOD * 1.1 * z, DHOOD * z, RBUTT }
    }
    sphere
    { 0, RHOOD scale <1, 1, 2>
      pigment { rgb WHITE }
    }
    cylinder
    { RHOOD * 0.9 * z, (DHOOD - THIN) * z, RBUTT - THIN
      pigment { rgb WHITE }
    }
    plane { z, 0 }
    pigment { rgb <0.40, 0.28, 0.11> }
  }
  cylinder // socket
  { 8.6 * z, 2 * RHOOD * z, 1.6
    pigment { rgb 0.75 }
    double_illuminate
  }
  translate -RBULB * z
  radiosity { importance 1 }
}


//---------- Light ---------------------

#declare Light_bulb = sphere
{ 0, RBULB
  pigment { rgb BULB_WHITE }
  finish { emission Surface }
  no_shadow
  no_radiosity
}

#declare Mask = sphere
{ 0, RBULB
  pigment { rgb BULB_WHITE }
  no_shadow
  no_image
  radiosity { importance 1 } // still necessary to prevent artifacts
}

light_source
{ 0, rgb Brightness
  fade_power 2
  fade_distance FADE
  area_light dBulbA * x, dBulbA * y, RES, RES
  circular jitter adaptive 0 orient
}

#switch (frame_number)

  #case (1)
    #declare s_Bulb = "omitted"
    #declare s_Mask = "none"
    #break

  #case (2)
    #declare s_Bulb = "no_radiosity"
    #declare s_Mask = "none"
    object { Light_bulb }
    #break

  #case (3)
    #declare s_Bulb = "no_radiosity"
    #declare s_Mask = "smaller"
    object { Light_bulb }
    object { Mask scale 0.99 }
    #break

  #case (4)
    #declare s_Bulb = "no_radiosity"
    #declare s_Mask = "same size"
    object { Light_bulb }
    object { Mask }
    #break

  #case (5)
    #declare s_Bulb = "no_radiosity"
    #declare s_Mask = "larger"
    object { Light_bulb }
    object { Mask scale 1.01 }
    #break

#end

//=============================== ANNOTATIONS ==================================

#macro Text (S, Y, Rjust)
  #local Obj = text { ttf "cyrvetic" S 0.001, 0 }
  object { Obj translate <(Rjust? Rjust - max_extent (Obj).x: 0), Y * 1.2, 0> }
#end

#default { finish { ambient 0 diffuse 0 emission 1 } }

Screen_Object
( union
  { Text (concat ("Light bulb: ", s_Bulb), 0, 0)
    Text (concat ("Mask: ", s_Mask), -1, 0)
    pigment { green 1 }
    scale 15 / image_height
  },
  <0, 1>, <0.015, 0.015>, yes, 1
)
#declare o_Desc = Text ("Test: no_radiosity", 0, 0)
Screen_Object
( union
  { object { o_Desc }
    Text (concat ("v", str (version, 0, 2)), -1, max_extent (o_Desc).x)
    pigment { green 1 }
    scale 15 / image_height
  },
  <1, 1>, <0.015, 0.015>, yes, 1
)
