/**********************************************************************************
 Persistence of Vision Ray Tracer Scene Description File
 File name   : Orbital2_.pov
 Version     : 3.8
 Description : Banks Orbital, seen from its surface.
               - seasonal variation
               - day/night animation
               
 Date        : June 2019
 Author      : Thomas A. M. de Groot
 E-mail      : thomdeg.01@gmail.com
 
 Copyright (C) 2019. All rights reserved.
 This file is licensed under the terms of the CC-LGPL.
 Use and/or modification of this scene is free, except for commercial purposes.
 Commercial distribution is not allowed without written permission from the author.
**********************************************************************************/

// +w675 +h1000 +am2 +a0.3 +bm2 +bs8 +wt6

//--------------------------------------------------------------------------
#version 3.8;

//=================================================================================
// GLOBAL SETTINGS
//=================================================================================
global_settings {
  assumed_gamma 1.0
  radiosity {
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.004          // end pretrace at this size
    count 50, 1000                // higher -> higher quality (1..1600) [35]
    nearest_count 10, 5           // higher -> higher quality (1..10) [5]
    error_bound 1                 // higher -> smoother, less accurate [1.8]
    recursion_limit 2             // how much interreflections are calculated (1..5+) [2]
    low_error_factor .3           // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    maximum_reuse 0.1							// new parameter 3.7 [0.2]
    brightness 1                  // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
    normal on                   // take surface normals into account [off]
    media off                    // take media into account [off]
    always_sample off           // turn sampling in final trace off [on]
    //max_sample 1.0              // maximum brightness of samples
  }
}

/* conversion table --------------------------------------------------------
dimension conversion table (POV units <-> km):
0.1 = 1*1e3 = 1000 km
1   = 1*1e4 = 10000 km
10  = 1*1e5 = 100000 km
100 = 1*1e6 = 1000000 km
*/

// CHANGE THIS! ------------------------------------------------------------
#declare Season   = -20;   //Location of the Sun through the year [0 - 360]
#declare Tilt     = 5;    //tilt of Orbital with respect to Sun
#declare DayNight = on;  //day cycle animation on/off
#declare In       = on;  //camera inside or outside atmosphere

// camera ------------------------------------------------------------------
#if (In)
  #declare Camera_Location = < 0.10, 0.00, -149.960> ; //inside atmosphere
#else
  #declare Camera_Location = < 0.10, 0.00, -149.800> ; //outside atmosphere
#end
#declare Camera_Look_At  = < 0.00, -180.00, 0.00> ;
#declare Camera_Angle    =  75 ;

camera {
  location Camera_Location
  right    x*image_width/image_height
  angle    Camera_Angle
  look_at  Camera_Look_At
  rotate   -90*z
}

// sky ---------------------------------------------------------------------
#if (DayNight)
  #declare Season_rotation   = <0, Season, Tilt>; //position of the Sun through the seasons;
  #declare Universe_rotation = <0, 360*clock, 0>; //animation through a full day
#else
  #declare Season_rotation   = <0, Season, Tilt>;
  #declare Universe_rotation = <0, 292, 0>;
#end

// Coordinate system -------------------------------------------------------
#local Coord =
torus {
  1.0, 0.0015
  scale 2e4
}

#declare Coord_sys =
union {
  //ecliptic:
  object {Coord}
  //celestial longitudes:
  #for (I, 0, 170, 30)
    object {Coord
      rotate 90*z
      rotate I*y
    }
  #end
  texture {
    pigment {color rgb 0.1}
    finish {diffuse 1 emission 0.025} 
  }
  no_shadow
}

light_group {
  light_source {
    <0, 0, 0>
    color <1, 1, 1>*2
  }
  object {Coord_sys}
  object {Coord //Sun seasonal path
    texture {
      pigment {color rgb <0.4, 0.4, 0.0>}
      finish {diffuse 1 emission <0.025, 0.025, 0.0>} 
    }
    no_shadow
    rotate Season_rotation //seasonal position of the Sun
  }
  rotate Universe_rotation  //rotation through a full day
}

// stars (Chris Colefax) ---------------------------------------------------
#declare galaxy_seed = 1234567890;
#declare galaxy_bg = off;
#declare galaxy_objects = off;
#declare galaxy_rotate = Universe_rotation;
#declare star_count = 5000;
#declare star_scale = 0.25;
#declare star_distance = 1*1e5;
#declare star_distribution = 0.5;
#include "Galaxy.inc"

// sun -------------------------------------------------------------------
//Sun dimensions:
//radius : 695700 km
//distance : 149597870.700 km

#include "CIE.inc"
#declare SunCol = Blackbody(6500)*20; //Sun temperature: 2800=red; 7500=blue white.
#local Aspect   = sphere {<0, 0, 0>, 69.57*2 pigment {rgb <0.5, 0.5, 0.0>} finish {emission 20}}

#local Sun =
light_source {
  <0, 0, 0>
  color SunCol
  looks_like {Aspect}
  area_light
  69.57*x, 69.57*z, // lights spread out across this distance (x * z)
  9, 9                // total number of lights in grid (4x*4z = 16 lights)
  adaptive 1          // 0,1,2,3...
  area_illumination on  // full area diffuse and specular
  //jitter              // adds random softening of light (switched off for animation)
  circular            // make the shape of the light circular
  orient              // orient light
  translate -14959.78707*z
  parallel
  point_at <0, 0, 0>
}

//seasonal rotation:
object {Sun rotate Season_rotation rotate Universe_rotation}  

//--------------------------------------------------------------------------
//---------------------------- objects in scene ----------------------------
//--------------------------------------------------------------------------
//The Banks Orbital (as described in the Iain M. Banks books)
//Typical dimensions:
//diameter    : 3*1e6 km
//width       : 6*1e3 km
//wall height : 1*1e3 km

#local T_landscape =
texture {
  pigment {
    granite
    pigment_map {
      [0.36 srgb <0, 0, 0.8>]
      [0.36 srgb <0.6, 0.6, 0.0>]
      [0.40 srgb <0, 0.8, 0.1>]
      [0.50 srgb <0.1, 0.4, 0.1>]
      [0.70 srgb <0.1, 0.4, 0.1>]
      [0.90 srgb <0, 0.8, 0.1>]
      [0.94 srgb <0.6, 0.6, 0.0>]
      [0.94 srgb <0, 0, 0.8>]
    }
  }
  normal {
    granite
    normal_map {
      [0.36 granite 0.0 scale 0.1]
      [0.36 granite 0.1 scale 0.1]
      [0.40 granite 0.4 scale 0.1]
      [0.50 granite 0.6 scale 0.1]
      [0.70 granite 0.6 scale 0.1]
      [0.90 granite 0.4 scale 0.1]
      [0.94 granite 0.1 scale 0.1]
      [0.94 granite 0.0 scale 0.1]
    }
  }
  finish {diffuse 0.7 emission 0.05}
}

#local T_orbital =
texture {
  pigment {rgb 0.25}
  normal {granite 0.1 scale 0.1}
  finish {diffuse 0.7 specular 0.01 roughness 0.001 emission 0.05}
}

#declare Orbital =
union {
  //the Orbital body:
  difference {
    cylinder {<0, -0.300, 0>, <0, 0.300, 0>, 150.000}  //outside width and diameter
    cylinder {<0, -0.400, 0>, <0, 0.400, 0>, 149.900}  //walls height
    cylinder {<0, -0.290, 0>, <0, 0.290, 0>, 149.980} //walls sides and inside width
    texture {T_orbital}
  }
  //the Orbital landscape:
  difference {
    cylinder {<0, -0.290, 0>, <0, 0.290, 0>, 149.985}
    cylinder {<0, -0.291, 0>, <0, 0.291, 0>, 149.975}
    texture {T_landscape}
  }
  //the Orbital atmosphere:
  difference {
    cylinder {<0, -0.288, 0>, <0, 0.288, 0>, 149.981}
    cylinder {<0, -0.289, 0>, <0, 0.289, 0>, 149.900}
    hollow
    pigment {rgbt 1}
    interior {
      media {
        samples 1
        scattering {3, rgb <0.5, 0.5, 0.9>*0.05}
        density {
          cylindrical
          colour_map {
            [0.0 rgb <0.5, 0.5, 1.0>*0.5]
            [0.5 rgb <0.5, 0.5, 1.0>*1.0]
          }
          rotate 90*x
        }
      }
    }
  }
  //an object on the Orbital floor:
  cone {
    <0, -0.001, 0>, 0.025, <0, 0.15, 0.0>, 0.005
    rotate 90*x
    translate 0.15*y
    translate -149.980*z
    rotate 0.5*y
    texture {
      pigment {srgb <0.8, 0.05, 0.01>}
      normal {granite 1 scale 0.01}
      finish {diffuse 0.7 emission 0.05}
    }
  }
}

Orbital

