
/**********************************************************************************
 Persistence of Vision Ray Tracer Scene Description File
 File name   : Geomorph_mcr22.pov
 Version     : 3.6
 Description : Version 2.2 of a macro for generating isosurface landscapes, with 
               special attention to islands and single rocks
               
 Date        : December 2004 - January 2005
 Author      : Thomas A. M. de Groot
 E-mail      : 
 
 Copyright (C) 2004-2005. All rights reserved.
**********************************************************************************/

#include "rand.inc"
#include "functions.inc"
#include "Thomas de Groot\TdG_Textures.inc"

default {
  texture {
    pigment { rgb <1,0,0> }
  }
}

// Run the macro as standalone. Else, in your scene, declare Standalone as false before including and calling the macro.
#ifndef (Standalone)     #declare Standalone = true;          #end

#if (Standalone)
global_settings {
  assumed_gamma 1.0
  max_intersections 64
  max_trace_level 5
  noise_generator 2
}
#end


// -------- Terrain textures (by Christoph Hormann) --------
#declare T_Sand=
texture {
  pigment { color rgb <1.1, 0.7, 0.3> }
  finish { 
    specular 0.06 
    ambient <0.8, 0.9, 1.4>*0.1
  }
  normal {
    granite 0.3
    scale 0.1
  }
}

#declare T_Grass=
texture {
  pigment { color rgb <0.5, 1.15, 0.3> }
  finish {
    specular 0.1
    diffuse 0.3
    brilliance 1.6
    ambient <0.8, 0.9, 1.4>*0.03
  }
  normal {
    granite 0.5
    accuracy 0.01
    scale 0.12
  }
}

#declare T_Rock=
texture {
  pigment {
    agate
    color_map {
      [0.2 color rgb <0.55, 0.50, 0.50> ]
      [0.6 color rgb <0.75, 0.50, 0.60> ]
      [1.0 color rgb <0.70, 0.60, 0.60> ]
    }
    scale 0.2
    warp { turbulence 0.5 }
  }
  finish {
    specular 0.2
    diffuse 0.4
    ambient <0.8, 0.9, 1.4>*0.06
  }
  normal {
    granite 0.6
    scale 0.1
  }
}

//======================================================================================================================================
#macro Geomorph() // start of the Geomorph macro

// (0) Declared landscape parameters:
// Substrate:
#ifndef (HeteroInf)			#local HeteroInf = true;						#end
#ifndef (HetH)					#local HetH = 1;										#end
#ifndef (HetLac)				#local HetLac = 1;									#end
#ifndef (HetOct)				#local HetOct = 1;									#end
#ifndef (HetOff)				#local HetOff = 0;									#end
#ifndef (HetT)					#local HetT = 0.2;									#end
#ifndef (HetNG)					#local HetNG = 2;										#end

#ifndef (Noised)				#local Noised = true;								#end

#ifndef (Ridged)				#local Ridged = true;								#end
#ifndef (RidgedInf)			#local RidgedInf = true;						#end
#ifndef (RidH)					#local RidH = 1.2;									#end
#ifndef (RidLac)				#local RidLac = 3;									#end
#ifndef (RidOct)				#local RidOct = 6;									#end
#ifndef (RidOff)				#local RidOff = 1.15;								#end
#ifndef (RidG)					#local RidG = 5;										#end
#ifndef (RidNG)					#local RidNG = 2;										#end

#ifndef (Fractal)				#local Fractal = false;							#end
#ifndef (Mandelbrot)		#local Mandelbrot = true;						#end
#ifndef (Complex)				#local Complex = <0.353, 0.288>;		#end
#ifndef (Iterations)		#local Iterations = 5;							#end
#ifndef (Exponent)			#local Exponent = 3;								#end
#ifndef (Interior)			#local Interior = 0;								#end
#ifndef (InFactor)			#local InFactor = 1.0;							#end
#ifndef (Exterior)			#local Exterior = 1;								#end
#ifndef (ExFactor)			#local ExFactor = 1.0;							#end

// noise:
#ifndef (Noise)					#local Noise = 0.5;									#end		// controller for the f_noise3d function

// patterns:
#ifndef (Wrinkled)			#local Wrinkled = true;							#end
#ifndef (Rippled)				#local Rippled = false;							#end
#ifndef (Waved)					#local Waved = false;								#end
#ifndef (Dented)				#local Dented = true;								#end
#ifdef (Sphered)				#local Isolated = Sphered;					#end		// This is for backward compatibility with version 1
#ifndef (Isolated)			#local Isolated = true;							#end		// when true, generates an island or an isolated rock
#ifndef (Isolator)			#local Isolator = 2;								#end		// a selector (1-4) that changes the shape of the island
#ifndef (Crackled)			#local Crackled = false;						#end
#ifndef (Hexed)					#local Hexed = false;								#end
#ifndef (Honeycomb)			#local Honeycomb = false;						#end

// warp:
#ifndef (Wrinkleswarp)	#local Wrinkleswarp = false;				#end
#ifndef (Rippleswarp)		#local Rippleswarp = false;					#end
#ifndef (Waveswarp)			#local Waveswarp = false;						#end
#ifndef (Dentswarp)			#local Dentswarp = false;						#end
#ifndef (Cracklewarp)		#local Cracklewarp = true;					#end

// gain:
#ifdef (Substrategain)	#local Subyscale = Substrategain;		#end		// This is for backward compatibility with version 2.1  
#ifdef (Ridgegain)			#local Ridgeyscale = Ridgegain;			#end		// This is for backward compatibility with version 2.1  
#ifndef (Wrinklesgain)	#local Wrinklesgain = 0.5;					#end  
#ifndef (Ripplesgain)		#local Ripplesgain = 0.8;	  				#end 
#ifndef (Wavesgain)			#local Wavesgain = 0.8;	  					#end 
#ifndef (Dentsgain)			#local Dentsgain = 0.5;	  					#end
#ifndef (Cracklegain)		#local Cracklegain = 2.0;	  				#end 
#ifndef (Hexgain)				#local Hexgain = 0.9;	  						#end 

// warp turbulence:
#ifndef (Wrinklesturb)	#local Wrinklesturb = <0.5, 0.9, 0.9>;		#end
#ifndef (Ripplesturb)		#local Ripplesturb = <0.5, 0.9, 0.9>;			#end
#ifndef (Wavesturb)			#local Wavesturb = <0.5, 0.9, 0.9>;				#end
#ifndef (Dentsturb)			#local Dentsturb = <0.5, 0.9, 0.9>;				#end
#ifndef (Crackleturb)		#local Crackleturb = <0.5, 0.9, 0.9>;			#end

// scale:
#ifndef (Substratescale)	#local Substratescale = <1.0, 1.0, 1.0>;	#end
#ifndef (Ridgescale)			#local Ridgescale = <1.0, 1.0, 1.0>;			#end
#ifndef (Fractalscale)		#local Fractalscale = <1, 1, 1>;					#end

#ifndef (Wrinklescale)		#local Wrinklescale = <1.5, 0.2, 1.5>;		#end
#ifndef (Ripplescale)			#local Ripplescale = <0.5, 0.2, 0.5>;			#end
#ifndef (Wavescale)				#local Wavescale = <0.5, 0.2, 0.5>;				#end
#ifndef (Dentscale)				#local Dentscale = <0.5, 0.2, 0.5>;				#end
#ifndef (Cracklescale)		#local Cracklescale = <0.5, 0.2, 0.5>;		#end
#ifndef (Cracksize)				#local Cracksize = 2;											#end		// multiplier for Cracklescale
#ifndef (Hexscale)				#local Hexscale = <0.2, 0.4, 0.2>;				#end

// angle of rotation:
#ifndef (Fractalangle)	#local Fractalangle = <0, 0, 0>;		#end

#ifndef (Wrinklesangle)	#local Wrinklesangle = <0, 0, -20>; #end
#ifndef (Ripplesangle)	#local Ripplesangle = <0, 0, -25>;	#end
#ifndef (Wavesangle)		#local Wavesangle = <0, 0, -5>;			#end
#ifndef (Dentsangle)		#local Dentsangle = <0, 0, -10>;		#end
#ifndef (Crackleangle)	#local Crackleangle = <0, 0, 0>;		#end
#ifndef (Hexangle)			#local Hexangle = <0, 0, 0>;				#end

// fractal translation:
#ifndef (Fractaltrans)	#local Fractaltrans = <0, 0, 0>;		#end

// island:
#ifdef (SphereShearX)		#local IslandShearX = SphereShearX;	#end		// This is for backward compatibility with version 1
#ifndef (IslandShearX)	#local IslandShearX = 15;						#end		// shear of the island function in the x direction
#ifdef (SphereShearZ)		#local IslandShearZ = SphereShearZ;	#end		// This is for backward compatibility with version 1
#ifndef (IslandShearZ)	#local IslandShearZ = 10;						#end		// shear of the island function in the z direction
#ifdef (Spheregain)			#local Islandgain = Spheregain;			#end		// This is for backward compatibility with version 1
#ifndef (Islandgain)		#local Islandgain = 0.5;	  				#end
#ifndef (Diameter)			#local Diameter = 1;								#end		// diameter of the island function

// isosurface:
#ifdef (FinalFunction)	#undef FinalFunction								#end
#ifndef (Evaluate)			#local Evaluate = false;						#end
#ifndef (Min_factor)		#local Min_factor = 0.6;						#end
#ifndef (Attenuation)		#local Attenuation = 0.7;						#end
#ifndef (UpperBound)		#local UpperBound = 3;							#end		// Bounding box upper limit
#ifndef (LowerBound)		#local LowerBound = -2;							#end		// Bounding box lower limit
#ifndef (MaxGrad)				#local MaxGrad = 10;								#end		// max_gradient of isosurface (change according to need)
#ifndef (Xsize)					#local Xsize = 10;									#end		// increases size of the bounding box (not scale!) for X
#ifndef (Zsize)					#local Zsize = 10;									#end		// increases size of the bounding box (not scale!) for Z
#ifndef (Isoscale)			#local Isoscale = <3, 3, 3>;				#end		// scale of the isosurface
#ifndef (Isorot)				#local Isorot = <0, 0, 0>;					#end		// rotation of the isosurface
#ifndef (Isotrans)			#local Isotrans = <0, 0, 0>;				#end		// translation of the isosurface

// texture:
#ifndef (Simpletex)			#local Simpletex = false;						#end		// just a choice between two nice textures
#ifndef (Slope)					#local Slope = true;								#end		// true = slope & altitude; false = gradient texture
#ifndef (SlopeValue)		#local SlopeValue = 0.6;						#end		// (0-1) higher values: more slope, less altitude, and vice versa

// internal default textures of the macro:
#ifndef (Landtex)
#declare Landtex =
#if (Simpletex)
  texture {
    pigment {
      #if (Slope)
        slope {<0.0, SlopeValue, 0.0> , 0.0 , 0.5
          altitude <0.0, 1-SlopeValue, 0.0> , LowerBound , UpperBound  // bounding box goes from LowerBound to UpperBound
        }
      #else
        gradient y
      #end
      colour_map {
        [0.00 rgb <0.1,0.4,0.1>]
        [0.25 rgb <0.5,0.4,0.3>]
        [0.50 rgb 0.50]
        [0.75 rgb 0.75]
        [1.00 rgb 0.90]
      }
    }
  }
#else
  texture {
    #if (Slope)
      slope {<0.0, SlopeValue, 0.0> , 0.0 , 0.5
        altitude <0.0, 1-SlopeValue, 0.0> , LowerBound , UpperBound  // bounding box goes from LowerBound to UpperBound
      }
    #else
      gradient y
    #end
    texture_map {
      [0.00 T_Rock]
      [0.25 T_Grass]
      [0.50 T_Sand]
      [0.75 T_Sand]
      [1.00 T_Rock]
    }
  }
#end
#end // end of ndef

//------------------------
// (1) Basic pattern functions:
#if (Wrinkleswarp)
  #local Wrinkles=function {pattern {wrinkles warp {turbulence Wrinklesturb octaves 3 omega 0.5 lambda 1} scale Wrinklescale }}
#else
  #local Wrinkles=function {pattern {wrinkles scale Wrinklescale }}
#end

#if (Rippleswarp)
  #local Ripples=function {pattern {ripples frequency 0.2 warp {turbulence Ripplesturb octaves 3 omega 0.5 lambda 1} scale Ripplescale }}
#else
  #local Ripples=function {pattern {ripples frequency 0.2 scale Ripplescale }}
#end

#if (Waveswarp)
  #local Waves=function {pattern {waves poly_wave 1 warp {turbulence Wavesturb octaves 3 omega 0.5 lambda 1} scale Wavescale }}
#else
  #local Waves=function {pattern {waves poly_wave 0.5 scale Wavescale }}
#end

#if (Dentswarp)
  #local Dents=function {pattern {dents warp {turbulence Dentsturb octaves 5 omega 0.5 lambda 2} scale Dentscale }}
#else
  #local Dents=function {pattern {dents scale Dentscale }}
#end

// the crackle function generates vertical faults and fault faces in the rocks.
#if (Cracklewarp)
  #local Crackle=function {pattern {crackle solid warp {turbulence Crackleturb octaves 5 omega 0.5 lambda 2} scale Cracklescale*Cracksize }}
#else
  #local Crackle=function {pattern {crackle solid scale Cracklescale*Cracksize }}
#end

//------------------------
// (2) 'landscape'functions:
#local Fractalscalex = Fractalscale.x;
#local Fractalscaley = Fractalscale.y;	// this one goes to final P function
#local Fractalscalez = Fractalscale.z;
#local Fractaltransx = Fractaltrans.x;
#local Fractaltransy = Fractaltrans.y;
#local Fractaltransz = Fractaltrans.z;
#local Mand = function {
                  pigment{ 
                    mandel Iterations
                    exponent Exponent
                    exterior Exterior, ExFactor
                    interior Interior, InFactor
                    color_map {
                      [0 rgb 0] 
                      [1 rgb 1]  
                    } 
                    scale <Fractalscalex, Fractalscalez, 1> 
                  } 
                }
#local Mandel = function {Mand(x,y,z).grey*Fractalscaley}

#local Jul = function {
                 pigment {
                   julia 	Complex, Iterations
                   exponent Exponent
                   exterior Exterior, ExFactor
                   interior Interior, InFactor
                   color_map { 
                     [0 rgb 0] 
                     [1 rgb 1]  
                   }
                   scale <Fractalscalex, Fractalscalez, 1> 
                 }
               }
#local Julia = function {Jul(x,y,z).grey*Fractalscaley}

#local Subxscale = Substratescale.x;
#local Subyscale = Substratescale.y; // see final function P
#local Subzscale = Substratescale.z;
#if (HeteroInf)
  #local Hetero = function {f_hetero_mf(x/Subxscale, 0, z/Subzscale, HetH, HetLac, HetOct, HetOff, HetT, HetNG ) } 
#else
  #local Hetero = function {f_hetero_mf(x/Subxscale, y, z/Subzscale, HetH, HetLac, HetOct, HetOff, HetT, HetNG ) } 
#end

#local Ridgexscale = Ridgescale.x;
#local Ridgeyscale = Ridgescale.y; // see final function P
#local Ridgezscale = Ridgescale.z;
#if (RidgedInf)
  #local Ridge = function {f_ridged_mf(x/Ridgexscale, 0, z/Ridgezscale, RidH, RidLac, RidOct, RidOff, RidG, RidNG ) } 
#else
  #local Ridge = function {f_ridged_mf(x/Ridgexscale, y, z/Ridgezscale, RidH, RidLac, RidOct, RidOff, RidG, RidNG ) } 
#end

#local Hexxscale = Hexscale.x;
#local Hexyscale = Hexscale.y;
#local Hexzscale = Hexscale.z;
#local Hex = function {f_hex_x(x/Hexxscale+abs(sin(x-y)), y/Hexyscale-abs(sin(x-y)), z/Hexzscale+abs(sin(x-y)), 0)}

#switch (Isolator) // this defines the "island".
#case (1)
#local Island = function {f_sphere(x,y,z,Diameter)}
#break
#case (2)
#local Island = function {f_sphere(x,y,z,Diameter-abs(cos(x+y))-abs(cos(z+y)))}
#break
#case (3)
#local Size = 1/(Diameter*10);
#local Island = function {f_quartic_paraboloid( x,0,z,-Size/( abs(cos(x+y))+abs(cos(z+y)) ) )}
#break
#case (4)
#local Size = 1/(Diameter*10);
#local Island = function {f_quartic_paraboloid( x,0,z,-Size/( abs(cos(x/z))+abs(cos(x/z)) ) )}
#break
#case (5)
#local Strength = 0.1/(Diameter*10);
#local Island = function {f_bifolia(x,0,z,-Strength,1)} 
#end

//------------------------
// (3) first, rotation around x:
#local Fractalxangle = Fractalangle.x;
#local Fractalyangle = Fractalangle.z; // Switched y and z for correct orientation!
#local Fractalzangle = Fractalangle.y;
#if (Mandelbrot)
  #local Mandelrotx=function {Mandel(x, z*sin(radians(Fractalxangle)) + y*cos(radians(Fractalxangle)), z*cos(radians(Fractalxangle)) - y*sin(radians(Fractalxangle))) }
#else
  #local Juliarotx=function {Julia(x, z*sin(radians(Fractalxangle)) + y*cos(radians(Fractalxangle)), z*cos(radians(Fractalxangle)) - y*sin(radians(Fractalxangle))) }
#end

#local Wrinklesxangle = Wrinklesangle.x;
#local Wrinklesyangle = Wrinklesangle.y;
#local Wrinkleszangle = Wrinklesangle.z;
#local Wrinklesrotx=function {Wrinkles(x, z*sin(radians(Wrinklesxangle)) + y*cos(radians(Wrinklesxangle)), z*cos(radians(Wrinklesxangle)) - y*sin(radians(Wrinklesxangle))) }

#local Ripplesxangle = Ripplesangle.x;
#local Ripplesyangle = Ripplesangle.y;
#local Rippleszangle = Ripplesangle.z;
#local Ripplesrotx=function {Ripples(x, z*sin(radians(Ripplesxangle)) + y*cos(radians(Ripplesxangle)), z*cos(radians(Ripplesxangle)) - y*sin(radians(Ripplesxangle))) }

#local Wavesxangle = Wavesangle.x;
#local Wavesyangle = Wavesangle.y;
#local Waveszangle = Wavesangle.z;
#local Wavesrotx=function {Waves(x, z*sin(radians(Wavesxangle)) + y*cos(radians(Wavesxangle)), z*cos(radians(Wavesxangle)) - y*sin(radians(Wavesxangle))) }

#local Dentsxangle = Dentsangle.x;
#local Dentsyangle = Dentsangle.y;
#local Dentszangle = Dentsangle.z;
#local Dentsrotx=function {Dents(x, z*sin(radians(Dentsxangle)) + y*cos(radians(Dentsxangle)), z*cos(radians(Dentsxangle)) - y*sin(radians(Dentsxangle))) }

#local Cracklexangle = Crackleangle.x;
#local Crackleyangle = Crackleangle.y;
#local Cracklezangle = Crackleangle.z;
#local Cracklerotx=function {Crackle(x, z*sin(radians(Cracklexangle)) + y*cos(radians(Cracklexangle)), z*cos(radians(Cracklexangle)) - y*sin(radians(Cracklexangle))) }

#local Hexxangle = Hexangle.x;
#local Hexyangle = Hexangle.y;
#local Hexzangle = Hexangle.z;
#local Hexrotx=function {Hex(x, z*sin(radians(Hexxangle)) + y*cos(radians(Hexxangle)), z*cos(radians(Hexxangle)) - y*sin(radians(Hexxangle))) }

//------------------------
// (4) then, rotation around z:
#if (Mandelbrot)
  #local Mandelrotz=function {Mandelrotx(x*cos(radians(Fractalzangle)) + y*sin(radians(Fractalzangle)), -x*sin(radians(Fractalzangle)) + y*cos(radians(Fractalzangle)), z) }
#else
  #local Juliarotz=function {Juliarotx(x*cos(radians(Fractalzangle)) + y*sin(radians(Fractalzangle)), -x*sin(radians(Fractalzangle)) + y*cos(radians(Fractalzangle)), z) }
#end
#local Wrinklesrotz=function {Wrinklesrotx(x*cos(radians(Wrinkleszangle)) + y*sin(radians(Wrinkleszangle)), -x*sin(radians(Wrinkleszangle)) + y*cos(radians(Wrinkleszangle)), z) }
#local Ripplesrotz=function {Ripplesrotx(x*cos(radians(Rippleszangle)) + y*sin(radians(Rippleszangle)), -x*sin(radians(Rippleszangle)) + y*cos(radians(Rippleszangle)), z) }
#local Wavesrotz=function {Wavesrotx(x*cos(radians(Waveszangle)) + y*sin(radians(Waveszangle)), -x*sin(radians(Waveszangle)) + y*cos(radians(Waveszangle)), z) }
#local Dentsrotz=function {Dentsrotx(x*cos(radians(Dentszangle)) + y*sin(radians(Dentszangle)), -x*sin(radians(Dentszangle)) + y*cos(radians(Dentszangle)), z) }
#local Cracklerotz=function {Cracklerotx(x*cos(radians(Cracklezangle)) + y*sin(radians(Cracklezangle)), -x*sin(radians(Cracklezangle)) + y*cos(radians(Cracklezangle)), z) }
#local Hexrotz=function {Hexrotx(x*cos(radians(Hexzangle)) + y*sin(radians(Hexzangle)), -x*sin(radians(Hexzangle)) + y*cos(radians(Hexzangle)), z) }

//------------------------
// (5) finally, rotation around y:
#if (Mandelbrot)
  #local Mandelroty=function {Mandelrotz(x*cos(radians(Fractalyangle)) - z*sin(radians(Fractalyangle)), y, x*sin(radians(Fractalyangle)) + z*cos(radians(Fractalyangle))) }
#else
  #local Juliaroty=function {Juliarotz(x*cos(radians(Fractalyangle)) - z*sin(radians(Fractalyangle)), y, x*sin(radians(Fractalyangle)) + z*cos(radians(Fractalyangle))) }
#end
#local Wrinklesroty=function {Wrinklesrotz(x*cos(radians(Wrinklesyangle)) - z*sin(radians(Wrinklesyangle)), y, x*sin(radians(Wrinklesyangle)) + z*cos(radians(Wrinklesyangle))) }
#local Ripplesroty=function {Ripplesrotz(x*cos(radians(Ripplesyangle)) - z*sin(radians(Ripplesyangle)), y, x*sin(radians(Ripplesyangle)) + z*cos(radians(Ripplesyangle))) }
#local Wavesroty=function {Wavesrotz(x*cos(radians(Wavesyangle)) - z*sin(radians(Wavesyangle)), y, x*sin(radians(Wavesyangle)) + z*cos(radians(Wavesyangle))) }
#local Dentsroty=function {Dentsrotz(x*cos(radians(Dentsyangle)) - z*sin(radians(Dentsyangle)), y, x*sin(radians(Dentsyangle)) + z*cos(radians(Dentsyangle))) }
#local Crackleroty=function {Cracklerotz(x*cos(radians(Crackleyangle)) - z*sin(radians(Crackleyangle)), y, x*sin(radians(Crackleyangle)) + z*cos(radians(Crackleyangle))) }
#local Hexroty=function {Hexrotz(x*cos(radians(Hexyangle)) - z*sin(radians(Hexyangle)), y, x*sin(radians(Hexyangle)) + z*cos(radians(Hexyangle))) }

//------------------------
// (6) Landscape pattern:
#local P=function {  Hetero(x, y, z)*Subyscale
                     #if (Noised)
										   + f_noise3d(x, y, z)*Noise
										 #end
										 #if (Ridged)
										   + Ridge(x, y, z)*Ridgeyscale
										 #end
										 #if (Fractal)
                       #if (Mandelbrot)
                         + Mandelroty(x-(0.5+Fractaltransx), z-Fractaltransz, y-(0.5+Fractaltransy))	// y and z are inverted to get the fractal correctly oriented!
                       #else																																					// 0.5 to compensate for isosurface centering on origin 
                         + Juliaroty(x-(0.5+Fractaltransx), z-Fractaltransz, y-(0.5+Fractaltransy))		// y and z are inverted to get the fractal correctly oriented!
                       #end
                     #end
										 #if (Wrinkled)
                       + Wrinklesroty(x, y, z)*Wrinklesgain
                     #end
                     #if (Rippled)
                       + Ripplesroty(x, y, z)*Ripplesgain
                     #end
                     #if (Waved)
                       + Wavesroty(x, y, z)*Wavesgain
                     #end
                     #if (Dented)
										   + Dentsroty(x, y, z)*Dentsgain
										 #end
                     #if (Crackled)
                       * Crackleroty(x, 0, z)*Cracklegain
                     #end
                     #if (Hexed)
                       #if (Honeycomb)
                         * Hexroty(x, z, y)*Hexgain			// y and z are inverted to get the hexagones vertical!
                       #else
                         * -Hexroty(x, z, y)*Hexgain		// y and z are inverted to get the hexagones vertical!
                       #end
                     #end
										 #if (Isolated)
                       - Island(x+y*tan(radians(IslandShearX)), y, z+y*tan(radians(IslandShearZ)))*Islandgain
                     #end
                    }

// For any external use of this function (for instance: use in foaming water borders):
#declare FinalFunction = function {P(x,y,z)}

//------------------------
// (7) Isosurface:
#declare Landscape =
isosurface {
  function {y - P(x, y, z)}
  //open  
  #if (Evaluate)
    evaluate MaxGrad*Min_factor, sqrt(MaxGrad/(MaxGrad*Min_factor)), Attenuation
  #else
    max_gradient MaxGrad
  #end
  accuracy 0.001
  max_trace 1
  //all_intersections 
  contained_by {box {<0.5-Xsize*0.75, LowerBound, 0.5-Zsize*0.75>, <0.5+Xsize*0.75, UpperBound, 0.5+Zsize*0.75>}}    
  translate <-0.5, 0, -0.5>
  texture {Landtex}
  scale Isoscale
  rotate Isorot
  translate Isotrans
}

//object {Landscape}


#end // end of the Geomorph macro

//======================================================================================================================================

#if (Standalone)
// Before calling the macro, use or change any of these parameters (defaults are defined inside the macro).
// (0) Declared landscape parameters:
// substrate:
#declare HeteroInf = true;  // in function, when true: y scaled infinitely 
#declare HetH = 1;
#declare HetLac = 1;
#declare HetOct = 1;
#declare HetOff = 0;
#declare HetT = 0.2;
#declare HetNG = 2;

#declare Noised = false;

#declare Ridged = false;
#declare RidgedInf = false;  // in function, when true: y scaled infinitely 
#declare RidH = 1.2;
#declare RidLac = 3;
#declare RidOct = 6;
#declare RidOff = 1.15;
#declare RidG = 5;
#declare RidNG = 2;

#declare Fractal = false;
#declare Mandelbrot = true;
#declare Complex = <0.1, 0.2>;
#declare Iterations = 10;
#declare Exponent = 3;
#declare Exterior = 1;
#declare ExFactor = 1.0;
#declare Interior = 0;
#declare InFactor = 1.0;

// noise:
#declare Noise = 0.4;										// controller for the f_noise3d function

// patterns:
#declare Wrinkled = true;
#declare Rippled = false;
#declare Waved = false;
#declare Dented = false;
#declare Isolated = false;									// when true, generates an island or an isolated rock
#declare Isolator = 3;
#declare Crackled = false;
#declare Hexed = true;
#declare Honeycomb = false;

// warp:
#declare Wrinkleswarp = true;
#declare Rippleswarp = true;
#declare Waveswarp = false;
#declare Dentswarp = true;
#declare Cracklewarp = false;

// gain:
#declare Wrinklesgain = 0.5;  
#declare Ripplesgain = 0.2;  
#declare Wavesgain = 0.5;  
#declare Dentsgain = 0.5;  
#declare Cracklegain = 1.0;
#declare Hexgain = 1.0; 

// warp turbulence:
#declare Wrinklesturb = <0.5, 0.9, 0.9>;
#declare Ripplesturb = <0.5, 2, 0.9>;
#declare Wavesturb = <0.5, 0.9, 0.9>;
#declare Dentsturb = <0.5, 0.9, 0.9>;
#declare Crackleturb = <0.5, 0.9, 0.9>;

// scale:
#declare Substratescale = <5.0, 2, 5.0>;
#declare Ridgescale = <0.5, 1.0, 0.5>;
#declare Fractalscale = <3, -1.5, 3>;

#declare Wrinklescale = <1.0, 0.5, 1.0>;
#declare Ripplescale = <0.1, 0.2, 0.1>;
#declare Wavescale = <0.5, 0.2, 0.5>;
#declare Dentscale = <0.5, 0.2, 0.5>;
#declare Cracklescale = <0.5, 0.2, 0.5>;
#declare Cracksize = 1;										// multiplier for Cracklescale
#declare Hexscale = <0.5, 1, 0.5>;

// angle of rotation:
#declare Fractalangle = <0, 0, 0>;

#declare Wrinklesangle = <0, 0, 60>;
#declare Ripplesangle = <0, 0, -30>;
#declare Wavesangle = <0, 0, 0>;
#declare Dentsangle = <0, 0, -10>;
#declare Crackleangle = <0, 0, 0>;
#declare Hexangle = <0, 0, 0>;

// fractal translation:
#declare Fractaltrans = <0, 0, 0>;

// island:
#declare IslandShearX = 0;							// shear of the island function in the x direction
#declare IslandShearZ = 0;							// shear of the island function in the z direction
#declare Islandgain = 0.5;  
#declare Diameter = 1;									// diameter of the island function

// isosurface:
#declare UpperBound = 2;								// Bounding box upper limit
#declare LowerBound = -3;								// Bounding box lower limit
#declare MaxGrad = 11;									// max_gradient of the isosurface (change according to need)
#declare Xsize = 100;										// increase size of the bounding box (not scale!) for X
#declare Zsize = 100;										// increase size of the bounding box (not scale!) for Z
#declare Isoscale = <10, 10, 10>;					// scale of the isosurface
#declare Isorot = <0, 0, 0>;						// rotation of the isosurface
#declare Isotrans = <0, 0, 0>;					// translation of the isosurface

// texture:
//#declare Landtex =										// define your own texture here
// for the internal textures, use:
#declare Simpletex = false;							// just a choice between two nice textures
#declare Slope = true;									// true = slope & altitude; false = gradient texture
#declare SlopeValue = 0.5;							// (0-1) higher values: more slope, less altitude, and vice versa


// now call the macro:
Geomorph()
// ...and the landscape:
object {Landscape}
// remember that you can use this object for the trace function (see below)

//----------------------------------------------------------
#declare Trees = false;
#declare Sea = false;
#declare Fog = false;
//----------------------------------------------------------


//----------------------------------------------------------
#if (Sea)
//Deep Water:
plane { y, -10.0 pigment {color rgb <0.119608, 0.137255, 0.556863>*0.01}}
//plane { y, -10.0 pigment {color rgb <1,0,0>}}

//----------------------------------------------------------
// Iso_Sea by Marc Jacquier (2003)
// Originally: Isosurface plugin 1beta by S. Shonfield, morayplugin@yahoo.ca
// adapted to left-handed orientation:

#declare Scale_x=7;
#declare Scale_z=5;
#declare Damp=1.2;
#declare Lacunarity=3;
#declare Octave=6;
#declare Offset=1.18;
#declare Gain=5;
#declare Wave_Height=0.8;
isosurface {
   function {
      y +1-f_ridged_mf(x/Scale_x, 0, z/Scale_z,Damp,Lacunarity,Octave,Offset,Gain,2)*Wave_Height
   }
   contained_by {box{<-500,-11,-500>,<500,1,500>}}
   threshold 0
   accuracy 0.001
   max_gradient 20
   //max_trace 1
   //all_intersections
   TdG_NorthSeaWater (
  		      					0.9,  // filter
                      0.25, // transparency
                      3,    // ScaleX
                      2,    // ScaleY
                      1,    // ScaleZ
                      1,    // fade distance
                      3     // fade power
                    )
   scale <1,1,1>
   rotate <0,0,0>
   translate <0, 0, 0>
} // end isosurface

#end

#if (Trees)
// -------- random seeds for the Trees -------------
#declare Seed=seed(2);
#declare Seed2=seed(1);
#declare Test_Render = false;

// -------- Tree textures -------------
#if (Test_Render)
  // simple textures for test renders
  #declare T_Wood=
  texture {
    pigment { color rgb <1, 0, 0> }
  }

  #declare T_Tree=
  texture {
    pigment { color rgb <0, 1, 0> }
  }

#else
  #declare T_Wood=
  texture {
    pigment { color rgb <0.4, 0.2, 0.05> }
    finish {
      specular 0.3
      diffuse 0.5
    }
    normal {
      bozo 0.6
      scale <0.1, 0.1, 0.5>
    }
  }

  #declare T_Tree=
  texture {
    pigment {
      agate
      color_map {
        [0.77 color rgbt 1]
        [0.77 color rgb <0.2, 0.5, 0.10> ]
        [0.85 color rgb <0.4, 0.6, 0.15> ]
        [0.97 color rgb <0.4, 0.6, 0.15> ]
        [0.99 color rgb <0.4, 0.2, 0.05> ]
      }
      scale 0.5
      warp { turbulence 0.4 }
    }
    finish {
      diffuse 0.5
      brilliance 1.5
      ambient 0.07
    }
    normal {
      wrinkles 0.6
      scale 0.5
    }
  }
#end

// -------- Tree macro -------------
#macro TreeA (Size)
  union {
    cylinder { 0, Size*y, Size*0.04 }       // Trunk
    union {                                 // Roots
      cylinder { 0, -Size*0.30*y, Size*0.025 rotate (40+rand(Seed)*20)*x rotate rand(Seed2)*360*y }
      cylinder { 0, -Size*0.25*y, Size*0.020 rotate (40+rand(Seed)*20)*x rotate rand(Seed2)*360*y }
      cylinder { 0, -Size*0.27*y, Size*0.022 rotate (40+rand(Seed)*20)*x rotate rand(Seed2)*360*y }
    }

    union {                                 // Branches
      cylinder {
        0, Size*0.35*y, Size*0.025
        rotate (40+rand(Seed)*35)*x
        rotate rand(Seed2)*360*y
        translate Size*(0.7+0.3*rand(Seed))*y
      }
      cylinder {
        0, Size*0.40*y, Size*0.026
        rotate (40+rand(Seed)*35)*x
        rotate rand(Seed2)*360*y
        translate Size*(0.7+0.3*rand(Seed))*y
      }
      cylinder {
        0, Size*0.27*y, Size*0.022
        rotate (40+rand(Seed)*35)*x
        rotate rand(Seed2)*360*y
        translate Size*(0.7+0.3*rand(Seed))*y
      }
    }

    #if (Test_Render)                       // Foliage
      sphere {
        Size*y, Size*(0.4+rand(Seed)*0.15)
        scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
        texture { T_Tree scale Size translate rand(Seed)*6 }
      }
    #else
      union {
        sphere {
          Size*y, Size*(0.4+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.3+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.2+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.3+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.2+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
        sphere {
          Size*y, Size*(0.3+rand(Seed)*0.15)
          scale <rand(Seed)*0.5+0.5, 1, rand(Seed)*0.5+0.5>
          texture { T_Tree scale Size translate rand(Seed)*6 }
        }
      }
    #end

    texture { T_Wood scale Size }
  }
#end


#include "C:\VirtualReality\Data files\POV-Tree files\TdG_PinParasol_mesh.inc"
// -------- Placing the trees -------------

#declare Spacing=0.7;
#declare Cnt=0;


#declare PosX=-15;

#while (PosX < 15)

  #declare PosZ=-25;

  #while (PosZ < 15)

    // trace function
    #declare Norm = <0, 0, 0>;
    #declare Start = <PosX+(rand(Seed)-0.5)*Spacing, UpperBound+Isoscale.y+1, PosZ+(rand(Seed)-0.5)*Spacing>;
    #declare Pos = trace (
                  Landscape,     	 // object to test
                  Start,           // starting point
                  -y,              // direction
                  Norm );          // normal


    #if (Norm.x != 0 | Norm.y != 0 | Norm.z != 0)   // if intersection is found, normal differs from 0
      #if ((vdot(Norm, y)>0.5) & (Pos.y < 5))  			// criteria for placing trees: not too steep and not too high
        object {
          //TreeA (0.5+rand(Seed)*0.02)
          TREE 
          scale RRand(0.6, 1.0, Seed)
          rotate RRand(0, 360, Seed2)*y
          translate Pos
        }
        #declare Cnt=Cnt+1;
      #end
    #end

    #declare PosZ=PosZ+Spacing;

  #end

  #declare PosX=PosX+Spacing;
#end

#end // end of if Trees

//=====================================================================================================================================
camera {
  location  <0, 5, -30> 
  look_at   <0, 5, 0> 
  right     x*image_width/image_height
  angle     54
}

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

light_source {<-1000,2000,-1000> colour rgb 1*2}
light_source {<1000,1000,-1000> colour rgb 0.8 shadowless}

#if (Fog)
  //fog {fog_type 1 distance 200 color rgb <0.6, 0.6, 0.7>}
  fog {
    fog_type   2
    distance   30
    color      rgb <0.6, 0.6, 0.7> 
    fog_offset -0.5
    fog_alt    2.0
    turbulence 0.8
  }
#end

#end // end of Standalone

/*
========================================================================================================================================
DIRECTIVES FOR TRANSFORMING FUNCTIONS:
-------------------------------------
Transforming an isosurface object is done like transforming any POV-Ray object. Simply use the object modifiers (scale, translate, rotate, ...). 

However, when you want to transform functions within the contained_by object, you have to substitute parameters in the functions. 

The results seem inverted to what you would normally expect. Here is an explanation: 
Take a Sphere(x,y,z). We know it sits at the origin because x=0. When we want it at x=2 (translating 2 units to the right) we need to write the second equation in the same form: x-2=0 
Now that both equations equal 0, we can replace parameter x with x-2 
So our Sphere(x-2, y,z) moves two units to the right. 

Let's scale our Sphere 0.5 in the y direction. Default size is y=1 (one unit). We want y=0.5. 
To get this equation in the same form as the first one, we have to multiply both sides by two. y*2 = 0.5*2, which gives y*2=1 
Now we can replace the y parameter in our sphere: Sphere(x, y*2, z). This squishes the y-size of the sphere by half. 
Well, this is the general idea of substitutions. 

Here's an overview of some useful substitutions: 
Using a declared object P(x,y,z) 

Scale
scale x : replace "x" with "x/scale" (idem other parameters) 
scale x*2   gives    P(x/2,y,z)

Scale Infinitely
scale x infinitely : replace "x" with "0" (idem other parameters) 
scale y infinitely   gives    P(x,0,z)

Translate
translate x : replace "x" with "x - translation" (idem other parameters) 
translate z*3   gives    P(x,y,z-3)

Shear
shear in XY-plane : replace "x" with "x + y*tan(radians(Angle))" (idem other parameters) 
shear 45 degrees left   gives    P(x+y*tan(radians(45)), y, z)

Rotate 
Note: these rotation substitutions work like normal POV-rotations: they already compensate for the inverse working 

rotate around X 
: replace "y" with "z*sin(radians(Angle)) + y*cos(radians(Angle))" 
: replace "z" with "z*cos(radians(Angle)) - y*sin(radians(Angle))" 

rotate around Y 
: replace "x" with "x*cos(radians(Angle)) - z*sin(radians(Angle))" 
: replace "z" with "x*sin(radians(Angle)) + z*cos(radians(Angle))" 

rotate around Z 
: replace "x" with "x*cos(radians(Angle)) + y*sin(radians(Angle))" 
: replace "y" with "-x*sin(radians(Angle)) + y*cos(radians(Angle)) " 

  rotate z*75   gives:
  P(x*cos(radians(75)) + y*sin(radians(75)),
    -x*sin(radians(75)) + y*cos(radians(75)), z)

Flip
flip X - Y : replace "x" with "y" and replace "y" with "-x" 
flip Y - Z : replace "y" with "z" and replace "z" with "-y" 
flip X - Z : replace "x" with "-z" and replace "z" with "x" 
flip x and y   gives    P(y, -x, z)

Twist
twist N turns/unit around X 
: replace "y" with "z*sin(x*2*pi*N) + y*cos(x*2*pi*N)" 
: replace "z" with "z*cos(x*2*pi*N) - y*sin(x*2*pi*N)" 


f_hetero_mf(x,y,z, P0, P1, P2, P3, P4, P5). 
f_hetero_mf (x,0,z) makes multifractal height fields and patterns of '1/f' noise 
'Multifractal' refers to their characteristic of having a fractal dimension which varies with altitude. Built from summing 
noise of a number of frequencies, the hetero_mf parameters determine how many, and which frequencies are to be summed. 
An advantage to using these instead of a height_field {} from an image (a number of height field programs output multifractal 
types of images) is that the hetero_mf function domain extends arbitrarily far in the x and z directions so huge landscapes can 
be made without losing resolution or having to tile a height field. Other functions of interest are f_ridged_mf and f_ridge. 

P0 : H is the negative of the exponent of the basis noise frequencies used in building these functions (each frequency f's amplitude 
     is weighted by the factor f - H ). In landscapes, and many natural forms, the amplitude of high frequency contributions are usually 
     less than the lower frequencies. 
     When H is 1, the fractalization is relatively smooth ("1/f noise"). 
     As H nears 0, the high frequencies contribute equally with low frequencies as in "white noise". 
P1 : Lacunarity' is the multiplier used to get from one 'octave' to the next. This parameter affects the size of the frequency gaps 
     in the pattern. Make this greater than 1.0 
P2 : Octaves is the number of different frequencies added to the fractal. Each 'Octave' frequency is the previous one multiplied by 
     'Lacunarity', so that using a large number of octaves can get into very high frequencies very quickly. 
P3 : Offset is the 'base altitude' (sea level) used for the heterogeneous scaling 
P4 : T scales the 'heterogeneity' of the fractal. T=0 gives 'straight 1/f' (no heterogeneous scaling). T=1 suppresses higher 
     frequencies at lower altitudes 
P5 : Generator type used to generate the noise3d. 0, 1, 2 and 3 are legal values. 

*/